/**
 * Copyright (C) 2012-2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.dchartsdemo.ui.tree;

import org.dussan.dchartsdemo.DChartsDemoWindow;
import org.dussan.dchartsdemo.charts.BarCharts;
import org.dussan.dchartsdemo.charts.BubbleCharts;
import org.dussan.dchartsdemo.charts.ChartEvents;
import org.dussan.dchartsdemo.charts.DonutCharts;
import org.dussan.dchartsdemo.charts.LineCharts;
import org.dussan.dchartsdemo.charts.PieCharts;
import org.dussan.dchartsdemo.charts.PyramidCharts;
import org.dussan.dchartsdemo.charts.types.AdvancedChart;
import org.dussan.dchartsdemo.charts.types.BarChart;
import org.dussan.dchartsdemo.charts.types.BubbleChart;
import org.dussan.dchartsdemo.charts.types.ChartDataEvents;
import org.dussan.dchartsdemo.charts.types.DonutChart;
import org.dussan.dchartsdemo.charts.types.LineChart;
import org.dussan.dchartsdemo.charts.types.PieChart;
import org.dussan.dchartsdemo.charts.types.PyramidChart;

import com.vaadin.data.Item;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.ui.Tree;

public class ChartsTree extends Tree implements Item.PropertySetChangeListener,
		ItemClickListener {

	public static final String PATH_DELIMITER = "/";

	private static final long serialVersionUID = 1264066782880112434L;
	private boolean isAttached = false;
	private String selectedChart = new String();

	public ChartsTree() {
		setImmediate(true);
		setNullSelectionAllowed(false);
	}

	public static boolean isNewChart(String chartType, String chart) {
		boolean isNew = false;
		switch (ChartTypes.fromString(chartType)) {
		case ADVANCED:
			isNew = AdvancedChart.fromString(chart).isNew();
			break;
		case BAR:
			isNew = BarChart.fromString(chart).isNew();
			break;
		case BUBBLE:
			isNew = BubbleChart.fromString(chart).isNew();
			break;
		case DONUT:
			isNew = DonutChart.fromString(chart).isNew();
			break;
		case EVENTS:
			isNew = ChartDataEvents.fromString(chart).isNew();
			break;
		case LINE:
			isNew = LineChart.fromString(chart).isNew();
			break;
		case PIE:
			isNew = PieChart.fromString(chart).isNew();
			break;
		case PYRAMID:
			isNew = PyramidChart.fromString(chart).isNew();
			break;
		case DEFAULT:
		default:
			break;
		}
		return isNew;
	}

	public void setSelectedChart(String selectedChart) {
		this.selectedChart = selectedChart;
	}

	private void init() {
		ChartEvents.addMenus(this);
		LineCharts.addMenus(this);
		BarCharts.addMenus(this);
		PieCharts.addMenus(this);
		DonutCharts.addMenus(this);
		BubbleCharts.addMenus(this);
		PyramidCharts.addMenus(this);
		// AdvancedCharts.addMenus(this);

		this.setItemStyleGenerator(new ItemStyleGenerator() {
			private static final long serialVersionUID = -2292320405418615223L;

			@Override
			public String getStyle(Object itemId) {
				String item = itemId.toString();
				if (item.contains(PATH_DELIMITER)) {
					if (isNewChart(item.split(PATH_DELIMITER)[0],
							item.split(PATH_DELIMITER)[1])) {
						return (!item.equals(selectedChart) ? "new"
								: "new-selected");
					}
				}
				return null;
			}
		});
	}

	private void process(String chartType, String chart) {
		ChartTypes type = ChartTypes.fromString(chartType);
		switch (type) {
		case ADVANCED:
			chart = AdvancedChart.fromString(chart).getPath();
			break;
		case BAR:
			chart = BarChart.fromString(chart).getPath();
			break;
		case BUBBLE:
			chart = BubbleChart.fromString(chart).getPath();
			break;
		case DONUT:
			chart = DonutChart.fromString(chart).getPath();
			break;
		case EVENTS:
			chart = ChartDataEvents.fromString(chart).getPath();
			break;
		case LINE:
			chart = LineChart.fromString(chart).getPath();
			break;
		case PIE:
			chart = PieChart.fromString(chart).getPath();
			break;
		case PYRAMID:
			chart = PyramidChart.fromString(chart).getPath();
			break;
		case DEFAULT:
		default:
			type = ChartTypes.DEFAULT;
			chart = new String();
			break;
		}

		chart = (chart != null && !chart.isEmpty() && !type.getPath().equals(
				chart)) ? (PATH_DELIMITER + chart) : "";
		((DChartsDemoWindow) getApplication().getMainWindow()).getUriTool()
				.setFragment(type.getPath() + chart);
	}

	@Override
	public void attach() {
		super.attach();
		if (!isAttached) {
			isAttached = true;
			addListener((ItemClickListener) this);
			init();
		}
	}

	@Override
	public void detach() {
		super.detach();
		if (isAttached) {
			isAttached = false;
			removeListener((ItemClickListener) this);
		}
	}

	@Override
	public void itemClick(ItemClickEvent event) {
		selectedChart = event.getItemId().toString();
		if (selectedChart.contains(PATH_DELIMITER)) {
			process(selectedChart.split(PATH_DELIMITER)[0],
					selectedChart.split(PATH_DELIMITER)[1]);
		} else {
			process(selectedChart, selectedChart);
		}
	}

	@Override
	public void itemPropertySetChange(Item.PropertySetChangeEvent event) {
		// nothing to do, only for activating function "setItemStyleGenerator"
	}

}
