/**
 * Copyright (C) 2012-2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.dchartsdemo.ui.context;

import java.awt.image.BufferedImage;
import java.util.Arrays;
import java.util.List;

import org.dussan.vaadin.dcharts.DCharts;
import org.dussan.vaadin.dcharts.events.ChartData;
import org.vaadin.aceeditor.AceEditor;
import org.vaadin.aceeditor.AceMode;
import org.vaadin.aceeditor.AceTheme;

import com.vaadin.server.Page;
import com.vaadin.ui.Alignment;
import com.vaadin.ui.Component;
import com.vaadin.ui.GridLayout;
import com.vaadin.ui.Notification;
import com.vaadin.ui.Notification.Type;
import com.vaadin.ui.Panel;
import com.vaadin.ui.VerticalLayout;

public abstract class ChartContext extends VerticalLayout {

	private static final long serialVersionUID = -4543823471416442797L;
	private boolean isAttached = false;
	private GridLayout mainLayout = null;

	public ChartContext() {
		setSizeFull();
		mainLayout = new GridLayout(2, 2);
		mainLayout.setSpacing(true);
		mainLayout.setStyleName("chartCode");
		mainLayout.setWidth(100, Unit.PERCENTAGE);
		mainLayout.setHeight(null);
		addComponent(mainLayout);
	}

	abstract protected void init();

	protected void setChart(Component component) {
		mainLayout.addComponent(component, 0, 0, 1, 0);
		mainLayout.setComponentAlignment(component, Alignment.TOP_CENTER);
		if (component instanceof DCharts) {
			((DCharts) component).setMarginBottom(8);
			((DCharts) component).show();
		}
	}

	protected void setJqPlotCode(String text) {
		if (text != null) {
			Panel panel = new Panel("jqPlot Code");
			panel.setSizeUndefined();
			panel.setWidth(99, Unit.PERCENTAGE);
			panel.setHeight(null);
			mainLayout.addComponent(panel, 0, 1);
			mainLayout.setComponentAlignment(panel, Alignment.BOTTOM_LEFT);

			AceEditor code = new AceEditor();
			code.setWidth(100, Unit.PERCENTAGE);
			code.setHeight(400, Unit.PIXELS);
			code.setMode(AceMode.javascript);
			code.setTheme(AceTheme.eclipse);
			code.setValue(text);
			code.setReadOnly(true);
			code.setCursorPosition(0);
			code.setCursorRowCol(0, 0);
			panel.setContent(code);
			panel.setScrollLeft(0);
		}
	}

	protected void setDChatsCode(String text) {
		if (text != null) {
			Panel panel = new Panel("dCharts Code");
			panel.setSizeUndefined();
			panel.setWidth(100, Unit.PERCENTAGE);
			panel.setHeight(null);
			mainLayout.addComponent(panel, 1, 1);
			mainLayout.setComponentAlignment(panel, Alignment.BOTTOM_RIGHT);

			AceEditor code = new AceEditor();
			code.setWidth(100, Unit.PERCENTAGE);
			code.setHeight(400, Unit.PIXELS);
			code.setMode(AceMode.java);
			code.setTheme(AceTheme.eclipse);
			code.setValue(text);
			code.setReadOnly(true);
			code.setCursorPosition(0);
			code.setCursorRowCol(0, 0);
			panel.setContent(code);
			panel.setScrollLeft(0);
		}
	}

	protected void showNotification(String event, Object chartObject) {
		String caption = "<span style='color:#ff6600'>Event: " + event
				+ "</span>";
		StringBuilder description = new StringBuilder();

		if (chartObject instanceof ChartData) {
			ChartData chartData = (ChartData) chartObject;
			description.append("<b>Chart id:</b> " + chartData.getChartId());

			if (chartData.getSeriesIndex() != null) {
				description.append("<br /><b>Series index:</b> "
						+ chartData.getSeriesIndex());
			}

			if (chartData.getPointIndex() != null) {
				description.append("<br /><b>Point index:</b> "
						+ chartData.getPointIndex());
			}

			if (chartData.getData() != null) {
				description.append("<br /><b>Chart data:</b> "
						+ Arrays.toString(chartData.getData()));
			}

			if (chartData.getOriginData() != null) {
				if (chartData.getOriginData() instanceof Object[]) {
					description.append("<br /><b>Origin data:</b> "
							+ Arrays.toString((Object[]) chartData
									.getOriginData()));
				} else {
					description.append("<br /><b>Origin data:</b> "
							+ chartData.getOriginData().toString());
				}
			}
		} else if (chartObject instanceof BufferedImage) {
			BufferedImage chartImage = (BufferedImage) chartObject;
			description.append("<b>Chart image width:</b> "
					+ chartImage.getWidth() + "px");
			description.append("<br /><b>Chart image height:</b> "
					+ chartImage.getHeight() + "px");
		}

		Notification notification = new Notification(caption,
				description.toString(), Type.TRAY_NOTIFICATION);
		notification.setDelayMsec(3000);
		notification.setHtmlContentAllowed(true);
		notification.show(Page.getCurrent());
	}

	public static void addMenus(List<String> descriptions) {
	}

	@Override
	public void attach() {
		super.attach();
		if (!isAttached) {
			isAttached = true;
			init();
		}
	}

}
