/**
 * Copyright (C) 2012-2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.dchartsdemo.ui.context;

import java.util.LinkedHashMap;
import java.util.Map;

import org.dussan.dchartsdemo.charts.AdditionalCharts;
import org.dussan.dchartsdemo.charts.BarCharts;
import org.dussan.dchartsdemo.charts.BlockCharts;
import org.dussan.dchartsdemo.charts.BubbleCharts;
import org.dussan.dchartsdemo.charts.ChartEvents;
import org.dussan.dchartsdemo.charts.DonutCharts;
import org.dussan.dchartsdemo.charts.LineCharts;
import org.dussan.dchartsdemo.charts.MeterGaugeCharts;
import org.dussan.dchartsdemo.charts.OhlcCharts;
import org.dussan.dchartsdemo.charts.PieCharts;
import org.dussan.dchartsdemo.charts.PyramidCharts;
import org.dussan.dchartsdemo.charts.WaterfallCharts;
import org.dussan.dchartsdemo.charts.summary.ChartsSummary;
import org.dussan.dchartsdemo.ui.tree.ChartTypes;
import org.dussan.dchartsdemo.ui.tree.ChartsTree;

import com.vaadin.ui.Alignment;
import com.vaadin.ui.Component;
import com.vaadin.ui.Panel;
import com.vaadin.ui.VerticalLayout;

public class ContextPanel extends Panel {

	private static final long serialVersionUID = -5169905786362916078L;
	private boolean isAttached = false;
	private VerticalLayout contextLayout = null;

	public ContextPanel() {
	}

	private void init() {
		setSizeFull();
		contextLayout = new VerticalLayout();
		contextLayout.setStyleName("context");
		contextLayout.setMargin(true);
		contextLayout.setWidth(100, Unit.PERCENTAGE);
		contextLayout.setHeight(null);
		setContent(contextLayout);
	}

	private void replaceChartComponent(Component newComponent) {
		contextLayout.removeAllComponents();
		contextLayout.addComponent(newComponent);
		contextLayout.setComponentAlignment(newComponent, Alignment.TOP_CENTER);
	}

	private Map<String, Map<String, String[]>> addCharts() {
		Map<String, Map<String, String[]>> charts = new LinkedHashMap<String, Map<String, String[]>>();
		charts.put(ChartTypes.ADDITIONAL.getPath(),
				AdditionalCharts.getCharts());
		charts.put(ChartTypes.EVENTS.getPath(), ChartEvents.getCharts());
		charts.put(ChartTypes.BAR.getPath(), BarCharts.getCharts());
		charts.put(ChartTypes.BLOCK.getPath(), BlockCharts.getCharts());
		charts.put(ChartTypes.BUBBLE.getPath(), BubbleCharts.getCharts());
		charts.put(ChartTypes.DONUT.getPath(), DonutCharts.getCharts());
		charts.put(ChartTypes.LINE.getPath(), LineCharts.getCharts());
		charts.put(ChartTypes.METER_GAUGE.getPath(),
				MeterGaugeCharts.getCharts());
		charts.put(ChartTypes.OHLC.getPath(), OhlcCharts.getCharts());
		charts.put(ChartTypes.PIE.getPath(), PieCharts.getCharts());
		charts.put(ChartTypes.PYRAMID.getPath(), PyramidCharts.getCharts());
		charts.put(ChartTypes.WATERFALL.getPath(), WaterfallCharts.getCharts());
		return charts;
	}

	public void processChangedFragment(String fragment) {
		String chartType = fragment.isEmpty() ? fragment : fragment
				.split(ChartsTree.PATH_DELIMITER)[0];
		String chart = !fragment.isEmpty()
				&& fragment.contains(ChartsTree.PATH_DELIMITER) ? fragment
				.split(ChartsTree.PATH_DELIMITER)[1] : new String();

		Component newComponent = null;
		switch (ChartTypes.fromString(chartType)) {
		case ADDITIONAL:
			newComponent = AdditionalCharts.processChart(chart);
			break;
		case BAR:
			newComponent = BarCharts.processChart(chart);
			break;
		case BLOCK:
			newComponent = BlockCharts.processChart(chart);
			break;
		case BUBBLE:
			newComponent = BubbleCharts.processChart(chart);
			break;
		case DONUT:
			newComponent = DonutCharts.processChart(chart);
			break;
		case EVENTS:
			newComponent = ChartEvents.processChart(chart);
			break;
		case LINE:
			newComponent = LineCharts.processChart(chart);
			break;
		case METER_GAUGE:
			newComponent = MeterGaugeCharts.processChart(chart);
			break;
		case OHLC:
			newComponent = OhlcCharts.processChart(chart);
			break;
		case PIE:
			newComponent = PieCharts.processChart(chart);
			break;
		case PYRAMID:
			newComponent = PyramidCharts.processChart(chart);
			break;
		case WATERFALL:
			newComponent = WaterfallCharts.processChart(chart);
			break;
		case DEFAULT:
		default:
			newComponent = new ChartsSummary(addCharts());
			break;
		}
		replaceChartComponent(newComponent);
		setScrollTop(0);
	}

	@Override
	public void attach() {
		super.attach();
		if (!isAttached) {
			init();
			contextLayout.addComponent(new ChartsSummary(addCharts()));
			isAttached = true;
		}
	}

}
