/**
 * Copyright (C) 2012-2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.dchartsdemo.ui.tree;

import org.dussan.dchartsdemo.charts.AdditionalCharts;
import org.dussan.dchartsdemo.charts.BarCharts;
import org.dussan.dchartsdemo.charts.BlockCharts;
import org.dussan.dchartsdemo.charts.BubbleCharts;
import org.dussan.dchartsdemo.charts.ChartEvents;
import org.dussan.dchartsdemo.charts.DonutCharts;
import org.dussan.dchartsdemo.charts.LineCharts;
import org.dussan.dchartsdemo.charts.MeterGaugeCharts;
import org.dussan.dchartsdemo.charts.OhlcCharts;
import org.dussan.dchartsdemo.charts.PieCharts;
import org.dussan.dchartsdemo.charts.PyramidCharts;
import org.dussan.dchartsdemo.charts.WaterfallCharts;
import org.dussan.dchartsdemo.charts.types.AdditionalOptions;
import org.dussan.dchartsdemo.charts.types.BarChart;
import org.dussan.dchartsdemo.charts.types.BlockChart;
import org.dussan.dchartsdemo.charts.types.BubbleChart;
import org.dussan.dchartsdemo.charts.types.ChartDataEvents;
import org.dussan.dchartsdemo.charts.types.DonutChart;
import org.dussan.dchartsdemo.charts.types.LineChart;
import org.dussan.dchartsdemo.charts.types.MeterGaugeChart;
import org.dussan.dchartsdemo.charts.types.OhlcChart;
import org.dussan.dchartsdemo.charts.types.PieChart;
import org.dussan.dchartsdemo.charts.types.PyramidChart;
import org.dussan.dchartsdemo.charts.types.WaterfallChart;

import com.vaadin.data.Item;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.server.Page;
import com.vaadin.ui.Tree;

public class ChartsTree extends Tree implements Item.PropertySetChangeListener,
		ItemClickListener {

	public static final String PATH_DELIMITER = "/";

	private static final long serialVersionUID = 1264066782880112434L;
	private boolean isAttached = false;
	private String selectedChart = new String();

	public ChartsTree() {
		setImmediate(true);
		setNullSelectionAllowed(false);
	}

	public static boolean isNewChart(String chartType, String chart) {
		boolean isNew = false;
		switch (ChartTypes.fromString(chartType)) {
		case ADDITIONAL:
			isNew = AdditionalOptions.fromString(chart).isNew();
			break;
		case BAR:
			isNew = BarChart.fromString(chart).isNew();
			break;
		case BLOCK:
			isNew = BlockChart.fromString(chart).isNew();
			break;
		case BUBBLE:
			isNew = BubbleChart.fromString(chart).isNew();
			break;
		case DONUT:
			isNew = DonutChart.fromString(chart).isNew();
			break;
		case EVENTS:
			isNew = ChartDataEvents.fromString(chart).isNew();
			break;
		case LINE:
			isNew = LineChart.fromString(chart).isNew();
			break;
		case METER_GAUGE:
			isNew = MeterGaugeChart.fromString(chart).isNew();
			break;
		case OHLC:
			isNew = OhlcChart.fromString(chart).isNew();
			break;
		case PIE:
			isNew = PieChart.fromString(chart).isNew();
			break;
		case PYRAMID:
			isNew = PyramidChart.fromString(chart).isNew();
			break;
		case WATERFALL:
			isNew = WaterfallChart.fromString(chart).isNew();
			break;
		case DEFAULT:
		default:
			break;
		}
		return isNew;
	}

	public void setSelectedChart(String selectedChart) {
		this.selectedChart = selectedChart;
	}

	private void init() {
		AdditionalCharts.addMenus(this);
		ChartEvents.addMenus(this);
		BarCharts.addMenus(this);
		BlockCharts.addMenus(this);
		BubbleCharts.addMenus(this);
		DonutCharts.addMenus(this);
		LineCharts.addMenus(this);
		MeterGaugeCharts.addMenus(this);
		OhlcCharts.addMenus(this);
		PieCharts.addMenus(this);
		PyramidCharts.addMenus(this);
		WaterfallCharts.addMenus(this);

		this.setItemStyleGenerator(new ItemStyleGenerator() {
			private static final long serialVersionUID = -3452473585530366787L;

			@Override
			public String getStyle(Tree source, Object itemId) {
				String item = itemId.toString();
				if (item.contains(PATH_DELIMITER)) {
					if (isNewChart(item.split(PATH_DELIMITER)[0],
							item.split(PATH_DELIMITER)[1])) {
						return (!item.equals(selectedChart) ? "new"
								: "new-selected");
					}
				}
				return null;
			}
		});
	}

	private void process(String chartAddress, boolean refreshFragment) {
		String chartType = chartAddress;
		String chart = chartAddress;
		if (chartAddress.contains(PATH_DELIMITER)) {
			chartType = chartAddress.split(PATH_DELIMITER)[0];
			chart = chartAddress.split(PATH_DELIMITER)[1];
		}
		selectedChart = chartAddress;

		ChartTypes type = ChartTypes.fromString(chartType);
		switch (type) {
		case ADDITIONAL:
			chart = AdditionalOptions.fromString(chart).getPath();
			break;
		case BAR:
			chart = BarChart.fromString(chart).getPath();
			break;
		case BLOCK:
			chart = BlockChart.fromString(chart).getPath();
			break;
		case BUBBLE:
			chart = BubbleChart.fromString(chart).getPath();
			break;
		case DONUT:
			chart = DonutChart.fromString(chart).getPath();
			break;
		case EVENTS:
			chart = ChartDataEvents.fromString(chart).getPath();
			break;
		case LINE:
			chart = LineChart.fromString(chart).getPath();
			break;
		case METER_GAUGE:
			chart = MeterGaugeChart.fromString(chart).getPath();
			break;
		case OHLC:
			chart = OhlcChart.fromString(chart).getPath();
			break;
		case PIE:
			chart = PieChart.fromString(chart).getPath();
			break;
		case PYRAMID:
			chart = PyramidChart.fromString(chart).getPath();
			break;
		case WATERFALL:
			chart = WaterfallChart.fromString(chart).getPath();
			break;
		case DEFAULT:
		default:
			type = ChartTypes.DEFAULT;
			chart = new String();
			break;
		}

		chart = chart.isEmpty() || type.getPath().equals(chart) ? type
				.getPath() : (type.getPath() + PATH_DELIMITER + chart);
		select(chart.isEmpty() ? null : chart);
		if (refreshFragment) {
			Page.getCurrent().setUriFragment(chart);
		}
	}

	@Override
	public void attach() {
		super.attach();
		if (!isAttached) {
			isAttached = true;
			addItemClickListener((ItemClickListener) this);
			init();
		}
	}

	@Override
	public void detach() {
		super.detach();
		if (isAttached) {
			isAttached = false;
			removeItemClickListener((ItemClickListener) this);
		}
	}

	@Override
	public void itemClick(ItemClickEvent event) {
		process(event.getItemId().toString(), true);
	}

	@Override
	public void itemPropertySetChange(Item.PropertySetChangeEvent event) {
		// nothing to do, only for activating function "setItemStyleGenerator"
	}

	public void processChangedFragment(String fragment) {
		process(fragment, false);
	}

}
