/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.demo.ui.tree;

import org.dussan.vaadin.dmenu.demo.menu.BasicMenus;
import org.dussan.vaadin.dmenu.demo.menu.AdvancedMenus;
import org.dussan.vaadin.dmenu.demo.menu.types.BasicMenuTypes;
import org.dussan.vaadin.dmenu.demo.menu.types.AdvancedMenuTypes;

import com.vaadin.data.Item;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.server.Page;
import com.vaadin.ui.Tree;

public class MenuTree
    extends Tree
    implements Item.PropertySetChangeListener, ItemClickListener
{

    public static final String PATH_DELIMITER = "/";

    private static final long serialVersionUID = -3033650945759215541L;

    private boolean isAttached = false;

    private String selectedMenu = null;

    public MenuTree()
    {
        selectedMenu = new String();
        setImmediate( true );
        setNullSelectionAllowed( false );
    }

    public static boolean isNewMenu( String menuType, String menu )
    {
        boolean isNew = false;
        switch ( MenuTypes.fromString( menuType ) )
        {
            case ADVANCED:
                isNew = AdvancedMenuTypes.fromString( menu )
                                   .isNew();
                break;
            case BASIC:
                isNew = BasicMenuTypes.fromString( menu )
                                      .isNew();
                break;
            case DEFAULT:
            default:
                break;
        }
        return isNew;
    }

    private void init()
    {
        BasicMenus.addMenu( this );
        AdvancedMenus.addMenu( this );

        this.setItemStyleGenerator( new ItemStyleGenerator()
        {
            private static final long serialVersionUID = -2292320405418615223L;

            @Override
            public String getStyle( Tree source, Object itemId )
            {
                String item = itemId.toString();
                if ( item.contains( PATH_DELIMITER )
                    && isNewMenu( item.split( PATH_DELIMITER )[0], item.split( PATH_DELIMITER )[1] ) )
                {
                    return !item.equals( selectedMenu ) ? "new" : "new-selected";
                }
                return null;
            }
        } );
    }

    private void process( String menuAddress, boolean refreshFragment )
    {
        if ( !selectedMenu.equals( menuAddress ) )
        {
            unselect( selectedMenu );

            String menuType = menuAddress;
            String menu = menuAddress;
            if ( menuAddress.contains( PATH_DELIMITER ) )
            {
                menuType = menuAddress.split( PATH_DELIMITER )[0];
                menu = menuAddress.split( PATH_DELIMITER )[1];
            }
            selectedMenu = menuAddress;

            MenuTypes type = MenuTypes.fromString( menuType );
            switch ( type )
            {
                case ADVANCED:
                    menu = AdvancedMenuTypes.fromString( menu )
                                     .getPath();
                    break;
                case BASIC:
                    menu = BasicMenuTypes.fromString( menu )
                                        .getPath();
                    break;
                case DEFAULT:
                default:
                    type = MenuTypes.DEFAULT;
                    menu = new String();
                    break;
            }

            menu = ( menu.isEmpty() || type.getPath()
                                         .equals( menu ) ) ? type.getPath() : ( type.getPath() + PATH_DELIMITER + menu );
            select( menu.isEmpty() ? null : menu );
            if ( refreshFragment )
            {
                Page.getCurrent()
                    .setUriFragment( menu );
            }
        }
    }

    @Override
    public void attach()
    {
        super.attach();
        if ( !isAttached )
        {
            isAttached = true;
            addItemClickListener( this );
            init();
        }
    }

    @Override
    public void detach()
    {
        super.detach();
        if ( isAttached )
        {
            isAttached = false;
            removeItemClickListener( this );
        }
    }

    @Override
    public void itemClick( ItemClickEvent event )
    {
        process( event.getItemId()
                      .toString(),
                 true );
    }

    @Override
    public void itemPropertySetChange( com.vaadin.data.Item.PropertySetChangeEvent event )
    {
        // nothing to do, only for activating function "setItemStyleGenerator"
    }

    public void processChangedFragment( String fragment )
    {
        process( fragment, false );
    }

}
