/**
 * Copyright (C) 2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.client.elements;

import static com.google.gwt.query.client.GQuery.$;

import org.dussan.vaadin.dmenu.client.VDMenu;

import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style.Cursor;
import com.google.gwt.dom.client.Style.Display;
import com.google.gwt.query.client.Function;
import com.google.gwt.query.client.GQuery;
import com.google.gwt.query.client.css.CSS;
import com.google.gwt.query.client.css.Length;
import com.google.gwt.user.client.Window;

public class MenuItems {

	private boolean enableMenuItemTooltip = true;
	private boolean enableMenuItemAutoShrink = true;
	private int menuDirection = VDMenu.MENU_DIRECTION_AUTO;
	private Element item = null;
	private Element floatingItem = null;

	public void enableMenuItemTooltip(boolean enable) {
		enableMenuItemTooltip = enable;
	}

	public void enableMenuItemAutoShrink(boolean enable) {
		enableMenuItemAutoShrink = enable;
	}

	private String getCaptionId(String id) {
		if (id != null && id.contains(VDMenu.CONTENT_ID)) {
			id = id.replace(VDMenu.CONTENT_ID, VDMenu.CAPTION_ID);
		}
		return id;
	}

	private String getContentId(String id) {
		if (id != null && id.contains(VDMenu.CAPTION_ID)) {
			id = id.replace(VDMenu.CAPTION_ID, VDMenu.CONTENT_ID);
		}
		return id;
	}

	private String getFloatingId(String id) {
		if (id != null && id.endsWith(VDMenu.HIDDEN_ID)) {
			id = id.substring(0, id.length() - VDMenu.HIDDEN_ID.length());
		}
		return getContentId(id);
	}

	private void showCaption(Element item) {
		$(item).children().eq(0).css(CSS.DISPLAY.getCssName(), VDMenu.NO_VALUE);
		$(item).children().gt(0).remove();
	}

	private String hideCaption(Element item) {
		if (item.getChildCount() == 1) {
			GQuery itemDivs = $(item).find(VDMenu.DIV);
			String caption = itemDivs.last().text();
			String classes = itemDivs.get(1).getClassName();
			$(VDMenu.EMPTY_DIV)
					.id(itemDivs.first().id() + VDMenu.HIDDEN_ID)
					.css(CSS.CURSOR.with(Cursor.POINTER))
					.append($(VDMenu.EMPTY_DIV)
							.addClass(classes, VDMenu.HIDDEN)
							.css(CSS.DISPLAY.with(Display.INLINE_BLOCK))
							.html(VDMenu.EMPTY_HTML)).appendTo($(item));
			itemDivs.first().css(CSS.DISPLAY.with(Display.NONE));
			return caption;
		}
		return VDMenu.NO_VALUE;
	}

	private void showContent(Element item) {
		$(item).children().eq(0).css(CSS.DISPLAY.getCssName(), VDMenu.NO_VALUE);
		$(item).children().gt(0).remove();
	}

	private void hideContent(Element item, String contentText) {
		if (item.getChildCount() == 1) {
			GQuery itemDivs = $(item).find(VDMenu.DIV);
			$(VDMenu.EMPTY_DIV).html(
					VDMenu.CONTENT_HIDDEN.replaceFirst(VDMenu.QUESTION,
							itemDivs.first().id() + VDMenu.HIDDEN_ID)
							.replaceFirst(VDMenu.QUESTION, contentText))
					.appendTo($(item));
			itemDivs.first().css(CSS.DISPLAY.with(Display.NONE));
		}
	}

	private void setTooltip(Element parentItem) {
		if (enableMenuItemAutoShrink) {
			Element item = parentItem.getFirstChildElement();
			String caption = $(item).text();

			$(item).text(VDMenu.NO_VALUE);
			int width = $(parentItem).innerWidth();
			int length = caption.length() - 1;
			$(item).text(caption);
			for (; 0 < length && width <= $(item).outerWidth(); length--) {
				$(item).text(caption.substring(0, length));
			}

			if (!$(item).text().equals(caption)) {
				length = Math.max(1, length - VDMenu.THREE_DOT.length());
				$(item).text(caption.substring(0, length) + VDMenu.THREE_DOT);
				$(VDMenu.EMPTY_DIV).text(caption)
						.addClass(VDMenu.CAPTION_TOOLTIP)
						.appendTo($(parentItem));
			}
		}
	}

	public Element getItem() {
		return item;
	}

	public void setItem(Element replacement) {
		if (item != null && item.hasParentElement()) {
			$(item).parent().append(replacement);
			item.removeFromParent();
		}
		item = replacement;
	}

	public void check(String menuId, int expandingDirection, String tabsLocation) {
		GQuery captions = $("[class^=\"" + VDMenu.CAPTION_CELL + "\"]", item)
				.not(VDMenu.CLASS(VDMenu.CAPTION_TOOLTIP));
		GQuery contents = $("[class^=\"" + VDMenu.CONTENT_CELL + "\"]", item);
		this.menuDirection = expandingDirection;

		// reset items
		contents.each(new Function() {
			public void f(Element item) {
				showContent(item);
			}
		});
		captions.each(new Function() {
			public void f(Element item) {
				showCaption(item);
				setTooltip(item.getFirstChildElement());
			}
		});

		// hide item if too long
		GQuery container = $(item).parents(
				"[id$=\"" + VDMenu.CONTAINER_ID + "\"]").parent();
		container.css(CSS.WIDTH.with(Length.px(0)));
		int width = $(VDMenu.ID(menuId)).parent().width();
		for (int i = captions.size() - 1; 0 <= i
				&& width < $(item).outerWidth(); i--) {
			String caption = hideCaption(captions.get(i));
			hideContent(contents.get(i), caption);
		}
		container.css(CSS.WIDTH.getCssName(), VDMenu.NO_VALUE);

		// check floating menu item visibility
		int height = $(item).height();
		int top = $(item).offset().top;
		int bottom = top + height;
		if (this.menuDirection == VDMenu.MENU_DIRECTION_AUTO) {
			this.menuDirection = VDMenu.TAB_LOCATION_TOP.equals(tabsLocation) ? VDMenu.MENU_DIRECTION_DOWN
					: VDMenu.MENU_DIRECTION_UP;
			if (VDMenu.TAB_LOCATION_TOP.equals(tabsLocation)
					&& (Window.getScrollTop() + Window.getClientHeight()) < (bottom + height)) {
				this.menuDirection = VDMenu.MENU_DIRECTION_UP;
			} else if (VDMenu.TAB_LOCATION_BOTTOM.equals(tabsLocation)
					&& (top - height) <= Window.getScrollTop()) {
				this.menuDirection = VDMenu.MENU_DIRECTION_DOWN;
			}
		}

		captions.parent().detach();
		switch (this.menuDirection) {
		case VDMenu.MENU_DIRECTION_UP:
			contents.parent().parent().prepend(captions.parent());
			captions.removeClass(VDMenu.CAPTION_CELL_BOTTOM).addClass(
					VDMenu.CAPTION_CELL_TOP);
			contents.removeClass(VDMenu.CONTENT_CELL_TOP).addClass(
					VDMenu.CONTENT_CELL_BOTTOM);
			break;
		case VDMenu.MENU_DIRECTION_DOWN:
		default:
			contents.parent().parent().append(captions.parent());
			captions.removeClass(VDMenu.CAPTION_CELL_TOP).addClass(
					VDMenu.CAPTION_CELL_BOTTOM);
			contents.removeClass(VDMenu.CONTENT_CELL_BOTTOM).addClass(
					VDMenu.CONTENT_CELL_TOP);
			break;
		}
	}

	public void showTooltip(Element item) {
		if (enableMenuItemTooltip && item != null) {
			String id = getCaptionId($(item).children().first().id());
			$(VDMenu.ID(id)).children(VDMenu.CLASS(VDMenu.CAPTION_TOOLTIP))
					.each(new Function() {
						public void f(Element tooltip) {
							$(tooltip)
									.css(CSS.DISPLAY.with(Display.BLOCK))
									.css(CSS.LEFT.with(Length.px($(tooltip)
											.parent().parent().offset().left)))
									.css(CSS.TOP.with(Length
											.px($(tooltip).parent().parent()
													.offset().top - 1)));
						}
					});
		}
	}

	public void hideTooltip(Element item) {
		if (enableMenuItemTooltip && item != null) {
			String id = getCaptionId($(item).children().first().id());
			$(VDMenu.ID(id)).children(VDMenu.CLASS(VDMenu.CAPTION_TOOLTIP))
					.each(new Function() {
						public void f(Element tooltip) {
							$(tooltip).css(CSS.DISPLAY.getCssName(),
									VDMenu.NO_VALUE);
						}
					});
		}
	}

	public boolean canItemFloating(GQuery item) {
		item = $(VDMenu.ID(getFloatingId(item.children().first().id())))
				.parent();
		return (item.children().size() > 1
				&& !item.children().eq(0).isVisible() && item.children().eq(1)
				.children().eq(0).hasClass(VDMenu.HIDDEN));
	}

	public boolean isFloatingItem(GQuery item) {
		String id = getFloatingId(item.children().eq(0).id());
		return (isFloatingItemVisible() && floatingItem.getId().equals(id));
	}

	public boolean isFloatingItemVisible() {
		return (floatingItem != null);
	}

	public void showFloatingItem(Element item, String tabsLocation) {
		item = $(VDMenu.ID(getFloatingId(item.getId()))).get(0);
		if (!isFloatingItem($(item))) {
			Element parent = item.getParentElement();
			$(item).addClass(VDMenu.CONTENT_FLOAT).css(
					CSS.DISPLAY.getCssName(), VDMenu.NO_VALUE);
			int top = parent.getAbsoluteBottom();
			int left = Math.max(0, parent.getAbsoluteRight()
					- $(item).outerWidth(true));
			if (menuDirection == VDMenu.MENU_DIRECTION_UP) {
				top = $(parent).offset().top - $(item).outerHeight(true);
			}
			$(item).css(CSS.LEFT.with(Length.px(left))).css(
					CSS.TOP.with(Length.px(top)));
			floatingItem = item;
		}
	}

	public void hideFloatingItem() {
		$(floatingItem).removeClass(VDMenu.CONTENT_FLOAT)
				.css(CSS.DISPLAY.with(Display.NONE))
				.css(CSS.LEFT.getCssName(), VDMenu.NO_VALUE)
				.css(CSS.TOP.getCssName(), VDMenu.NO_VALUE);
		floatingItem = null;
	}

}
