/**
 * Copyright (C) 2013  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.elements;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.dussan.vaadin.dmenu.MenuDirection;
import org.dussan.vaadin.dmenu.TabsLocation;

import com.vaadin.ui.Component;
import com.vaadin.ui.CustomLayout;
import com.vaadin.ui.HorizontalLayout;

public class Menu implements Serializable {

	private static final String MENU_MINIMIZED = "minimized";
	private static final String MENU_TABS = "v-dmenu-tabs";
	private static final String MENU_TABS_ID = "-tab";
	private static final String MENU_TABS_ROW = "<ul id='?' class='?'>?</ul>";
	private static final String MENU_TABS_CELL = "<li id='?'>?</li>";
	private static final String MENU_TABS_CELL_ACTIVE = "<li id='?' class='selected'>?</li>";

	private static final String MENU_ITEMS_ID = "-item";
	private static final String MENU_ITEMS_CONTAINER_ID = "-container";
	private static final String MENU_ITEMS_CONTAINER_TOP = "v-dmenu-items-container-top";
	private static final String MENU_ITEMS_CONTAINER_BOTTOM = "v-dmenu-items-container-bottom";
	private static final String MENU_ITEMS_TESTER = "v-dmenu-items-tester";
	private static final String MENU_ITEMS_TESTER_ID = "-item-tester";

	private static final long serialVersionUID = 3890095799327312073L;
	private List<MenuItem> container = null;

	public Menu() {
		container = new ArrayList<MenuItem>();
	}

	public static String generateCode(String target, String... replacements) {
		for (String replacement : replacements) {
			target = target.replaceFirst("\\?", replacement);
		}
		return target;
	}

	private String getMenuItemContainerClass(TabsLocation tabsLocation) {
		switch (tabsLocation) {
		case BOTTOM_LEFT:
		case BOTTOM_CENTER:
		case BOTTOM_RIGHT:
			return MENU_ITEMS_CONTAINER_BOTTOM;
		case TOP_LEFT:
		case TOP_CENTER:
		case TOP_RIGHT:
		default:
			return MENU_ITEMS_CONTAINER_TOP;
		}
	}

	public int check(int activeTab) {
		return Math.max(0, Math.min(activeTab, container.size() - 1));
	}

	public Component getTabs(String menuId, int activeTab,
			TabsLocation tabsLocation) {
		try {
			StringBuilder tabCells = new StringBuilder();
			for (int item = 0; item < container.size(); item++) {
				tabCells.append(generateCode(
						item == activeTab ? MENU_TABS_CELL_ACTIVE
								: MENU_TABS_CELL, menuId + MENU_TABS_ID + item,
						container.get(item).getCaption()));
			}
			return (new CustomLayout(new ByteArrayInputStream(generateCode(
					MENU_TABS_ROW, menuId + MENU_TABS_ID,
					MENU_TABS + "-" + tabsLocation.getLocation(),
					tabCells.toString()).getBytes())));
		} catch (IOException e) {
			return null;
		}
	}

	public Component getContent(String menuId, int activeTab,
			TabsLocation tabsLocation,
			MenuDirection expandDirection, boolean isMenuHidden) {
		HorizontalLayout content = new HorizontalLayout();
		content.setSizeFull();
		content.addComponent(container.get(activeTab).getContent(
				menuId + MENU_ITEMS_ID + activeTab, expandDirection));
		for (int item = 0; item < container.size(); item++) {
			if (item != activeTab) {
				content.addComponent(container.get(item).getContent(
						menuId + MENU_ITEMS_ID + item, expandDirection));
			}
		}
		content.setId(menuId + MENU_ITEMS_CONTAINER_ID);
		content.setStyleName(getMenuItemContainerClass(tabsLocation));
		if (isMenuHidden) {
			content.addStyleName(MENU_MINIMIZED);
		}

		// add tester item which is needed only for hiding inactive menu items
		// when calculate menu items maximum height
		HorizontalLayout testerItem = new HorizontalLayout();
		testerItem.setId(menuId + MENU_ITEMS_TESTER_ID);
		testerItem.setStyleName(MENU_ITEMS_TESTER);
		content.addComponent(testerItem, 1);

		return content;
	}

	public boolean isEmpty() {
		return container.isEmpty();
	}

	public void addMenuItem(MenuItem item) {
		if (item == null || item.isEmpty()) {
			throw new IllegalArgumentException("Menu item is empty.");
		} else if (container.contains(item)) {
			throw new IllegalArgumentException("Menu already have an item.");
		}
		container.add(item);
	}

	public void addMenuItem(MenuItem item, int index) {
		if (item == null || item.isEmpty()) {
			throw new IllegalArgumentException("Menu item is empty.");
		} else if (container.contains(item)) {
			throw new IllegalArgumentException("Menu already have an item.");
		} else if (index < 0) {
			throw new IllegalArgumentException(
					"Index have to be greater than or equal 0.");
		}
		container.add(index, item);
	}

	public MenuItem getMenuItem(int index) {
		return container.get(index);
	}

	public void addMenuItems(MenuItem... items) {
		if (items == null || items.length == 0) {
			throw new IllegalArgumentException(
					"Menu item array does not contains any elements.");
		}
		for (MenuItem item : items) {
			addMenuItem(item);
		}
	}

	public List<MenuItem> getMenuItems() {
		return container;
	}

	public Iterator<MenuItem> iterator() {
		return container.iterator();
	}

}
