/**
 * Copyright (C) 2013-2015  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.client;

import static com.google.gwt.query.client.GQuery.$;
import static com.google.gwt.query.client.GQuery.window;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dussan.vaadin.dmenu.client.elements.FloatingMenu;
import org.dussan.vaadin.dmenu.client.elements.MenuItems;
import org.dussan.vaadin.dmenu.client.elements.MenuTabs;
import org.dussan.vaadin.dmenu.client.events.MenuEvents;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.query.client.Function;
import com.google.gwt.query.client.GQuery;
import com.google.gwt.query.client.css.CSS;
import com.google.gwt.query.client.css.Length;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.Event.NativePreviewEvent;
import com.google.gwt.user.client.Event.NativePreviewHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.RootPanel;
import com.vaadin.client.ui.VHorizontalLayout;

public class VDMenu extends VHorizontalLayout implements
		HasValueChangeHandlers<Object[]> {

	private static final int ACTION_ENABLE_MENU_HIDE = 0;
	private static final int ACTION_ENABLE_MENU_TABS_AUTO_SHOW = 1;
	private static final int ACTION_ENABLE_MENU_ITEM_TOOLTIP = 2;
	private static final int ACTION_ENABLE_MENU_ITEM_AUTO_SHRINK = 3;
	private static final int ACTION_ENABLE_FLOATING_MENU_ITEM_AUTO_SHOW = 4;
	private static final int ACTION_ENABLE_FLOATING_ELEMENTS_AUTO_HIDE = 5;
	private static final int ACTION_MENU_DIRECTION = 6;
	private static final int ACTION_MENU_MARGINS = 7;
	private static final int ACTION_ACTIVE_TAB = 8;
	private static final int ACTION_HIDE_MENU = 9;

	public static final String HIDDEN = "hidden";
	public static final String HIDDEN_ID = "-hidden";
	public static final String ITEM_ID = "-item";
	public static final String ITEM_TESTER_ID = "-item-tester";
	public static final String MINIMIZED = "minimized";
	public static final String THREE_DOT = "...";

	public static final String BOTTOM = "-bottom";
	public static final String CENTER = "-center";
	public static final String LEFT = "-left";
	public static final String RIGHT = "-right";
	public static final String TOP = "-top";

	public static final String CLASS = "class";
	public static final String DIV = "div";
	public static final String EMPTY_DIV = "<div />";
	public static final String EMPTY_HTML = "&nbsp;";
	public static final String NO_VALUE = "";
	public static final String QUESTION = "\\?";

	public static final String CONTAINER = "v-dmenu-items-container-";
	public static final String CONTAINER_ID = "-container";
	public static final String CAPTION_ID = "-caption";
	public static final String CAPTION_CELL = "v-dmenu-items-caption-cell";
	public static final String CAPTION_CELL_TOP = "v-dmenu-items-caption-cell-top";
	public static final String CAPTION_CELL_BOTTOM = "v-dmenu-items-caption-cell-bottom";
	public static final String CAPTION_TOOLTIP = "v-dmenu-items-caption-cell-tooltip";
	public static final String CONTENT_ID = "-content";
	public static final String CONTENT_CELL = "v-dmenu-items-content-cell";
	public static final String CONTENT_CELL_TOP = "v-dmenu-items-content-cell-top";
	public static final String CONTENT_CELL_BOTTOM = "v-dmenu-items-content-cell-bottom";
	public static final String CONTENT_FLOAT = "float";
	public static final String CONTENT_HIDDEN = "<div id='?' class='hidden'><div><div><div>?</div></div></div></div>";

	public static final String TAB = "li";
	public static final String TAB_ID = "-tab";
	public static final String TAB_ACTIVE = "selected";
	public static final String TAB_VISIBLE = "li:visible";
	public static final String TAB_LOCATION_BOTTOM = "v-dmenu-bottom";
	public static final String TAB_LOCATION_TOP = "v-dmenu-top";

	public static final String TAB_BUTTON_LEFT_ID = "v-dmenu-tab-button-left";
	public static final String TAB_BUTTON_RIGHT_ID = "v-dmenu-tab-button-right";
	public static final String TAB_BUTTON_LEFT_ACTIVE = "left-active";
	public static final String TAB_BUTTON_LEFT_INACTIVE = "left";
	public static final String TAB_BUTTON_RIGHT_ACTIVE = "right-active";
	public static final String TAB_BUTTON_RIGHT_INACTIVE = "right";
	public static final String TAB_BUTTON_BOTTOM_LEFT = "v-dmenu-tab-button-bottom-left";
	public static final String TAB_BUTTON_BOTTOM_RIGHT = "v-dmenu-tab-button-bottom-right";
	public static final String TAB_BUTTON_TOP_LEFT = "v-dmenu-tab-button-top-left";
	public static final String TAB_BUTTON_TOP_RIGHT = "v-dmenu-tab-button-top-right";

	public static final int MENU_DIRECTION_AUTO = 0;
	public static final int MENU_DIRECTION_UP = 1;
	public static final int MENU_DIRECTION_DOWN = 2;

	private boolean windowHasFocus = true;
	private boolean windowResizeIsRunning = false;
	private boolean menuIsPrepared = false;
	private boolean enableMenuHide = false;
	private boolean enableMenuTabsAutoShow = false;
	private boolean enableFloatingMenuItemAutoShow = false;
	private boolean enableFloatingElementsAutoHide = false;

	private int windowWidth = 0;
	private String menuId = null;
	private MenuTabs menuTabs = null;
	private MenuItems menuItems = null;
	private FloatingMenu floatingMenu = null;
	private Map<Integer, String> menuActions = null;
	private Map<String, Element> menuElements = null;

	private GQuery mouseLeaveWindowEventHandler = null;
	private GQuery windowEventsHandler = null;
	private GQuery tabsEventHandler = null;
	private GQuery tabLeftButtonEventHandler = null;
	private GQuery tabRightButtonEventHandler = null;
	private GQuery containerEventHandler = null;
	private GQuery captionItemEventHandler = null;
	private GQuery contentItemEventHandler = null;
	private GQuery floatingContentItemEventHandler = null;
	private GQuery floatingElementsEventHandler = null;

	private VDMenu() {
		menuActions = new HashMap<Integer, String>();
		menuItems = new MenuItems();
		floatingMenu = new FloatingMenu();
	}

	public static String ID(String id) {
		return ("[id='" + id + "']");
	}

	public static String CLASS(String styleClass) {
		return ("[class='" + styleClass + "']");
	}

	public static int LENGTH(String value) {
		if (value != null && !value.isEmpty()) {
			return Integer.parseInt(value.endsWith("px") ? value
					.substring(0, 1) : value);
		}
		return 0;
	}

	@Override
	protected void onLoad() {
		super.onLoad();
		Event.addNativePreviewHandler(new NativePreviewHandler() {
			@Override
			public void onPreviewNativeEvent(NativePreviewEvent event) {
				switch (event.getTypeInt()) {
				case Event.ONCLICK:
					String id = getId(Element.as(event.getNativeEvent()
							.getEventTarget()));
					if (id.isEmpty()
							|| (!id.startsWith(menuId)
									&& !id.equals(TAB_BUTTON_LEFT_ID) && !id
										.equals(TAB_BUTTON_RIGHT_ID))) {
						processHideFloatingItemOrFloatingMenu();
					}
					break;
				case Event.ONKEYDOWN:
					if (event.getNativeEvent().getKeyCode() == KeyCodes.KEY_ESCAPE) {
						processHideFloatingItemOrFloatingMenu();
					}
					break;
				default:
					break;
				}
			}
		});
	}

	@Override
	protected void onUnload() {
		activateMouseLeaveWindowEventHandler(false);
		activateWindowEventsHandler(false);
		activateTabsEventHandler(false);
		activateTabLeftButtonEventHandler(false);
		activateTabRightButtonEventHandler(false);
		activateContainerEventHandler(false);
		activateCaptionItemEventHandler(false);
		activateContentItemEventHandler(false);
		activateFloatingtContentItemEventHandler(false);
		activateFloatingElementsEventHandler(false);
		super.onUnload();
	}

	@Override
	public HandlerRegistration addValueChangeHandler(
			ValueChangeHandler<Object[]> handler) {
		HandlerRegistration registration = this.addHandler(handler,
				ValueChangeEvent.getType());
		return registration;
	}

	private String getId(Element item) {
		while (item != null) {
			if (item.getId() != null && !item.getId().trim().isEmpty()) {
				return item.getId();
			}
			item = item.getParentElement();
		}
		return NO_VALUE;
	}

	private int getItemsContainerHeight() {
		GQuery clone = $(EMPTY_DIV).css(CSS.POSITION.with(Position.FIXED))
				.css(CSS.TOP.with(Length.px(-10000)))
				.css(CSS.LEFT.with(Length.px(-10000)))
				.append($(ID(menuId)).clone()).appendTo($(getElement()));
		clone.find("[class*=' " + MINIMIZED + " ']").removeClass(MINIMIZED);
		int height = clone.find(ID(menuId + CONTAINER_ID)).height();
		clone.remove();
		return height;
	}

	private void archiveMenu() {
		int height = getItemsContainerHeight();
		menuTabs = new MenuTabs(menuId);
		menuElements = new HashMap<String, Element>();
		floatingMenu.setContainerHeight(height);
		for (int i = 0; i < menuTabs.getTabs().size(); i++) {
			GQuery tab = menuTabs.getTab(i);
			Element item = $(VDMenu.ID(menuTabs.getTabItemId(i)))
					.height(height).get(0);
			menuElements.put(menuTabs.getTab(i).id(), item);
			if (!menuTabs.isActiveTab(tab)) {
				item.getParentElement().removeFromParent();
			} else {
				menuItems.setItem(item);
			}
		}
		checkMenuElements();
		$(ID(menuId + CONTAINER_ID)).css(CSS.MARGIN_BOTTOM.with(Length.px(0)));

		// remove tester item which is needed only for hiding inactive menu
		// items to get menu items maximum height
		Element testerElement = $(ID(menuId + ITEM_TESTER_ID)).get(0);
		if (testerElement != null) {
			testerElement.getParentElement().removeFromParent();
		}
	}

	private void activateMouseLeaveWindowEventHandler(boolean enable) {
		if (enable) {
			activateMouseLeaveWindowEventHandler(false);
			mouseLeaveWindowEventHandler = $(RootPanel.get()).mouseleave(
					new Function() {
						public void f() {
							processHideFloatingItemAndFloatingMenu();
						}
					});
		} else if (!enable && mouseLeaveWindowEventHandler != null) {
			mouseLeaveWindowEventHandler.die();
			mouseLeaveWindowEventHandler = null;
		}
	}

	private void activateWindowEventsHandler(boolean enable) {
		if (enable) {
			activateWindowEventsHandler(false);
			windowEventsHandler = $(window).resize(new Function() {
				public void f() {
					if (!windowResizeIsRunning) {
						processWindowResize();
					}
				};
			}).blur(new Function() {
				public void f() {
					windowHasFocus = false;
					processHideFloatingItemAndFloatingMenu();
				}
			}).focus(new Function() {
				public void f() {
					windowHasFocus = true;
				}
			});
		} else if (!enable && windowEventsHandler != null) {
			windowEventsHandler.die();
			windowEventsHandler = null;
		}
	}

	private void activateTabsEventHandler(boolean enable) {
		if (enable) {
			activateTabsEventHandler(false);
			tabsEventHandler = menuTabs.getTabs()
					.live(Event.ONMOUSEDOWN, new Function() {
						public boolean f(Event event) {
							// disable text selection
							return false;
						}
					}).live(Event.ONMOUSEOVER, new Function() {
						public void f(Element tab) {
							if (enableMenuTabsAutoShow
									&& !menuItems.isFloatingItemVisible()) {
								processTabClick(tab);
							}
						}
					}).live(Event.ONCLICK, new Function() {
						public void f(Element tab) {
							processTabClick(tab);
						}
					});
			// TODO: need better double click handler
			// .live(Event.ONDBLCLICK, new Function() {
			// public void f() {
			// processTabDblClick(false);
			// }
			// });
		} else if (!enable && tabsEventHandler != null) {
			tabsEventHandler.die(Event.ONMOUSEDOWN | Event.ONMOUSEOVER
					| Event.ONCLICK | Event.ONDBLCLICK);
			tabsEventHandler = null;
		}
	}

	private void activateTabLeftButtonEventHandler(boolean enable) {
		if (enable) {
			activateTabLeftButtonEventHandler(false);
			tabLeftButtonEventHandler = $(CLASS(TAB_BUTTON_LEFT_ACTIVE),
					menuTabs.getTabs().parent().get(0)).live(Event.ONCLICK,
					new Function() {
						public void f() {
							menuTabs.goLeft();
						}
					});
		} else if (!enable && tabLeftButtonEventHandler != null) {
			tabLeftButtonEventHandler.die(Event.ONCLICK);
			tabLeftButtonEventHandler = null;
		}
	}

	private void activateTabRightButtonEventHandler(boolean enable) {
		if (enable) {
			activateTabRightButtonEventHandler(false);
			tabLeftButtonEventHandler = $(CLASS(TAB_BUTTON_RIGHT_ACTIVE),
					menuTabs.getTabs().parent().get(0)).live(Event.ONCLICK,
					new Function() {
						public void f() {
							menuTabs.goRight();
						}
					});
		} else if (!enable && tabRightButtonEventHandler != null) {
			tabRightButtonEventHandler.die(Event.ONCLICK);
			tabRightButtonEventHandler = null;
		}
	}

	private void activateContainerEventHandler(boolean enable) {
		if (enable) {
			activateContainerEventHandler(false);
			containerEventHandler = $(ID(menuId + VDMenu.CONTAINER_ID)).live(
					Event.ONCLICK, new Function() {
						public void f() {
							processHideFloatingItem();
						}
					});
		} else if (!enable && containerEventHandler != null) {
			containerEventHandler.die(Event.ONCLICK);
			containerEventHandler = null;
		}
	}

	private void activateCaptionItemEventHandler(boolean enable) {
		if (enable) {
			activateCaptionItemEventHandler(false);
			captionItemEventHandler = $("[class^=\"" + CAPTION_CELL + "\"]",
					$(ID(menuId)).get(0))
					.live(Event.ONMOUSEDOWN, new Function() {
						public boolean f(Event event) {
							// disable text selection
							return false;
						}
					}).live(Event.ONMOUSEOVER, new Function() {
						public void f(Element item) {
							if (windowHasFocus
									&& !menuItems.isFloatingItemVisible()) {
								menuItems.showTooltip(item);
							}
						}
					}).live(Event.ONMOUSEOUT, new Function() {
						public void f() {
							if (windowHasFocus
									&& !menuItems.isFloatingItemVisible()) {
								menuItems.hideTooltip();
							}
						}
					}).live(Event.ONCLICK, new Function() {
						public boolean f(Event event) {
							processItemClick($(event));
							return false;
						}
					});
		} else if (!enable && captionItemEventHandler != null) {
			captionItemEventHandler.die(Event.ONMOUSEDOWN | Event.ONMOUSEOVER
					| Event.ONMOUSEOUT | Event.ONCLICK);
			captionItemEventHandler = null;
		}
	}

	private void activateContentItemEventHandler(boolean enable) {
		if (enable) {
			activateContentItemEventHandler(false);
			contentItemEventHandler = $("[class^=\"" + CONTENT_CELL + "\"]",
					$(ID(menuId)).get(0))
					.live(Event.ONMOUSEOVER, new Function() {
						public void f(Element item) {
							if (windowHasFocus
									&& !menuItems.isFloatingItemVisible()) {
								menuItems.showTooltip(item);
							}
						}
					}).live(Event.ONMOUSEOUT, new Function() {
						public void f() {
							if (windowHasFocus
									&& !menuItems.isFloatingItemVisible()) {
								menuItems.hideTooltip();
							}
						}
					}).live(Event.ONCLICK, new Function() {
						public boolean f(Event event) {
							processItemClick($(event));
							return false;
						}
					});
		} else if (!enable && contentItemEventHandler != null) {
			contentItemEventHandler.die(Event.ONMOUSEDOWN | Event.ONMOUSEOVER
					| Event.ONMOUSEOUT | Event.ONCLICK);
			contentItemEventHandler = null;
		}
	}

	private void activateFloatingtContentItemEventHandler(boolean enable) {
		if (enable) {
			activateFloatingtContentItemEventHandler(false);
			floatingContentItemEventHandler = $(
					"[id^=\"" + menuId + VDMenu.ITEM_ID + "\"][id*=\""
							+ VDMenu.CONTENT_ID + "\"][id$=\""
							+ VDMenu.HIDDEN_ID + "\"]").live(Event.ONMOUSEOVER,
					new Function() {
						public void f(Element item) {
							if (enableFloatingMenuItemAutoShow
									&& !menuItems.isFloatingItem($(item)
											.parent())) {
								processHideAndShowFloatingItem(item);
							}
						}
					});
		} else if (!enable && floatingContentItemEventHandler != null) {
			floatingContentItemEventHandler.die(Event.ONMOUSEOVER);
			floatingContentItemEventHandler = null;
		}
	}

	private void activateFloatingElementsEventHandler(boolean enable) {
		if (enable) {
			activateFloatingElementsEventHandler(false);
			floatingElementsEventHandler = $(ID(menuId)).mouseleave(
					new Function() {
						public void f() {
							if (enableFloatingElementsAutoHide) {
								processHideFloatingItemAndFloatingMenu();
							}
						}
					});
		} else if (!enable && floatingElementsEventHandler != null) {
			floatingElementsEventHandler.die();
			floatingElementsEventHandler = null;
		}
	}

	private void processWindowResize() {
		windowResizeIsRunning = true;
		menuItems.hideFloatingItem();
		Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {
			@Override
			public boolean execute() {
				boolean widthIsChanged = (windowWidth == Window.getClientWidth());
				windowWidth = Window.getClientWidth();
				if (menuIsPrepared && !widthIsChanged) {
					checkMenuElements();
					windowResizeIsRunning = false;
				}
				return (!menuIsPrepared || widthIsChanged);
			}
		}, 50);
	}

	private void processShowFloatingItem(Element item) {
		if (windowHasFocus) {
			menuItems.showFloatingItem(item, menuTabs.getTabsLocation());
			MenuEvents.fireFloatingMenuItemChange(this, true);
		}
	}

	private void processHideFloatingItem() {
		if (menuItems.isFloatingItemVisible()) {
			menuItems.hideFloatingItem();
			MenuEvents.fireFloatingMenuItemChange(this, false);
		}
	}

	private void processHideAndShowFloatingItem(Element item) {
		if (windowHasFocus) {
			processHideFloatingItem();
			processShowFloatingItem(item);
		}
	}

	private void processShowFloatingMenu() {
		if (floatingMenu.isFloatingMenuEnabled()
				&& !floatingMenu.isFloatingMenuVisible()) {
			floatingMenu.showFloatingMenu(menuTabs.getTabsLocation());
			menuItems.check(menuId, floatingMenu.getMenuDirection(),
					menuTabs.getTabsLocation());
			MenuEvents.fireFloatingMenuChange(this, true);
		}
	}

	private void processHideFloatingMenu() {
		if (floatingMenu.isFloatingMenuVisible()) {
			menuItems.hideTooltip();
			floatingMenu.hideFloatingMenu();
			MenuEvents.fireFloatingMenuChange(this, false);
		}
	}

	private void processHideFloatingItemOrFloatingMenu() {
		if (menuItems.isFloatingItemVisible()) {
			processHideFloatingItem();
		} else if (floatingMenu.isFloatingMenuVisible()) {
			processHideFloatingMenu();
		}
	}

	private void processHideFloatingItemAndFloatingMenu() {
		processHideFloatingItem();
		processHideFloatingMenu();
	}

	private void processItemClick(GQuery item) {
		if (menuItems.isFloatingItemVisible()
				&& !menuItems.isFloatingItem(item)
				&& menuItems.canItemFloating(item)) {
			processHideAndShowFloatingItem(item.children().get(0));
		} else if (!menuItems.isFloatingItem(item)
				&& menuItems.canItemFloating(item)) {
			processShowFloatingItem(item.children().get(0));
		} else if (!menuItems.isFloatingItem(item)) {
			processHideFloatingItem();
		}
	}

	private void processTabDblClick(boolean hideOnStartup) {
		if (windowHasFocus) {
			if (enableMenuHide || hideOnStartup) {
				processHideFloatingItem();
				if (!floatingMenu.isFloatingMenuEnabled() || hideOnStartup) {
					floatingMenu.enableFloatingMenu(menuId);
					if (!hideOnStartup) {
						MenuEvents.fireFloatingMenuChange(this, true);
					}
				} else if (floatingMenu.isFloatingMenuEnabled()
						&& !hideOnStartup) {
					floatingMenu.disableFloatingMenu();
					MenuEvents.fireFloatingMenuChange(this, false);
				}
				if (hideOnStartup) {
					processHideFloatingMenu();
				}
			}
		}
	}

	private void processTabClick(Element tab) {
		if (windowHasFocus) {
			if (!menuTabs.isActiveTab($(tab))) {
				processHideFloatingItem();

				menuTabs.setActiveTab($(tab));
				MenuEvents.fireActiveTabChange(this,
						menuTabs.getActiveTabNumber());

				menuItems.setItem(menuElements
						.get(menuTabs.getActiveTab().id()));
				menuItems.check(menuId, floatingMenu.getMenuDirection(),
						menuTabs.getTabsLocation());
				processShowFloatingMenu();
			} else if (floatingMenu.isFloatingMenuEnabled()) {
				if (!floatingMenu.isFloatingMenuVisible()) {
					processShowFloatingMenu();
				} else {
					processHideFloatingMenu();
				}
			}

		}
	}

	public void checkMenuElements() {
		menuItems.check(menuId, floatingMenu.getMenuDirection(),
				menuTabs.getTabsLocation());
		menuTabs.check(true);
	}

	public void processMenuActions(Map<Integer, String> menuActions) {
		if (menuActions == null || menuActions.isEmpty()) {
			return;
		}

		List<Integer> actionsIndex = new ArrayList<Integer>(
				menuActions.keySet());
		Collections.sort(actionsIndex);
		for (int index : actionsIndex) {
			String action = menuActions.get(index).toString();
			switch (index) {
			case ACTION_ENABLE_MENU_HIDE:
				enableMenuHide = Boolean.parseBoolean(action);
				menuActions.remove(index);
				break;
			case ACTION_ENABLE_MENU_TABS_AUTO_SHOW:
				enableMenuTabsAutoShow = Boolean.parseBoolean(action);
				menuActions.remove(index);
				break;
			case ACTION_ENABLE_MENU_ITEM_TOOLTIP:
				menuItems.enableMenuItemTooltip(Boolean.parseBoolean(action));
				menuActions.remove(index);
				break;
			case ACTION_ENABLE_MENU_ITEM_AUTO_SHRINK:
				menuItems
						.enableMenuItemAutoShrink(Boolean.parseBoolean(action));
				menuActions.remove(index);
				break;
			case ACTION_ENABLE_FLOATING_MENU_ITEM_AUTO_SHOW:
				enableFloatingMenuItemAutoShow = Boolean.parseBoolean(action);
				menuActions.remove(index);
				break;
			case ACTION_ENABLE_FLOATING_ELEMENTS_AUTO_HIDE:
				enableFloatingElementsAutoHide = Boolean.parseBoolean(action);
				menuActions.remove(index);
				break;
			case ACTION_MENU_DIRECTION:
				floatingMenu.setMenuDirection(Integer.parseInt(action));
				menuActions.remove(index);
				break;
			case ACTION_MENU_MARGINS:
				String[] margins = action.replaceAll("\\[", NO_VALUE)
						.replaceAll("\\]", NO_VALUE).split(",");
				$(getElement()).css(
						CSS.PADDING_TOP.with(Length.px(Integer
								.parseInt(margins[0].trim()))));
				$(getElement()).css(
						CSS.PADDING_RIGHT.with(Length.px(Integer
								.parseInt(margins[1].trim()))));
				$(getElement()).css(
						CSS.PADDING_BOTTOM.with(Length.px(Integer
								.parseInt(margins[2].trim()))));
				$(getElement()).css(
						CSS.PADDING_LEFT.with(Length.px(Integer
								.parseInt(margins[3].trim()))));
				menuActions.remove(index);
				break;
			case ACTION_ACTIVE_TAB:
				if (menuIsPrepared) {
					GQuery tabs = $(ID(menuId + TAB_ID)).children(TAB);
					int tab = Math.min(Integer.parseInt(action),
							tabs.size() - 1);
					if (!menuTabs.isActiveTab(menuTabs.getTab(tab))) {
						processTabClick(tabs.get(tab));
						if (floatingMenu.isFloatingMenuEnabled()) {
							processHideFloatingMenu();
						}
					}
					menuActions.remove(index);
				}
				break;
			case ACTION_HIDE_MENU:
				if (menuIsPrepared) {
					processTabDblClick(Boolean.parseBoolean(action));
					menuActions.remove(index);
				}
				break;
			default:
				break;
			}
		}

		if (!menuIsPrepared) {
			this.menuActions = menuActions;
		} else {
			checkMenuElements();
			this.menuActions.clear();
			MenuEvents.fireActionsAreProcessed(this);
		}
	}

	public boolean isMenuPrepared() {
		return menuIsPrepared;
	}

	public void prepareMenu() {
		menuId = getElement().getFirstChildElement().getId();
		archiveMenu();

		activateMouseLeaveWindowEventHandler(true);
		activateWindowEventsHandler(true);
		activateTabsEventHandler(true);
		activateTabLeftButtonEventHandler(true);
		activateTabRightButtonEventHandler(true);
		activateContainerEventHandler(true);
		activateCaptionItemEventHandler(true);
		activateContentItemEventHandler(true);
		activateFloatingtContentItemEventHandler(true);
		activateFloatingElementsEventHandler(true);

		menuIsPrepared = true;
		processMenuActions(menuActions);
	}

}
