/**
 * Copyright (C) 2013-2015  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.elements;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.dussan.vaadin.dmenu.MenuDirection;

import com.vaadin.ui.CustomLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.VerticalLayout;

public class MenuItem implements Serializable {

	private static final String CAPTION = "caption";
	private static final String CONTENT = "content";

	private static final String MENU_ITEMS = "v-dmenu-items";
	private static final String MENU_ITEMS_CAPTION_ROW = "v-dmenu-items-caption-row";
	private static final String MENU_ITEMS_CONTENT_ROW = "v-dmenu-items-content-row";
	private static final String MENU_ITEMS_CAPTION_CELL_TOP = "v-dmenu-items-caption-cell-top";
	private static final String MENU_ITEMS_CAPTION_CELL_BOTTOM = "v-dmenu-items-caption-cell-bottom";
	private static final String MENU_ITEMS_CONTENT_CELL_TOP = "v-dmenu-items-content-cell-top";
	private static final String MENU_ITEMS_CONTENT_CELL_BOTTOM = "v-dmenu-items-content-cell-bottom";

	private static final String TABLE = "<table class='?'>?</table>";
	private static final String TABLE_ROW = "<tr id='?' class='?'>?</tr>";
	private static final String TABLE_CELL = "<td class='?'><div id='?' location='?'></div></td>";

	private static final long serialVersionUID = -1163698964285675670L;
	private String caption = null;
	private List<MenuItemElement> container = null;

	public MenuItem(String caption) {
		this.caption = caption;
		container = new ArrayList<MenuItemElement>();
	}

	public String getCaption() {
		return caption;
	}

	public void addMenuElement(MenuItemElement itemElement) {
		if (itemElement == null || itemElement.getContent() == null) {
			throw new IllegalArgumentException("Menu element is empty.");
		} else if (itemElement.getContent().getParent() != null) {
			throw new IllegalArgumentException(
					"Menu element already have a parent.");
		}
		container.add(itemElement);
	}

	public void addMenuElement(MenuItemElement itemElement, int index) {
		if (itemElement == null || itemElement.getContent() == null) {
			throw new IllegalArgumentException("Menu element is empty.");
		} else if (itemElement.getContent().getParent() != null) {
			throw new IllegalArgumentException(
					"Menu element already have a parent.");
		} else if (index < 0) {
			throw new IllegalArgumentException(
					"Index have to be greater than or equal 0.");
		}
		container.add(index, itemElement);
	}

	public MenuItemElement getMenuElement(int index) {
		return container.get(index);
	}

	public void addMenuElements(MenuItemElement... itemElements) {
		if (itemElements == null || itemElements.length == 0) {
			throw new IllegalArgumentException(
					"Menu array does not contains any elements.");
		}
		for (MenuItemElement itemElement : itemElements) {
			addMenuElement(itemElement);
		}
	}

	public List<MenuItemElement> getMenuElements() {
		return container;
	}

	public boolean isEmpty() {
		return (caption.isEmpty() || container.isEmpty());
	}

	public CustomLayout getContent(String id,
			MenuDirection expandDirection) {
		String captionClass = null;
		String contentClass = null;
		switch (expandDirection) {
		case UP:
			captionClass = MENU_ITEMS_CAPTION_CELL_TOP;
			contentClass = MENU_ITEMS_CONTENT_CELL_BOTTOM;
			break;

		case DOWN:
		case AUTO:
		default:
			contentClass = MENU_ITEMS_CONTENT_CELL_TOP;
			captionClass = MENU_ITEMS_CAPTION_CELL_BOTTOM;
			break;
		}

		// build menu item table
		StringBuilder contentRow = new StringBuilder();
		StringBuilder captionRow = new StringBuilder();
		for (int col = 0; col < container.size(); col++) {
			contentRow.append(Menu.generateCode(TABLE_CELL, contentClass, id
					+ "-" + CONTENT + col, CONTENT + col));
			captionRow.append(Menu.generateCode(TABLE_CELL, captionClass, id
					+ "-" + CAPTION + col, CAPTION + col));
		}
		StringBuilder table = new StringBuilder();
		contentRow = new StringBuilder(Menu.generateCode(TABLE_ROW, id + "-"
				+ CONTENT, MENU_ITEMS_CONTENT_ROW, contentRow.toString()));
		captionRow = new StringBuilder(Menu.generateCode(TABLE_ROW, id + "-"
				+ CAPTION, MENU_ITEMS_CAPTION_ROW, captionRow.toString()));
		table.append(Menu.generateCode(TABLE, MENU_ITEMS, contentRow.toString()
				+ captionRow.toString()));

		// fill menu item table
		try {
			CustomLayout menuLayout = new CustomLayout(
					new ByteArrayInputStream(table.toString().getBytes()));
			menuLayout.setId(id);
			menuLayout.setHeight("100%");
			for (int item = 0; item < container.size(); item++) {
				VerticalLayout content = new VerticalLayout();
				content.addComponent(container.get(item).getContent());
				menuLayout.addComponent(content, CONTENT + item);
				Label caption = new Label(container.get(item).getCaption());
				caption.setSizeUndefined();
				menuLayout.addComponent(caption, CAPTION + item);
			}
			return menuLayout;
		} catch (IOException e) {
			return null;
		}
	}

}
