/**
 * Copyright (C) 2013-2015  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu;

import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.dussan.vaadin.dmenu.client.rpc.MenuServerRpc;
import org.dussan.vaadin.dmenu.client.state.MenuState;
import org.dussan.vaadin.dmenu.elements.Menu;
import org.dussan.vaadin.dmenu.elements.MenuItem;
import org.dussan.vaadin.dmenu.events.ActiveTabChangeEvent;
import org.dussan.vaadin.dmenu.events.ActiveTabChangeHandler;
import org.dussan.vaadin.dmenu.events.FloatingMenuChangeEvent;
import org.dussan.vaadin.dmenu.events.FloatingMenuChangeHandler;
import org.dussan.vaadin.dmenu.events.FloatingMenuItemChangeEvent;
import org.dussan.vaadin.dmenu.events.FloatingMenuItemChangeHandler;
import org.dussan.vaadin.dmenu.helper.ManifestHelper;

import com.google.gwt.event.shared.HandlerManager;
import com.vaadin.ui.AbstractSingleComponentContainer;
import com.vaadin.ui.Component;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.VerticalLayout;

public class DMenu implements Serializable {

	private static final long serialVersionUID = -4820907157920611592L;
	private static final String MENU_ID = "dmenu-" + System.currentTimeMillis();
	private static final String MENU_TOP = "v-dmenu-top";
	private static final String MENU_BOTTOM = "v-dmenu-bottom";
	private static final String MENU_MINIMIZED = "minimized";
	private static final String MENU_ITEMS_CONTAINER_BORDER_TOP = "v-dmenu-items-container-border-top";
	private static final String MENU_ITEMS_CONTAINER_BORDER_BOTTOM = "v-dmenu-items-container-border-bottom";
	private static final int MARGIN_DEFAULT = 5;

	// TODO: need better double click handler
	// private static final int ACTION_ENABLE_MENU_HIDE = 0;
	private static final int ACTION_ENABLE_MENU_TABS_AUTO_SHOW = 1;
	private static final int ACTION_ENABLE_MENU_ITEM_TOOLTIP = 2;
	private static final int ACTION_ENABLE_MENU_ITEM_AUTO_SHRINK = 3;
	private static final int ACTION_ENABLE_FLOATING_MENU_ITEM_AUTO_SHOW = 4;
	private static final int ACTION_ENABLE_FLOATING_ELEMENTS_AUTO_HIDE = 5;
	private static final int ACTION_MENU_DIRECTION = 6;
	private static final int ACTION_MENU_MARGINS = 7;
	private static final int ACTION_ACTIVE_TAB = 8;
	private static final int ACTION_HIDE_MENU = 9;

	// TODO: need better double click handler
	// private boolean enableMenuHide = false;
	private boolean enableMenuTabsAutoShow = false;
	private boolean enableMenuItemTooltip = true;
	private boolean enableMenuItemAutoShrink = true;
	private boolean enableFloatingMenuItemAutoShow = false;
	private boolean enableFloatingElementsAutoHide = false;
	private boolean menuIsHidden = false;
	private boolean floatingMenuIsVisible = false;
	private boolean floatingMenuItemIsVisible = false;
	private int activeTab = 0;
	private MenuDirection menuDirection = MenuDirection.AUTO;
	private int[] margins = new int[] { 0, 0, 0, 0 };

	private HandlerManager handlerManager = null;
	private MenuComponent menuComponent = null;
	private TabsLocation tabsLocation = null;

	private DMenu getThis() {
		return this;
	}

	public class MenuComponent extends AbstractSingleComponentContainer {

		private static final long serialVersionUID = 5301442910169642982L;
		private Menu menu = null;
		private Map<Integer, String> menuActions = null;

		private MenuComponent() {
			menuActions = new HashMap<Integer, String>();
			registerRpc(new MenuServerRpc() {
				private static final long serialVersionUID = 5482852905498385196L;

				@Override
				public void onActionsAreProcessed() {
					menuActions.clear();
				}

				@Override
				public void onActiveTabValueChange(int activeTab) {
					getThis().activeTab = activeTab;
					handlerManager
							.fireEvent(new ActiveTabChangeEvent(activeTab));
				}

				@Override
				public void onFloatingMenuValueChange(boolean visible) {
					getThis().floatingMenuIsVisible = visible;
					handlerManager.fireEvent(new FloatingMenuChangeEvent(
							visible));
				}

				@Override
				public void onFloatingItemValueChange(boolean visible) {
					getThis().floatingMenuItemIsVisible = visible;
					handlerManager.fireEvent(new FloatingMenuItemChangeEvent(
							visible));
				}
			});
		}

		public Map<Integer, String> getMenuActions() {
			markAsDirty();
			return menuActions;
		}

		@Override
		protected MenuState getState() {
			return (MenuState) super.getState();
		}

		private MenuComponent(Menu menu) {
			this();
			this.menu = menu;
		}

		private MenuComponent(MenuItem... menuItems) {
			this();
			menu = new Menu();
			menu.addMenuItems(menuItems);
		}

		private HorizontalLayout getBorder(String style, boolean menuIsHidden) {
			HorizontalLayout border = new HorizontalLayout();
			border.setStyleName(style);
			if (menuIsHidden) {
				border.addStyleName(MENU_MINIMIZED);
			}
			return border;
		}

		private void createMenuComponent(TabsLocation tabsLocation) {
			VerticalLayout component = new VerticalLayout();
			component.setSizeFull();
			component.setId(MENU_ID);
			component.addComponent(getBorder(MENU_ITEMS_CONTAINER_BORDER_TOP,
					menuIsHidden));
			component.addComponent(menu.getContent(MENU_ID, activeTab,
					tabsLocation, menuDirection, menuIsHidden));
			component.addComponent(getBorder(
					MENU_ITEMS_CONTAINER_BORDER_BOTTOM, menuIsHidden));

			activeTab = menu.check(activeTab);
			switch (tabsLocation) {
			case BOTTOM_LEFT:
			case BOTTOM_CENTER:
			case BOTTOM_RIGHT:
				component.setStyleName(MENU_BOTTOM);
				component.addComponent(menu.getTabs(MENU_ID, activeTab,
						tabsLocation));
				break;
			case TOP_LEFT:
			case TOP_CENTER:
			case TOP_RIGHT:
			default:
				component.setStyleName(MENU_TOP);
				component.addComponent(
						menu.getTabs(MENU_ID, activeTab, tabsLocation), 0);
				break;
			}

			setContent(component);
			markAsDirty();
		}

		@Override
		public void attach() {
			super.attach();
			setWidth("100%");
		}

		@Override
		public void setCaption(String caption) {
			throw new IllegalArgumentException("Currently not supported.");
		}

		@Override
		public void beforeClientResponse(boolean initial) {
			super.beforeClientResponse(initial);
			if (menuActions.size() > 0) {
				getState().setMenuActions(menuActions);
			}
		}
	}

	private DMenu() {
		tabsLocation = TabsLocation.TOP_LEFT;
		handlerManager = new HandlerManager(this);
	}

	public DMenu(Menu menu) {
		this();
		menuComponent = new MenuComponent(menu);
	}

	public DMenu(MenuItem... menuItems) {
		this();
		menuComponent = new MenuComponent(menuItems);
	}

	public static String getVersion() {
		if (ManifestHelper.getManifest() != null) {
			return ManifestHelper.getManifest().getMainAttributes()
					.getValue("Implementation-Version");
		}
		return null;
	}

	public static String getGitVersion() {
		if (ManifestHelper.getManifest() != null) {
			return ManifestHelper.getManifest().getMainAttributes()
					.getValue("Git-Version");
		}
		return null;
	}

	public void addHandler(ActiveTabChangeHandler handler) {
		handlerManager.addHandler(ActiveTabChangeEvent.getType(), handler);
	}

	public void removeHandler(ActiveTabChangeHandler handler) {
		if (handlerManager.isEventHandled(ActiveTabChangeEvent.getType())) {
			handlerManager.removeHandler(ActiveTabChangeEvent.getType(),
					handler);
		}
	}

	public void addHandler(FloatingMenuChangeHandler handler) {
		handlerManager.addHandler(FloatingMenuChangeEvent.getType(), handler);
	}

	public void removeHandler(FloatingMenuChangeHandler handler) {
		if (handlerManager.isEventHandled(FloatingMenuChangeEvent.getType())) {
			handlerManager.removeHandler(FloatingMenuChangeEvent.getType(),
					handler);
		}
	}

	public void addHandler(FloatingMenuItemChangeHandler handler) {
		handlerManager.addHandler(FloatingMenuItemChangeEvent.getType(),
				handler);
	}

	public void removeHandler(FloatingMenuItemChangeHandler handler) {
		if (handlerManager
				.isEventHandled(FloatingMenuItemChangeEvent.getType())) {
			handlerManager.removeHandler(FloatingMenuItemChangeEvent.getType(),
					handler);
		}
	}

	public Component getComponent() {
		menuComponent.createMenuComponent(tabsLocation);
		return menuComponent;
	}

	public boolean isFloatingMenuItemVisible() {
		return floatingMenuItemIsVisible;
	}

	public boolean isFloatingMenuVisible() {
		return floatingMenuIsVisible;
	}

	public TabsLocation getTabsLocation() {
		return tabsLocation;
	}

	public void setTabsLocation(TabsLocation tabsLocation) {
		this.tabsLocation = tabsLocation;
	}

	public int getActiveTab() {
		return activeTab;
	}

	public void setActiveTab(int activeTab) {
		this.activeTab = activeTab;
		menuComponent.getMenuActions().put(ACTION_ACTIVE_TAB,
				Integer.toString(activeTab));
	}

	// TODO: need better double click handler
	// public boolean isEnableHideMenu() {
	// return enableMenuHide;
	// }
	//
	// public void enableMenuHide(boolean enable) {
	// enableMenuHide = enable;
	// menuComponent.getMenuActions().put(ACTION_ENABLE_MENU_HIDE,
	// Boolean.toString(enable));
	// }

	public boolean isMenuHidden() {
		return menuIsHidden;
	}

	public void setMenuHidden(boolean hideMenu) {
		menuIsHidden = hideMenu;
		menuComponent.getMenuActions().put(ACTION_HIDE_MENU,
				Boolean.toString(hideMenu));
	}

	public boolean isEnableMenuTabsAutoShow() {
		return enableMenuTabsAutoShow;
	}

	public void enableMenuTabsAutoShow(boolean enable) {
		enableMenuTabsAutoShow = enable;
		menuComponent.getMenuActions().put(ACTION_ENABLE_MENU_TABS_AUTO_SHOW,
				Boolean.toString(enable));
	}

	public boolean isEnableMenuItemTooltip() {
		return enableMenuItemTooltip;
	}

	public void enableMenuItemTooltip(boolean enable) {
		enableMenuItemTooltip = enable;
		menuComponent.getMenuActions().put(ACTION_ENABLE_MENU_ITEM_TOOLTIP,
				Boolean.toString(enable));
	}

	public boolean isEnableMenuItemAutoShrink() {
		return enableMenuItemAutoShrink;
	}

	public void enableMenuItemAutoShrink(boolean enable) {
		enableMenuItemAutoShrink = enable;
		menuComponent.getMenuActions().put(ACTION_ENABLE_MENU_ITEM_AUTO_SHRINK,
				Boolean.toString(enable));
	}

	public boolean isEnableFloatingMenuItemAutoShow() {
		return enableFloatingMenuItemAutoShow;
	}

	public void enableFloatingMenuItemAutoShow(boolean enable) {
		enableFloatingMenuItemAutoShow = enable;
		menuComponent.getMenuActions().put(
				ACTION_ENABLE_FLOATING_MENU_ITEM_AUTO_SHOW,
				Boolean.toString(enable));
	}

	public boolean isEnableFloatingElementsAutoHide() {
		return enableFloatingElementsAutoHide;
	}

	public void enableFloatingElementsAutoHide(boolean enable) {
		enableFloatingElementsAutoHide = enable;
		menuComponent.getMenuActions().put(
				ACTION_ENABLE_FLOATING_ELEMENTS_AUTO_HIDE,
				Boolean.toString(enable));
	}

	public MenuDirection getMenuDirection() {
		return menuDirection;
	}

	public void setMenuDirection(MenuDirection menuDirection) {
		this.menuDirection = menuDirection;
		menuComponent.getMenuActions().put(ACTION_MENU_DIRECTION,
				menuDirection.toString());
	}

	public int[] getMargins() {
		return margins;
	}

	public void setMargins(boolean enable) {
		if (enable) {
			this.margins = new int[] { MARGIN_DEFAULT, MARGIN_DEFAULT,
					MARGIN_DEFAULT, MARGIN_DEFAULT };
		} else {
			this.margins = new int[] { 0, 0, 0, 0 };
		}
		menuComponent.getMenuActions().put(ACTION_MENU_MARGINS,
				Arrays.toString(this.margins));
	}

	public void setMargins(boolean marginTop, boolean marginRight,
			boolean marginBottom, boolean marginLeft) {
		margins[0] = marginTop ? MARGIN_DEFAULT : 0;
		margins[1] = marginRight ? MARGIN_DEFAULT : 0;
		margins[2] = marginBottom ? MARGIN_DEFAULT : 0;
		margins[3] = marginLeft ? MARGIN_DEFAULT : 0;
		menuComponent.getMenuActions().put(ACTION_MENU_MARGINS,
				Arrays.toString(margins));
	}

	public void setMargins(int marginTop, int marginRight, int marginBottom,
			int marginLeft) {
		margins = new int[] { Math.max(0, marginTop), Math.max(0, marginRight),
				Math.max(0, marginBottom), Math.max(0, marginLeft) };
		menuComponent.getMenuActions().put(ACTION_MENU_MARGINS,
				Arrays.toString(margins));
	}

}
