/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.dussan.vaadin.dmenu.client.events.MenuEvents;
import org.dussan.vaadin.dtabs.client.TabsWidget;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Node;
import com.google.gwt.dom.client.NodeList;
import com.google.gwt.dom.client.Style.Display;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.dom.client.Style.Visibility;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.EventListener;
import com.google.gwt.user.client.ui.RootPanel;
import com.vaadin.client.ApplicationConnection;
import com.vaadin.client.LayoutManager;

public class VDMenu
    extends TabsWidget
    implements HasValueChangeHandlers<Object[]>
{

    private static final String CLASS_NAME = "className";

    private static final String EMPTY_VALUE = "";

    private static final String SPACE_VALUE = " ";

    private static final String COMBOBOX_OPTIONLIST_ID = "VAADIN_COMBOBOX_OPTIONLIST";

    private static final String COMBOBOX_OPTIONLIST_CLASS_NAME = "v-filterselect-suggestpopup";

    private static final String MENU_ITEM_ID = "dmenu-";

    private static final String MENU_SPACING = "v-spacing";

    private static final String MENU_TABSHEET = "v-tabsheet";

    private static final String MENU_DECO = MENU_TABSHEET + "-deco";

    private static final String MENU_TAB_CONTAINER = MENU_TABSHEET + "-tabcontainer";

    private static final String MENU_TABS = MENU_TABSHEET + "-tabs";

    private static final String MENU_PANEL = MENU_TABSHEET + "-tabsheetpanel";

    private static final String MENU_CONTENT = MENU_TABSHEET + "-content";

    private static final String MENU_LAYOUT = "v-horizontallayout-v-dmenu";

    private static final String MENU_SCROLLER_PREV = "v-dmenu-scroller-prev";

    private static final String MENU_SCROLLER_NEXT = "v-dmenu-scroller-next";

    private static final String MENU_ITEM_HAS_NO_CAPTION = "v-slot-has-no-caption";

    private static final String MENU_ITEM_SLOT = "v-slot-v-dmenu-item";

    private static final String MENU_ITEM_CAPTION_BOTTOM = MENU_ITEM_SLOT + "-caption-bottom";

    private static final String MENU_ITEM_CAPTION_TOP = MENU_ITEM_SLOT + "-caption-top";

    private static final String MENU_ITEM_SLOT_CONTEXT = MENU_ITEM_SLOT + "-context";

    private static final String MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT =
        MENU_ITEM_SLOT_CONTEXT + "-items-same-height";

    private static final String MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT =
        MENU_ITEM_SLOT_CONTEXT + "-elements-same-height";

    private static final int MENU_LAYOUT_PADDING = 100;

    private static final int MENU_ZINDEX = 10000;

    private ApplicationConnection connection = null;

    private List<Integer> firstVisibleMenuItemElementIndex = null;

    private boolean menuItemsSameHeight = false;

    private boolean menuItemsSameHeightDefined = false;

    private boolean menuItemElementsSameHeight = false;

    private boolean menuItemAutoChange = false;

    private boolean floatingMenu = false;

    private boolean floatingMenuVisible = false;

    private boolean floatingMenuItemAutoHide = false;

    private boolean floatingMenuItemAutoChange = false;

    private int activeMenuItem = 0;

    private int menuItemElementsCount = 0;

    private int menuPanelMaxHeight = 0;

    private int menuItemMaxHeight = 0;

    private int menuItemCaptionMaxHeight = 0;

    private int menuLayoutElementPaddingRight = MENU_LAYOUT_PADDING;

    private int menuLayoutElementPaddingLeft = MENU_LAYOUT_PADDING;

    private String menuId = null;

    private String activeMenuTabId = null;

    private Element scrollerPrev;

    private Element scrollerNext;

    private Element foreignMenuItemElements = null;

    public VDMenu()
    {
        firstVisibleMenuItemElementIndex = new ArrayList<>();
        foreignMenuItemElements = DOM.createDiv();

        // scroller for previous menu item elements
        scrollerPrev = DOM.createButton();
        scrollerPrev.setTabIndex( -1 );
        scrollerPrev.setPropertyString( CLASS_NAME, MENU_SCROLLER_PREV );
        DOM.sinkEvents( scrollerPrev, Event.ONCLICK );
        Event.setEventListener( scrollerPrev, new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                if ( Event.ONCLICK == event.getTypeInt() )
                {
                    int index = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() );
                    Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );

                    // hide spacer
                    if ( 0 < index && Element.as( menuLayoutElement.getChild( index - 1 ) )
                                             .hasClassName( MENU_SPACING ) )
                    {
                        index -= 1;
                    }

                    // hide menu item element
                    index -= 1;

                    // save index of first visible menu item element
                    firstVisibleMenuItemElementIndex.set( getActiveMenuItem(), Math.max( 0, index ) );

                    // check menu item elements visibility
                    checkMenuItemElementsVisibility( false );
                }
            }
        } );

        // scroller for next menu item elements
        scrollerNext = DOM.createButton();
        scrollerNext.setTabIndex( -1 );
        scrollerNext.setPropertyString( CLASS_NAME, MENU_SCROLLER_NEXT );
        DOM.sinkEvents( scrollerNext, Event.ONCLICK );
        Event.setEventListener( scrollerNext, new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                if ( Event.ONCLICK == event.getTypeInt() )
                {
                    int index = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() );
                    Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );

                    // hide menu item element
                    index += 1;

                    // hide spacer
                    if ( index < menuLayoutElement.getChildCount() && Element.as( menuLayoutElement.getChild( index ) )
                                                                             .hasClassName( MENU_SPACING ) )
                    {
                        index += 1;
                    }

                    // save index of first visible menu item element
                    firstVisibleMenuItemElementIndex.set( getActiveMenuItem(),
                                                          Math.min( menuLayoutElement.getChildCount() - 1, index ) );

                    // check menu item elements visibility
                    checkMenuItemElementsVisibility( false );
                }
            }
        } );
    }

    private static Element getParentElementByClassName( Element element, String className )
    {
        String tempClassName = EMPTY_VALUE;
        while ( element.hasParentElement() && tempClassName.isEmpty() )
        {
            if ( element.hasClassName( className ) )
            {
                if ( element.hasClassName( className ) )
                {
                    return element;
                }
                tempClassName = element.getClassName();
            }
            element = element.getParentElement();
        }
        return null;
    }

    private static List<Element> getChildElementsByClassName( Element element, String className )
    {
        ArrayList<Element> findedElements = new ArrayList<>();
        findChildElementsByClassName( findedElements, element, className );
        return findedElements;
    }

    private static void findChildElementsByClassName( ArrayList<Element> findedChildElements, Element element,
                                                      String className )
    {
        if ( element != null )
        {
            String elementClasses = element.getPropertyString( CLASS_NAME );
            if ( elementClasses != null )
            {
                List<String> classes = new ArrayList<>( Arrays.asList( elementClasses.split( SPACE_VALUE ) ) );
                if ( classes.contains( className ) )
                {
                    findedChildElements.add( element );
                }
            }

            for ( int i = 0; i < DOM.getChildCount( element ); i++ )
            {
                Element child = DOM.getChild( element, i );
                findChildElementsByClassName( findedChildElements, child, className );
            }
        }
    }

    private boolean isFloatingMenuVisible()
    {
        return floatingMenuVisible;
    }

    private void setFloatingMenuVisible( Element menuPanelElement, Element menuContentElement,
                                         boolean floatingMenuVisible )
    {
        this.floatingMenuVisible = floatingMenuVisible;
        if ( floatingMenuVisible )
        {
            menuPanelElement.getStyle()
                            .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            menuPanelElement.getStyle()
                            .clearVisibility();

            if ( isTabBarBottom() )
            {
                menuContentElement.getStyle()
                                  .clearHeight();
                menuContentElement.getStyle()
                                  .setTop( -getMenuPanelMaxHeight(), Unit.PX );
            }

            // show option list of combo box if exits
            if ( RootPanel.get( COMBOBOX_OPTIONLIST_ID ) != null )
            {
                RootPanel.get( COMBOBOX_OPTIONLIST_ID )
                         .getElement()
                         .getStyle()
                         .clearDisplay();
            }
        }
        else
        {
            menuPanelElement.getStyle()
                            .clearHeight();
            menuPanelElement.getStyle()
                            .setVisibility( Visibility.HIDDEN );

            if ( isTabBarBottom() )
            {
                menuContentElement.getStyle()
                                  .setHeight( 1, Unit.PX );
                menuContentElement.getStyle()
                                  .setTop( -1, Unit.PX );
            }
            else
            {
                menuContentElement.getStyle()
                                  .clearHeight();
                menuContentElement.getStyle()
                                  .clearTop();
            }

            // hide option list of combo box if exits
            if ( RootPanel.get( COMBOBOX_OPTIONLIST_ID ) != null )
            {
                RootPanel.get( COMBOBOX_OPTIONLIST_ID )
                         .getElement()
                         .getStyle()
                         .setDisplay( Display.NONE );
            }
        }
        processFloatingMenuItemShow( floatingMenuVisible );
    }

    private String getActiveMenuTabId()
    {
        return activeMenuTabId;
    }

    private void setActiveMenuTabId( String activeMenuTabId )
    {
        this.activeMenuTabId = activeMenuTabId;
    }

    public int getMenuItemCaptionMaxHeight()
    {
        return menuItemCaptionMaxHeight;
    }

    public void setMenuItemCaptionMaxHeight( int menuItemCaptionMaxHeight )
    {
        this.menuItemCaptionMaxHeight = menuItemCaptionMaxHeight;
    }

    private int getMenuItemMaxHeight()
    {
        return menuItemMaxHeight;
    }

    private void setMenuItemMaxHeight( int menuItemMaxHeight )
    {
        this.menuItemMaxHeight = menuItemMaxHeight;
    }

    private int getMenuPanelMaxHeight()
    {
        return menuPanelMaxHeight;
    }

    private void setMenuPanelMaxHeight( int menuPanelMaxHeight )
    {
        this.menuPanelMaxHeight = menuPanelMaxHeight;
    }

    private void processMoveMenuItemElementsToItsParent()
    {
        MenuEvents.fireMoveMenuItemElementsToItsParent( this );
    }

    private void processMenuItemChange( String activeMenuTabId )
    {
        if ( isMenuItemAutoChange() && !getActiveMenuTabId().equals( activeMenuTabId ) )
        {
            setActiveMenuTabId( activeMenuTabId );
            MenuEvents.fireMenuItemChange( this, activeMenuTabId );
        }
    }

    private void processMenuItemElementClick( Integer menuItemElementId, Boolean doubleClick )
    {
        MenuEvents.fireMenuItemElementClick( this, menuItemElementId, doubleClick );
    }

    private void processFloatingMenuItemChange( String menuItemId )
    {
        if ( isFloatingMenu() && isFloatingMenuItemAutoShow() && !getActiveMenuTabId().equals( menuItemId ) )
        {
            setActiveMenuTabId( menuItemId );
            MenuEvents.fireFloatingMenuItemChange( this, menuItemId );
        }
    }

    private void processFloatingMenuItemShow( boolean floatingMenuVisible )
    {
        if ( isFloatingMenu() )
        {
            MenuEvents.fireFloatingMenuItemShow( this, floatingMenuVisible );
        }
    }

    private void floatingMenuEvents( final Element menuPanelElement, final Element menuContentElement )
    {
        DOM.sinkEvents( RootPanel.getBodyElement(), Event.ONCLICK | Event.ONKEYDOWN | Event.ONMOUSEOUT );
        Event.setEventListener( RootPanel.getBodyElement(), new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                if ( isFloatingMenu() && isFloatingMenuVisible() )
                {
                    if ( ( event.getTypeInt() == Event.ONKEYDOWN && event.getKeyCode() == KeyCodes.KEY_ESCAPE )
                        || ( event.getTypeInt() == Event.ONMOUSEOUT && event.getRelatedEventTarget() == null ) )
                    {
                        setFloatingMenuVisible( menuPanelElement, menuContentElement, false );
                        setActiveMenuTabId( EMPTY_VALUE );
                    }
                    else if ( event.getTypeInt() == Event.ONCLICK || isFloatingMenuItemAutoHide() )
                    {
                        String elementEventClassName = EMPTY_VALUE;
                        Element elementEvent = null;
                        if ( event.getTypeInt() == Event.ONCLICK )
                        {
                            elementEvent = Element.as( event.getEventTarget() );
                        }
                        else
                        {
                            elementEvent = Element.as( event.getRelatedEventTarget() );
                        }
                        Element eventElement = elementEvent;

                        // get parent for menu item element
                        while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                        {
                            if ( elementEvent.hasClassName( MENU_TABSHEET )
                                || elementEvent.hasClassName( COMBOBOX_OPTIONLIST_CLASS_NAME ) )
                            {
                                elementEventClassName = elementEvent.getClassName();
                            }
                            elementEvent = elementEvent.getParentElement();
                        }

                        // get id for menu item
                        while ( eventElement.hasParentElement() && ( eventElement.getId()
                                                                                 .isEmpty()
                            || ( !eventElement.getId()
                                              .startsWith( COMBOBOX_OPTIONLIST_ID )
                                && !eventElement.getId()
                                                .startsWith( MENU_ITEM_ID ) ) ) )
                        {
                            eventElement = eventElement.getParentElement();
                        }

                        // clicked out of menu area - hide float menu
                        if ( elementEventClassName.isEmpty() || ( !getMenuId().equals( eventElement.getId() )
                            && !COMBOBOX_OPTIONLIST_ID.equals( eventElement.getId() ) ) )
                        {
                            setFloatingMenuVisible( menuPanelElement, menuContentElement, false );
                            setActiveMenuTabId( EMPTY_VALUE );
                        }
                    }
                }
            }
        } );
    }

    private void floatingMenuTabsEvents( final Element menuPanelElement, final Element menuContentElement )
    {
        Element menuTabsElement = getChildElementsByClassName( getElement(), MENU_TABS ).get( 0 );
        DOM.sinkEvents( menuTabsElement, Event.ONCLICK | Event.ONMOUSEOVER );
        Event.setEventListener( menuTabsElement, new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                String elementEventClassName = EMPTY_VALUE;
                Element elementEvent = Element.as( event.getEventTarget() );
                while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                {
                    if ( elementEvent.getClassName()
                                     .contains( MENU_ITEM_ID ) )
                    {
                        elementEventClassName = elementEvent.getClassName()
                                                            .split( MENU_ITEM_ID )[1];
                        elementEventClassName = MENU_ITEM_ID + elementEventClassName.split( SPACE_VALUE )[0];
                    }
                    elementEvent = elementEvent.getParentElement();
                }

                if ( !elementEventClassName.isEmpty() )
                {
                    switch ( event.getTypeInt() )
                    {
                        case Event.ONCLICK:
                            if ( !elementEventClassName.equals( getActiveMenuTabId() ) )
                            {
                                setFloatingMenuVisible( menuPanelElement, menuContentElement, true );
                                processFloatingMenuItemChange( elementEventClassName );
                            }
                            else
                            {
                                setFloatingMenuVisible( menuPanelElement, menuContentElement,
                                                        !isFloatingMenuVisible() );
                            }
                            setActiveMenuTabId( elementEventClassName );
                            break;
                        case Event.ONMOUSEOVER:
                            if ( isFloatingMenuItemAutoShow() )
                            {
                                if ( !elementEventClassName.equals( getActiveMenuTabId() ) )
                                {
                                    setFloatingMenuVisible( menuPanelElement, menuContentElement, true );
                                    processFloatingMenuItemChange( elementEventClassName );
                                }
                                else
                                {
                                    if ( !isFloatingMenuVisible() )
                                    {
                                        setActiveMenuTabId( EMPTY_VALUE );
                                    }
                                    else
                                    {
                                        setActiveMenuTabId( getActiveMenuTabId() );
                                    }
                                }
                            }
                            break;
                        default:
                            break;
                    }
                }
            }
        } );
    }

    private void menuItemElementClickEvent( Element element )
    {
        DOM.sinkEvents( element, Event.ONCLICK | Event.ONDBLCLICK );
        Event.setEventListener( element, new EventListener()
        {
            boolean doubleClickEvent = false;

            @Override
            public void onBrowserEvent( Event event )
            {
                Element elementEvent = Element.as( event.getEventTarget() );
                elementEvent = getParentElementByClassName( elementEvent, MENU_ITEM_SLOT );
                final int elementIndex = DOM.getChildIndex( elementEvent.getParentElement(), elementEvent ) / 2;

                doubleClickEvent = false;
                if ( Event.ONDBLCLICK == event.getTypeInt() )
                {
                    doubleClickEvent = true;
                    processMenuItemElementClick( elementIndex, doubleClickEvent );
                }
                else if ( Event.ONCLICK == event.getTypeInt() )
                {
                    Scheduler.get()
                             .scheduleFixedDelay( new RepeatingCommand()
                             {
                                 @Override
                                 public boolean execute()
                                 {
                                     if ( !doubleClickEvent )
                                     {
                                         processMenuItemElementClick( elementIndex, doubleClickEvent );
                                     }
                                     return false;
                                 }
                             }, 300 );
                }
            }
        } );
    }

    public ApplicationConnection getConnection()
    {
        return connection;
    }

    public void setConnection( ApplicationConnection connection )
    {
        this.connection = connection;
    }

    public String getMenuId()
    {
        return menuId;
    }

    public void setMenuId( String menuId )
    {
        this.menuId = menuId;
        getElement().setId( getMenuId() );
    }

    public int getActiveMenuItem()
    {
        return activeMenuItem;
    }

    public void setSelectedMenuItem( int activeMenuItem )
    {
        this.activeMenuItem = activeMenuItem;
    }

    public boolean isMenuItemsSameHeight()
    {
        return menuItemsSameHeight;
    }

    public void setMenuItemsSameHeight( boolean menuItemsSameHeight )
    {
        this.menuItemsSameHeight = menuItemsSameHeight;
    }

    public boolean isMenuItemsSameHeightDefined()
    {
        return menuItemsSameHeightDefined;
    }

    public void setMenuItemsSameHeightDefined( boolean menuItemsSameHeightDefined )
    {
        this.menuItemsSameHeightDefined = menuItemsSameHeightDefined;
    }

    public void setMenuItemsSameHeight()
    {
        if ( isMenuItemsSameHeight() )
        {
            // show all menu item elements
            List<Element> elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT );
            for ( Element element : elements )
            {
                element.getStyle()
                       .clearVisibility();
            }

            // reset menu item elements height
            elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );
            for ( Element element : elements )
            {
                element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT );
                element.getStyle()
                       .clearHeight();
            }

            // get max height of menu item caption elements
            if ( !isMenuItemsSameHeightDefined() && getMenuItemCaptionMaxHeight() == 0 )
            {
                elements = getChildElementsByClassName( getElement(), MENU_ITEM_CAPTION_TOP );
                elements.addAll( getChildElementsByClassName( getElement(), MENU_ITEM_CAPTION_BOTTOM ) );
                for ( Element element : elements )
                {
                    setMenuItemCaptionMaxHeight( Math.max( getMenuItemCaptionMaxHeight(), element.getClientHeight() ) );
                }
            }

            // get max height of menu item elements
            if ( !isMenuItemsSameHeightDefined() && isMenuItemElementsSameHeight() )
            {
                elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );
                for ( Element element : elements )
                {
                    setMenuItemMaxHeight( Math.max( getMenuItemMaxHeight(), element.getClientHeight() ) );
                }
            }

            // set same height of menu item elements
            if ( isMenuItemElementsSameHeight() )
            {
                for ( int index = 0; index < getMenuItemElementsCount() && index < elements.size(); index++ )
                {
                    elements.get( index )
                            .addClassName( MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT );
                    if ( !elements.get( index )
                                  .hasClassName( MENU_ITEM_HAS_NO_CAPTION ) )
                    {
                        elements.get( index )
                                .getStyle()
                                .setHeight( getMenuItemMaxHeight(), Unit.PX );
                    }
                    else
                    {
                        elements.get( index )
                                .getStyle()
                                .setHeight( getMenuItemMaxHeight() + getMenuItemCaptionMaxHeight(), Unit.PX );
                    }
                }
            }

            // remove menu item elements which not belong to this tab
            elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT );
            for ( int index = elements.size() - 1; getMenuItemElementsCount() <= index; index-- )
            {
                foreignMenuItemElements.appendChild( elements.get( index ) );
            }

            // get menu items max height
            elements = getChildElementsByClassName( getElement(), MENU_PANEL );
            if ( !isMenuItemsSameHeightDefined() )
            {
                for ( Element element : elements )
                {
                    setMenuPanelMaxHeight( Math.max( getMenuPanelMaxHeight(), element.getClientHeight() ) );
                }
            }

            // set menu items same height
            for ( Element element : elements )
            {
                element.getStyle()
                       .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            }
        }
    }

    public boolean isMenuItemElementsSameHeight()
    {
        return menuItemElementsSameHeight;
    }

    public void setMenuItemElementsSameHeight( boolean menuItemElementsSameHeight )
    {
        this.menuItemElementsSameHeight = menuItemElementsSameHeight;
    }

    public void setMenuItemElementsSameHeight()
    {
        if ( !isMenuItemsSameHeight() )
        {
            List<Element> elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );
            List<Element> captionElements = getChildElementsByClassName( getElement(), MENU_ITEM_CAPTION_TOP );
            captionElements.addAll( getChildElementsByClassName( getElement(), MENU_ITEM_CAPTION_BOTTOM ) );

            // get max height of menu item elements
            setMenuItemMaxHeight( 0 );
            for ( Element element : elements )
            {
                setMenuItemMaxHeight( Math.max( getMenuItemMaxHeight(), element.getClientHeight() ) );
            }

            // get max height of menu item caption elements
            if ( getMenuItemCaptionMaxHeight() == 0 )
            {
                for ( Element element : captionElements )
                {
                    setMenuItemCaptionMaxHeight( Math.max( getMenuItemCaptionMaxHeight(), element.getClientHeight() ) );
                }
            }

            // reset height of menu item elements
            for ( Element element : elements )
            {
                if ( element.hasClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT ) )
                {
                    element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .clearHeight();
                }
            }

            if ( isMenuItemElementsSameHeight() )
            {
                // set same height to all menu item elements
                for ( Element element : elements )
                {
                    element.addClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .setHeight( getMenuItemMaxHeight(), Unit.PX );
                }
            }
            else
            {
                // delete height of menu item elements
                for ( Element element : elements )
                {
                    element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .clearHeight();
                }
            }

            // set menu panel height
            setMenuPanelMaxHeight( getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 )
                                                                                           .getClientHeight() );
            getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 )
                                                                   .getStyle()
                                                                   .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            if ( isFloatingMenu() && isTabBarBottom() )
            {
                getChildElementsByClassName( getElement(), MENU_CONTENT ).get( 0 )
                                                                         .getStyle()
                                                                         .setTop( -getMenuPanelMaxHeight(), Unit.PX );
            }
        }
    }

    public boolean isMenuItemAutoChange()
    {
        return menuItemAutoChange;
    }

    public void setMenuItemAutoChange( boolean menuItemAutoChange )
    {
        this.menuItemAutoChange = menuItemAutoChange;
    }

    public void setMenuItemAutoChange()
    {
        if ( isMenuItemAutoChange() )
        {
            Element menuTabsElement = getChildElementsByClassName( getElement(), MENU_TABS ).get( 0 );
            DOM.sinkEvents( menuTabsElement, Event.ONMOUSEOVER );
            Event.setEventListener( menuTabsElement, new EventListener()
            {
                @Override
                public void onBrowserEvent( Event event )
                {
                    if ( Event.ONMOUSEOVER == event.getTypeInt() )
                    {
                        String elementEventClassName = EMPTY_VALUE;
                        Element elementEvent = Element.as( event.getEventTarget() );
                        while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                        {
                            if ( elementEvent.getClassName()
                                             .contains( MENU_ITEM_ID ) )
                            {
                                elementEventClassName = elementEvent.getClassName()
                                                                    .split( MENU_ITEM_ID )[1];
                                elementEventClassName = MENU_ITEM_ID + elementEventClassName.split( SPACE_VALUE )[0];
                                processMenuItemChange( elementEventClassName );
                            }
                            elementEvent = elementEvent.getParentElement();
                        }
                    }
                }
            } );
        }
    }

    public boolean isFloatingMenu()
    {
        return floatingMenu;
    }

    public void setFloatingMenu( boolean floatingMenu )
    {
        this.floatingMenu = floatingMenu;
    }

    public void setFloatingMenu()
    {
        if ( isFloatingMenu() )
        {
            final Element menuPanelElement = getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 );
            final Element menuContentElement = getChildElementsByClassName( getElement(), MENU_CONTENT ).get( 0 );
            Element menuTabContainerElement = getChildElementsByClassName( getElement(), MENU_TAB_CONTAINER ).get( 0 );
            Element menuDecoElement = getChildElementsByClassName( getElement(), MENU_DECO ).get( 0 );

            menuContentElement.getStyle()
                              .setPosition( Position.ABSOLUTE );
            menuContentElement.getStyle()
                              .setZIndex( MENU_ZINDEX );
            menuTabContainerElement.getStyle()
                                   .setZIndex( MENU_ZINDEX );
            menuDecoElement.getStyle()
                           .setZIndex( MENU_ZINDEX );
            menuDecoElement.getStyle()
                           .setMarginBottom( 0, Unit.PX );
            menuDecoElement.getStyle()
                           .setBottom( -getMenuPanelMaxHeight() / 2, Unit.PX );
            setFloatingMenuVisible( menuPanelElement, menuContentElement, isFloatingMenuVisible() );

            // tab's events
            floatingMenuTabsEvents( menuPanelElement, menuContentElement );

            // browser's events
            floatingMenuEvents( menuPanelElement, menuContentElement );
        }
    }

    public boolean isFloatingMenuItemAutoHide()
    {
        return floatingMenuItemAutoHide;
    }

    public void setFloatingMenuItemAutoHide( boolean floatingMenuItemAutoHide )
    {
        this.floatingMenuItemAutoHide = floatingMenuItemAutoHide;
    }

    public boolean isFloatingMenuItemAutoShow()
    {
        return floatingMenuItemAutoChange;
    }

    public void setFloatingMenuItemAutoShow( boolean floatingMenuItemAutoShow )
    {
        this.floatingMenuItemAutoChange = floatingMenuItemAutoShow;
    }

    public int getMenuItemElementsCount()
    {
        return menuItemElementsCount;
    }

    public void setMenuItemElementsCount( int menuItemElementsCount )
    {
        this.menuItemElementsCount = menuItemElementsCount;
    }

    public void resetHiddenMenuItemElements()
    {
        // prepare index for visible item elements
        if ( firstVisibleMenuItemElementIndex.isEmpty() )
        {
            for ( int index = 0; index < getTabCount(); index++ )
            {
                firstVisibleMenuItemElementIndex.add( 0 );
            }
        }

        // remove all child elements from hiddenMenuItemElementsNotBelong
        if ( foreignMenuItemElements == null )
        {
            foreignMenuItemElements = DOM.createDiv();
        }
        else
        {
            foreignMenuItemElements.removeAllChildren();
        }
    }

    public void checkMenuItemElementsVisibility( boolean moveMenuItemElementsToItsParent )
    {
        int firstVisibleMenuItemIndex = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() );
        Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );
        Element menuPanelElement = getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 );
        NodeList<Node> menuLayoutNodes = menuLayoutElement.getChildNodes();

        menuLayoutElementPaddingLeft = Math.min( menuLayoutElementPaddingLeft, LayoutManager.get( getConnection() )
                                                                                            .getPaddingLeft( menuLayoutElement ) );
        menuLayoutElementPaddingRight = Math.min( menuLayoutElementPaddingRight, LayoutManager.get( getConnection() )
                                                                                              .getPaddingRight( menuLayoutElement ) );

        // remove unused spacing elements after last menu item element
        if ( 1 < menuLayoutNodes.getLength() && Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) )
                                                       .hasClassName( MENU_SPACING )
            && !Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 2 ) )
                       .hasClassName( MENU_SPACING ) )
        {
            while ( Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) )
                           .hasClassName( MENU_SPACING ) )
            {
                foreignMenuItemElements.appendChild( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) );
            }
        }

        // append scrollers
        menuPanelElement.insertFirst( scrollerPrev );
        menuPanelElement.appendChild( scrollerNext );

        // menu item width
        int menuItemWidth = menuPanelElement.getClientWidth();

        // left scroller is visible
        if ( 0 < firstVisibleMenuItemIndex )
        {
            menuItemWidth -= scrollerPrev.getOffsetWidth();
        }

        // menu item elements width
        int menuItemElementsWidth = menuLayoutElementPaddingLeft + menuLayoutElementPaddingRight;

        // hide menu item elements which index is smaller than firstVisibleMenuItemIndex
        boolean scrollerPrevIsHidden = true;
        for ( int index = 0; index < firstVisibleMenuItemIndex; index++ )
        {
            scrollerPrevIsHidden = false;
            Element element = Element.as( menuLayoutNodes.getItem( index ) );
            element.getStyle()
                   .setVisibility( Visibility.HIDDEN );
            element.getStyle()
                   .setWidth( 0, Unit.PX );
        }

        // hide/show menu item elements of menuLayoutNodes
        boolean scrollerNextIsHidden = true;
        for ( int index = firstVisibleMenuItemIndex; index < menuLayoutNodes.getLength(); index++ )
        {
            Element element = Element.as( menuLayoutNodes.getItem( index ) );
            element.getStyle()
                   .clearWidth();
            menuItemElementsWidth += element.getOffsetWidth();
            if ( menuItemWidth < menuItemElementsWidth )
            {
                // hide menu item elements which are out from visible area
                if ( scrollerNextIsHidden )
                {
                    index = Math.max( 1, index - 2 );
                    scrollerNextIsHidden = false;
                    menuItemWidth -= scrollerNext.getOffsetWidth();
                }
                else
                {
                    element.getStyle()
                           .setVisibility( Visibility.HIDDEN );
                    element.getStyle()
                           .setWidth( 0, Unit.PX );
                }
            }
            else
            {
                // show menu item elements which are in visible area
                element.getStyle()
                       .clearVisibility();
                menuItemElementClickEvent( element );
            }
        }

        // set menu items elements height
        if ( isFloatingMenu() && isFloatingMenuVisible() && !isMenuItemsSameHeight() )
        {
            setMenuItemElementsSameHeight();
        }

        // hide/show scrollerPrev
        if ( scrollerPrevIsHidden )
        {
            menuLayoutElement.getStyle()
                             .clearPaddingLeft();
            scrollerPrev.getStyle()
                        .setVisibility( Visibility.HIDDEN );
        }
        else
        {
            menuLayoutElement.getStyle()
                             .setPaddingLeft( scrollerPrev.getOffsetWidth() + menuLayoutElementPaddingLeft, Unit.PX );
            scrollerPrev.getStyle()
                        .clearVisibility();
        }

        // hide/show scrollerNext
        if ( scrollerNextIsHidden )
        {
            scrollerNext.getStyle()
                        .setVisibility( Visibility.HIDDEN );
        }
        else
        {
            scrollerNext.getStyle()
                        .clearVisibility();
        }

        // move menu item elements to its parent
        if ( moveMenuItemElementsToItsParent )
        {
            processMoveMenuItemElementsToItsParent();
        }
    }

    @Override
    public HandlerRegistration addValueChangeHandler( ValueChangeHandler<Object[]> handler )
    {
        return this.addHandler( handler, ValueChangeEvent.getType() );
    }

}
