/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.menuitem;

import java.util.Iterator;

import com.vaadin.ui.Button;
import com.vaadin.ui.CheckBox;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.Component;
import com.vaadin.ui.DateField;
import com.vaadin.ui.DragAndDropWrapper;
import com.vaadin.ui.DragAndDropWrapper.DragStartMode;
import com.vaadin.ui.HasComponents;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Link;
import com.vaadin.ui.MenuBar;
import com.vaadin.ui.OptionGroup;
import com.vaadin.ui.Panel;
import com.vaadin.ui.Table;
import com.vaadin.ui.TextArea;
import com.vaadin.ui.TextField;
import com.vaadin.ui.TreeTable;
import com.vaadin.ui.VerticalLayout;

public class MenuItemElement
    extends Panel
{

    private static final long serialVersionUID = -4754397816991773376L;

    private static final String CONTEXT_IS_EMPTY = "Context is empty.";

    private static final String COMPONENT_SMALL = "small";

    private static final String COMPONENT_TINY = "tiny";

    private static final String MENU_ITEM = "v-dmenu-item";

    private static final String MENU_ITEM_NO_CAPTION = CaptionLocation.NO_CAPTION.getLocation();

    private static final String MENU_ITEM_CAPTION = MENU_ITEM + "-caption-";

    private static final String MENU_ITEM_CONTEXT = MENU_ITEM + "-context";

    private boolean menuItemElementDraggable = false;

    private boolean menuItemElementEnabled = true;

    private boolean menuItemElementReadOnly = false;

    private boolean menuItemElementVisible = false;

    private int parentMenuItemIndex = 0;

    private Panel caption = null;

    private CaptionLocation captionLocation = null;

    private DragAndDropWrapper contextDragWrapper = null;

    private DragStartMode menuItemElementDragMode = null;

    private VerticalLayout context = null;

    private Component menuItemElementComponent = null;

    private ComponentViewState componentViewState = null;

    private MenuItemElement()
    {
        componentViewState = new ComponentViewState();
        prepareContent();
    }

    public MenuItemElement( String caption )
    {
        this();
        setCaption( caption );
    }

    public MenuItemElement( String caption, Component component )
    {
        this();
        setCaption( caption );
        setContext( component );
    }

    private void prepareContent()
    {
        addStyleName( MENU_ITEM );
        setSizeUndefined();
        context = new VerticalLayout();
        context.setSizeUndefined();
        context.setMargin( false );

        // set menu item element drag enabled/disabled
        contextDragWrapper = new DragAndDropWrapper( context );
        setMenuItemElementDragMode( DragStartMode.WRAPPER );

        // add context drag wrapper to content
        setContent( contextDragWrapper );
        captionLocation = CaptionLocation.BOTTOM;
    }

    private void addSmallOrTinyStyle( Component component )
    {
        if ( component instanceof CheckBox || component instanceof Link || component instanceof MenuBar )
        {
            component.addStyleName( COMPONENT_SMALL );
        }
        else if ( component instanceof OptionGroup || component instanceof Table || component instanceof TreeTable )
        {
            component.addStyleName( COMPONENT_SMALL );
        }
        else if ( component instanceof Button || component instanceof ComboBox || component instanceof DateField )
        {
            component.addStyleName( COMPONENT_TINY );
        }
        else if ( component instanceof Label || component instanceof TextArea || component instanceof TextField )
        {
            component.addStyleName( COMPONENT_TINY );
        }
        else if ( component instanceof HasComponents )
        {
            for ( Iterator<Component> iterator = ( (HasComponents) component ).iterator(); iterator.hasNext(); )
            {
                addSmallOrTinyStyle( iterator.next() );
            }
        }
    }

    private void setComponentEnabled( Component component, boolean enabled )
    {
        if ( component != null )
        {
            if ( component instanceof HasComponents )
            {
                boolean isProcessed = false;
                for ( Iterator<Component> iterator = ( (HasComponents) component ).iterator(); iterator.hasNext(); )
                {
                    isProcessed = true;
                    setComponentEnabled( iterator.next(), enabled );
                }

                // ensure that property is set to component
                if ( !isProcessed )
                {
                    componentViewState.setEnabled( component, enabled );
                }
            }
            else
            {
                componentViewState.setEnabled( component, enabled );
            }

            setMenuItemElementDraggable( menuItemElementDraggable );
        }
    }

    private void setComponentReadOnly( Component component, boolean readOnly )
    {
        if ( component != null )
        {
            boolean isProcessed = false;
            if ( component instanceof HasComponents )
            {
                for ( Iterator<Component> iterator = ( (HasComponents) component ).iterator(); iterator.hasNext(); )
                {
                    setComponentReadOnly( iterator.next(), readOnly );
                }

                // ensure that property is set to component
                if ( !isProcessed )
                {
                    componentViewState.setReadOnly( component, readOnly );
                }
            }
            else
            {
                componentViewState.setReadOnly( component, readOnly );
            }

            setMenuItemElementDraggable( menuItemElementDraggable );
        }
    }

    private void setComponentVisible( Component component, boolean visible )
    {
        if ( component != null )
        {
            if ( component instanceof HasComponents )
            {
                boolean isProcessed = false;
                for ( Iterator<Component> iterator = ( (HasComponents) component ).iterator(); iterator.hasNext(); )
                {
                    isProcessed = true;
                    setComponentVisible( iterator.next(), visible );
                }

                // ensure that property is set to component
                if ( !isProcessed )
                {
                    componentViewState.setVisible( component, visible );
                }
            }
            else
            {
                componentViewState.setVisible( component, visible );
            }

            setMenuItemElementDraggable( menuItemElementDraggable );
        }
    }

    public Component getMenuItemElementComponent()
    {
        return menuItemElementComponent;
    }

    @Override
    public boolean isEnabled()
    {
        return menuItemElementEnabled;
    }

    @Override
    public void setEnabled( boolean enabled )
    {
        menuItemElementEnabled = enabled;
        if ( caption != null )
        {
            caption.setEnabled( enabled );
        }
        if ( menuItemElementComponent != null )
        {
            setComponentEnabled( menuItemElementComponent, enabled );
        }
    }

    @Override
    public boolean isReadOnly()
    {
        return menuItemElementReadOnly;
    }

    @Override
    public void setReadOnly( boolean readOnly )
    {
        menuItemElementReadOnly = readOnly;
        if ( caption != null )
        {
            caption.setReadOnly( readOnly );
        }
        if ( menuItemElementComponent != null )
        {
            setComponentReadOnly( menuItemElementComponent, readOnly );
        }
    }

    @Override
    public boolean isVisible()
    {
        return menuItemElementVisible;
    }

    @Override
    public void setVisible( boolean visible )
    {
        menuItemElementVisible = visible;
        if ( caption != null )
        {
            caption.setEnabled( visible );
        }
        if ( menuItemElementComponent != null )
        {
            setComponentVisible( menuItemElementComponent, visible );
        }
    }

    public boolean isMenuItemElementDraggable()
    {
        return menuItemElementDraggable;
    }

    public void setMenuItemElementDraggable( boolean menuItemElementDraggable )
    {
        this.menuItemElementDraggable = menuItemElementDraggable;
        if ( contextDragWrapper != null )
        {
            if ( isMenuItemElementDraggable() && isEnabled() && !isReadOnly() )
            {
                contextDragWrapper.setDragStartMode( getMenuItemElementDragMode() );
            }
            else
            {
                contextDragWrapper.setDragStartMode( DragStartMode.NONE );
            }
            markAsDirty();
        }
    }

    public DragStartMode getMenuItemElementDragMode()
    {
        return menuItemElementDragMode;
    }

    public void setMenuItemElementDragMode( DragStartMode menuItemElementDragMode )
    {
        this.menuItemElementDragMode = menuItemElementDragMode;
        if ( menuItemElementDragMode != null && menuItemElementDragMode instanceof DragStartMode )
        {
            setMenuItemElementDraggable( isMenuItemElementDraggable() );
        }
    }

    public int getParentMenuItemIndex()
    {
        return parentMenuItemIndex;
    }

    public void setParentMenuItemIndex( int parentMenuItemIndex )
    {
        this.parentMenuItemIndex = parentMenuItemIndex;
    }

    public String getCaption()
    {
        if ( caption != null )
        {
            return caption.getCaption();
        }
        return null;
    }

    public void setCaption( String caption )
    {
        if ( caption != null && !caption.isEmpty() )
        {
            this.caption = new Panel( caption );
            this.caption.addStyleName( MENU_ITEM_CAPTION + getCaptionLocation().getLocation() );
            this.caption.setSizeFull();
            this.caption.setHeightUndefined();
            this.caption.setEnabled( menuItemElementEnabled );
            this.caption.setReadOnly( menuItemElementReadOnly );
        }
        else
        {
            setCaptionLocation( CaptionLocation.NO_CAPTION );
        }
    }

    public Component getContext()
    {
        return getContent();
    }

    public void setContext( Component context )
    {
        if ( context == null )
        {
            throw new IllegalArgumentException( CONTEXT_IS_EMPTY );
        }

        this.context.removeAllComponents();
        context.removeStyleName( MENU_ITEM_NO_CAPTION );

        if ( caption != null && captionLocation == CaptionLocation.TOP )
        {
            this.context.addComponent( caption );
        }

        // all base components have to have small or tiny style
        addSmallOrTinyStyle( context );

        if ( context instanceof HorizontalLayout )
        {
            ( (HorizontalLayout) context ).setMargin( true );
            ( (HorizontalLayout) context ).setSpacing( true );
            ( (HorizontalLayout) context ).setWidthUndefined();
            context.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                context.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( context );
        }
        else if ( context instanceof VerticalLayout )
        {
            ( (VerticalLayout) context ).setMargin( true );
            ( (VerticalLayout) context ).setSpacing( true );
            ( (VerticalLayout) context ).setWidthUndefined();
            context.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                context.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( context );
        }
        else
        {
            VerticalLayout elementContext = new VerticalLayout();
            elementContext.setMargin( true );
            elementContext.setSpacing( true );
            elementContext.setWidthUndefined();
            elementContext.addComponent( context );
            elementContext.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                elementContext.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( elementContext );
        }
        setComponentEnabled( context, menuItemElementEnabled );
        setComponentReadOnly( context, menuItemElementReadOnly );
        menuItemElementComponent = context;

        if ( caption != null && captionLocation == CaptionLocation.BOTTOM )
        {
            this.context.addComponent( caption );
        }
    }

    public CaptionLocation getCaptionLocation()
    {
        return captionLocation;
    }

    public void setCaptionLocation( CaptionLocation captionLocation )
    {
        if ( caption == null || captionLocation == null || captionLocation == CaptionLocation.NO_CAPTION )
        {
            caption = null;
            this.captionLocation = CaptionLocation.NO_CAPTION;
        }
        else
        {
            this.captionLocation = captionLocation;
            caption.removeStyleName( MENU_ITEM_CAPTION + CaptionLocation.BOTTOM.getLocation() );
            caption.removeStyleName( MENU_ITEM_CAPTION + CaptionLocation.TOP.getLocation() );
            caption.addStyleName( MENU_ITEM_CAPTION + captionLocation.getLocation() );
        }

        if ( menuItemElementComponent != null )
        {
            setContext( menuItemElementComponent );
        }
    }

}
