/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dussan.vaadin.dmenu.client.rpc.MenuServerRpc;
import org.dussan.vaadin.dmenu.client.state.MenuState;
import org.dussan.vaadin.dmenu.events.FloatMenuItemChangeEvent;
import org.dussan.vaadin.dmenu.events.FloatMenuItemChangeHandler;
import org.dussan.vaadin.dmenu.events.FloatMenuItemLockEvent;
import org.dussan.vaadin.dmenu.events.FloatMenuItemLockHandler;
import org.dussan.vaadin.dmenu.events.FloatMenuItemShowEvent;
import org.dussan.vaadin.dmenu.events.FloatMenuItemShowHandler;
import org.dussan.vaadin.dmenu.events.MenuItemChangeEvent;
import org.dussan.vaadin.dmenu.events.MenuItemChangeHandler;
import org.dussan.vaadin.dmenu.events.MenuItemElementClickEvent;
import org.dussan.vaadin.dmenu.events.MenuItemElementClickHandler;
import org.dussan.vaadin.dmenu.helper.ManifestHelper;
import org.dussan.vaadin.dmenu.menuitem.MenuItem;
import org.dussan.vaadin.dmenu.menuitem.MenuItemElement;
import org.dussan.vaadin.dtabs.DTabs;
import org.dussan.vaadin.dtabs.DTabsProperty;

import com.google.gwt.event.shared.HandlerManager;
import com.vaadin.server.Resource;
import com.vaadin.ui.Component;
import com.vaadin.ui.HorizontalLayout;

public class DMenu
    extends DTabs
{

    private static final long serialVersionUID = -1766784516812336397L;

    private static final String VERSION = "Implementation-Version";

    private static final String GIT_VERSION = "Git-Version";

    private static final String MENU_ITEM_ID = "dmenu-";

    private static final String MENU_ITEM_IS_EMPTY = "Menu item is empty.";

    private static final String MENU_ALREADY_HAVE_THIS_ITEM = "Menu already have this item.";

    private static final String INDEX_HAVE_TO_BE_GREATER_THAN_OR_EQUAL_0 = "Index have to be greater than or equal 0.";

    private static final String COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM = "Component is not instance of MenuItem.";

    private static final String MENU_ID = MENU_ITEM_ID + "{0}";

    private boolean menuTabsAutoChange = false;

    private boolean menuItemsSameHeight = false;

    private boolean menuItemElementsSameHeight = false;

    private boolean floatingMenu = false;

    private boolean floatingMenuLock = false;

    private boolean floatingMenuItemAutoHide = false;

    private boolean floatingMenuItemAutoShow = false;

    private int activeMenuItem = 0;

    private int menuItemCounter = 0;

    private int menuPanelMaxHeight = 0;

    private String menuId = null;

    private List<MenuItem> menuItems = null;

    private Map<Integer, Integer> firstVisibleMenuItemElement = null;

    private HandlerManager handlerManager = null;

    public DMenu()
    {
        firstVisibleMenuItemElement = new HashMap<>();
        menuItems = new ArrayList<>();
        menuId =
            MessageFormat.format( MENU_ID, Long.toString( Math.abs( new Date().getTime() * new Date().getTime() ) ) );

        setProperties( DTabsProperty.COMPACTED_TABBAR, DTabsProperty.FRAMED_TABS );
        setMenuItemsSameHeight( true );
        setMenuItemElementsSameHeight( true );

        // events handler manager
        handlerManager = new HandlerManager( this );

        registerRpc( new MenuServerRpc()
        {
            private static final long serialVersionUID = 9039890628414457601L;

            @Override
            public void firstVisibleMenuItemElementChange( Integer activeMenuItem, Integer firstVisibleMenuItemElement )
            {
                setFirstVisibleMenuItemElement( activeMenuItem, firstVisibleMenuItemElement );
            }

            @Override
            public void floatingMenuItemChange( String menuItemId, Integer menuPanelMaxHeight )
            {
                setMenuPanelMaxHeight( menuPanelMaxHeight );
                for ( int index = 0; index < getMenuItems().size(); index++ )
                {
                    if ( menuItemId.equals( getTab( index ).getStyleName() ) )
                    {
                        setActiveMenuItem( index );
                        handlerManager.fireEvent( new FloatMenuItemChangeEvent( getMenuItem( getActiveMenuItem() ) ) );
                        break;
                    }
                }
            }

            @Override
            public void floatingMenuItemShow( Boolean floatingMenuShow )
            {
                handlerManager.fireEvent( new FloatMenuItemShowEvent( getMenuItem( getActiveMenuItem() ),
                                                                      floatingMenuShow ) );
            }

            @Override
            public void floatingMenuLocked( Boolean floatingMenuLocked )
            {
                setFloatingMenu( floatingMenuLocked );
                handlerManager.fireEvent( new FloatMenuItemLockEvent( !floatingMenuLocked ) );
            }

            @Override
            public void menuItemChange( String activeTabId, Integer menuPanelMaxHeight )
            {
                setMenuPanelMaxHeight( menuPanelMaxHeight );
                for ( int index = 0; index < getMenuItems().size(); index++ )
                {
                    if ( activeTabId.equals( getTab( index ).getStyleName() ) )
                    {
                        setActiveMenuItem( index );
                        handlerManager.fireEvent( new MenuItemChangeEvent( getMenuItem( getActiveMenuItem() ) ) );
                        break;
                    }
                }
            }

            @Override
            public void menuItemElementClick( Integer menuItemElementId, Boolean doubleClick )
            {
                MenuItemElement menuItemElement =
                    getMenuItem( getActiveMenuItem() ).getMenuItemElement( menuItemElementId );
                if ( menuItemElement.isEnabled() && !menuItemElement.isReadOnly() )
                {
                    handlerManager.fireEvent( new MenuItemElementClickEvent( menuItemElement, doubleClick ) );
                }
            }

            @Override
            public void moveMenuItemElementsToItsParent()
            {
                for ( int index = 0; index < menuItems.size(); index++ )
                {
                    for ( MenuItemElement menuItemElement : getMenuItem( index ).getMenuItemElements() )
                    {
                        if ( menuItemElement.getParentMenuItemIndex() != index )
                        {
                            menuItems.get( menuItemElement.getParentMenuItemIndex() )
                                     .addComponent( menuItemElement );
                        }
                    }
                }
            }
        } );

        addSelectedTabChangeListener( new SelectedTabChangeListener()
        {
            private static final long serialVersionUID = -1047609548802091735L;

            @Override
            public void selectedTabChange( SelectedTabChangeEvent event )
            {
                setActiveMenuItem( getMenuItemIndex( event.getTabSheet()
                                                          .getSelectedTab() ) );
                if ( isFloatingMenu() )
                {
                    handlerManager.fireEvent( new FloatMenuItemChangeEvent( getMenuItem( getActiveMenuItem() ) ) );
                }
                else
                {
                    handlerManager.fireEvent( new MenuItemChangeEvent( getMenuItem( getActiveMenuItem() ) ) );
                }
            }
        } );
    }

    public DMenu( MenuItem... menuItems )
    {
        this();
        addMenuItems( menuItems );
    }

    public void addHandler( FloatMenuItemChangeHandler handler )
    {
        handlerManager.addHandler( FloatMenuItemChangeEvent.getType(), handler );
    }

    public void removeHandler( FloatMenuItemChangeHandler handler )
    {
        if ( handlerManager.isEventHandled( FloatMenuItemChangeEvent.getType() ) )
        {
            handlerManager.removeHandler( FloatMenuItemChangeEvent.getType(), handler );
        }
    }

    public void addHandler( FloatMenuItemShowHandler handler )
    {
        handlerManager.addHandler( FloatMenuItemShowEvent.getType(), handler );
    }

    public void removeHandler( FloatMenuItemShowHandler handler )
    {
        if ( handlerManager.isEventHandled( FloatMenuItemShowEvent.getType() ) )
        {
            handlerManager.removeHandler( FloatMenuItemShowEvent.getType(), handler );
        }
    }

    public void addHandler( FloatMenuItemLockHandler handler )
    {
        handlerManager.addHandler( FloatMenuItemLockEvent.getType(), handler );
    }

    public void removeHandler( FloatMenuItemLockHandler handler )
    {
        if ( handlerManager.isEventHandled( FloatMenuItemLockEvent.getType() ) )
        {
            handlerManager.removeHandler( FloatMenuItemLockEvent.getType(), handler );
        }
    }

    public void addHandler( MenuItemChangeHandler handler )
    {
        handlerManager.addHandler( MenuItemChangeEvent.getType(), handler );
    }

    public void removeHandler( MenuItemChangeHandler handler )
    {
        if ( handlerManager.isEventHandled( MenuItemChangeEvent.getType() ) )
        {
            handlerManager.removeHandler( MenuItemChangeEvent.getType(), handler );
        }
    }

    public void addHandler( MenuItemElementClickHandler handler )
    {
        handlerManager.addHandler( MenuItemElementClickEvent.getType(), handler );
    }

    public void removeHandler( MenuItemElementClickHandler handler )
    {
        if ( handlerManager.isEventHandled( MenuItemElementClickEvent.getType() ) )
        {
            handlerManager.removeHandler( MenuItemElementClickEvent.getType(), handler );
        }
    }

    @Override
    public void setSelectedTab( int position )
    {
        setActiveMenuItem( position );
    }

    public int getActiveMenuItem()
    {
        activeMenuItem = getMenuItemIndex( getSelectedTab() );
        return activeMenuItem;
    }

    public void setActiveMenuItem( int position )
    {
        super.setSelectedTab( Math.max( 0, Math.min( position, menuItems.size() - 1 ) ) );
    }

    public void setSelectedMenuItem( MenuItem menuItem )
    {
        super.setSelectedTab( getMenuItemIndex( menuItem ) );
    }

    public Integer getFirstVisibleMenuItemElement()
    {
        if ( ( getActiveMenuItem() + 1 ) <= this.firstVisibleMenuItemElement.size() )
        {
            return firstVisibleMenuItemElement.get( getActiveMenuItem() );
        }
        return 0;
    }

    public void setFirstVisibleMenuItemElement( int activeMenuItem, int firstVisibleMenuItemElement )
    {
        this.firstVisibleMenuItemElement.put( activeMenuItem, firstVisibleMenuItemElement );
    }

    public int getMenuPanelMaxHeight()
    {
        return menuPanelMaxHeight;
    }

    public void setMenuPanelMaxHeight( int menuPanelMaxHeight )
    {
        this.menuPanelMaxHeight = Math.max( this.menuPanelMaxHeight, menuPanelMaxHeight );
    }

    public boolean isMenuItemsSameHeight()
    {
        return menuItemsSameHeight;
    }

    public void setMenuItemsSameHeight( boolean menuItemsSameHeight )
    {
        if ( isMenuItemsSameHeight() != menuItemsSameHeight )
        {
            this.menuItemsSameHeight = menuItemsSameHeight;
            getState( true ).setMenuItemsSameHeight( menuItemsSameHeight );
        }
    }

    public boolean isMenuItemElementsSameHeight()
    {
        return menuItemElementsSameHeight;
    }

    public void setMenuItemElementsSameHeight( boolean menuItemElementsSameHeight )
    {
        if ( isMenuItemElementsSameHeight() != menuItemElementsSameHeight )
        {
            this.menuItemElementsSameHeight = menuItemElementsSameHeight;
            getState( true ).setMenuItemElementsSameHeight( menuItemElementsSameHeight );
        }
    }

    public boolean isMenuTabsAutoChange()
    {
        return menuTabsAutoChange;
    }

    public void setMenuTabsAutoChange( boolean menuTabsAutoChange )
    {
        if ( isMenuTabsAutoChange() != menuTabsAutoChange )
        {
            this.menuTabsAutoChange = menuTabsAutoChange;
            getState( true ).setMenuItemAutoChange( menuTabsAutoChange );
        }
    }

    public boolean isFloatingMenu()
    {
        return floatingMenu;
    }

    public void setFloatingMenu( boolean floatingMenu )
    {
        if ( isFloatingMenu() != floatingMenu )
        {
            this.floatingMenu = floatingMenu;
            getState( true ).setFloatingMenu( floatingMenu );
        }
    }

    public boolean isFloatingMenuLockEnabled()
    {
        return floatingMenuLock;
    }

    public void setFloatingMenuLockEnabled( boolean floatingMenuLock )
    {
        if ( isFloatingMenuLockEnabled() != floatingMenuLock )
        {
            this.floatingMenuLock = floatingMenuLock;
            getState( true ).setFloatingMenuLock( floatingMenuLock );
        }
    }

    public boolean isFloatingMenuItemAutoHide()
    {
        return floatingMenuItemAutoHide;
    }

    public void setFloatingMenuItemAutoHide( boolean floatingMenuItemAutoHide )
    {
        if ( isFloatingMenuItemAutoHide() != floatingMenuItemAutoHide )
        {
            this.floatingMenuItemAutoHide = floatingMenuItemAutoHide;
            getState( true ).setFloatingMenuItemAutoHide( floatingMenuItemAutoHide );
        }
    }

    public boolean isFloatingMenuItemAutoShow()
    {
        return floatingMenuItemAutoShow;
    }

    public void setFloatingMenuItemAutoShow( boolean floatingMenuItemAutoShow )
    {
        if ( isFloatingMenuItemAutoShow() != floatingMenuItemAutoShow )
        {
            this.floatingMenuItemAutoShow = floatingMenuItemAutoShow;
            getState( true ).setFloatingMenuItemAutoShow( floatingMenuItemAutoShow );
        }
    }

    public List<MenuItem> getMenuItems()
    {
        return menuItems;
    }

    public void addMenuItems( MenuItem... menuItems )
    {
        for ( MenuItem menuItem : menuItems )
        {
            addMenuItem( menuItem );
        }
    }

    public int getMenuItemIndex( Component component )
    {
        return menuItems.indexOf( component );
    }

    public MenuItem getMenuItem( int index )
    {
        return menuItems.get( index );
    }

    public void addMenuItem( MenuItem menuItem )
    {
        if ( menuItem == null || menuItem.isEmpty() )
        {
            throw new IllegalArgumentException( MENU_ITEM_IS_EMPTY );
        }
        else if ( menuItems.contains( menuItem ) )
        {
            throw new IllegalArgumentException( MENU_ALREADY_HAVE_THIS_ITEM );
        }

        addMenuItem( menuItem, null, menuItems.size() );
    }

    public void addMenuItem( MenuItem menuItem, int position )
    {
        if ( menuItem == null || menuItem.isEmpty() )
        {
            throw new IllegalArgumentException( MENU_ITEM_IS_EMPTY );
        }
        else if ( menuItems.contains( menuItem ) )
        {
            throw new IllegalArgumentException( MENU_ALREADY_HAVE_THIS_ITEM );
        }
        else if ( position < 0 )
        {
            throw new IllegalArgumentException( INDEX_HAVE_TO_BE_GREATER_THAN_OR_EQUAL_0 );
        }

        addMenuItem( menuItem, null, position );
    }

    public void addMenuItem( MenuItem menuItem, Resource icon, int position )
    {
        if ( menuItem == null || menuItem.isEmpty() )
        {
            throw new IllegalArgumentException( MENU_ITEM_IS_EMPTY );
        }
        else if ( menuItems.contains( menuItem ) )
        {
            throw new IllegalArgumentException( MENU_ALREADY_HAVE_THIS_ITEM );
        }
        else if ( position < 0 )
        {
            throw new IllegalArgumentException( INDEX_HAVE_TO_BE_GREATER_THAN_OR_EQUAL_0 );
        }

        menuItem.setParentMenuItemIndex( position );
        menuItems.add( position, menuItem );
        super.addTab( menuItem, menuItem.getCaption(), icon, position );
        super.getTab( menuItem ).setStyleName( MENU_ITEM_ID + menuItemCounter++ );
    }

    public void removeMenuItem( int position )
    {
        menuItems.remove( position );
        super.removeTab( super.getTab( position ) );
    }

    public boolean isEmpty()
    {
        return menuItems.isEmpty();
    }

    public static String getVersion()
        throws IOException
    {
        if ( ManifestHelper.getManifest() != null )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( VERSION );
        }
        return null;
    }

    public static String getGitVersion()
        throws IOException
    {
        if ( ManifestHelper.getManifest() != null )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( GIT_VERSION );
        }
        return null;
    }

    @Override
    protected MenuState getState()
    {
        return (MenuState) super.getState();
    }

    @Override
    protected MenuState getState( boolean markAsDirty )
    {
        return (MenuState) super.getState( markAsDirty );
    }

    @Override
    public Tab addTab( Component c )
    {
        if ( !( c instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        return super.addTab( c );
    }

    @Override
    public Tab addTab( Component c, String caption )
    {
        if ( !( c instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        return super.addTab( c, caption );
    }

    @Override
    public Tab addTab( Component c, String caption, Resource icon )
    {
        if ( !( c instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        return super.addTab( c, caption, icon );
    }

    @Override
    public Tab addTab( Component component, int position )
    {
        if ( !( component instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        return super.addTab( component, position );
    }

    @Override
    public Tab addTab( Component tabComponent, String caption, Resource icon, int position )
    {
        if ( !( tabComponent instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        return super.addTab( tabComponent, caption, icon, position );
    }

    @Override
    public void addComponent( Component c )
    {
        if ( !( c instanceof MenuItem ) )
        {
            throw new IllegalArgumentException( COMPONENT_IS_NOT_INSTANCE_OF_MENU_ITEM );
        }
        addMenuItem( (MenuItem) c );
    }

    @Override
    public void addComponents( Component... components )
    {
        for ( Component component : components )
        {
            addComponent( component );
        }
    }

    @Override
    public void beforeClientResponse( boolean initial )
    {
        super.beforeClientResponse( initial );
        getState( true ).setMenuId( menuId );
        getState( true ).setActiveMenuItem( getActiveMenuItem() );
        getState( true ).setFirstVisibleMenuItemElement( getFirstVisibleMenuItemElement() );
        getState( true ).setMenuPanelMaxHeight( getMenuPanelMaxHeight() );

        if ( isMenuItemsSameHeight() )
        {
            getState( true ).setMenuItemsSameHeightDefined( !initial );
            getState( true ).setMenuItemElementsCount( menuItems.get( getActiveMenuItem() )
                                                                .getMenuItemElements()
                                                                .size() );
            if ( initial )
            {
                // move all menu item elements to active menu item
                // so we can get max height of menu item elements
                HorizontalLayout layout = (HorizontalLayout) getTab( getActiveMenuItem() ).getComponent();
                for ( int index = 0; index < getComponentCount(); index++ )
                {
                    if ( index != getActiveMenuItem() )
                    {
                        layout.moveComponentsFrom( (HorizontalLayout) getTab( index ).getComponent() );
                    }
                }
            }
        }
    }

}
