/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.menuitem;

import java.util.ArrayList;
import java.util.List;

import com.vaadin.ui.DragAndDropWrapper.DragStartMode;
import com.vaadin.ui.HorizontalLayout;

public class MenuItem
    extends HorizontalLayout
{

    private static final long serialVersionUID = 1102855995145820363L;

    private static final String MENU = "v-dmenu";

    private static final String MENU_ELEMENT_IS_EMPTY = "Menu element is empty.";

    private static final String INDEX_HAVE_TO_BE_GREATER_THAN_OR_EQUAL_0 = "Index have to be greater than or equal 0.";

    private boolean menuItemElementsDraggable = false;

    private int parentMenuItemIndex = 0;

    private CaptionLocation menuItemElementsCaptionLocation = null;

    private DragStartMode menuItemElementsDragMode = DragStartMode.WRAPPER;

    private MenuItem()
    {
        addStyleName( MENU );
        setMargin( true );
        setSpacing( true );
    }

    public MenuItem( String caption )
    {
        this();
        setCaption( caption );
    }

    public int getParentMenuItemIndex()
    {
        return parentMenuItemIndex;
    }

    public MenuItem setParentMenuItemIndex( int parentMenuItemIndex )
    {
        this.parentMenuItemIndex = parentMenuItemIndex;
        for ( MenuItemElement menuItemElement : getMenuItemElements() )
        {
            menuItemElement.setParentMenuItemIndex( parentMenuItemIndex );
        }
        return this;
    }

    public boolean isEmpty()
    {
        return getMenuItemElements().isEmpty();
    }

    public List<MenuItemElement> getMenuItemElements()
    {
        List<MenuItemElement> menuItemElements = new ArrayList<>();
        for ( int index = 0; index < getComponentCount(); index++ )
        {
            menuItemElements.add( getMenuItemElement( index ) );
        }
        return menuItemElements;
    }

    public MenuItem addMenuItemElements( MenuItemElement... menuItemElements )
    {
        for ( MenuItemElement menuItemElement : menuItemElements )
        {
            addMenuItemElement( menuItemElement );
        }
        return this;
    }

    public MenuItemElement getMenuItemElement( int index )
    {
        return (MenuItemElement) getComponent( index );
    }

    public MenuItem addMenuItemElement( MenuItemElement menuItemElement )
    {
        addMenuItemElement( menuItemElement, getComponentCount() );
        return this;
    }

    public MenuItem addMenuItemElement( MenuItemElement menuItemElement, int position )
    {
        if ( menuItemElement == null || menuItemElement.getContent() == null )
        {
            throw new IllegalArgumentException( MENU_ELEMENT_IS_EMPTY );
        }
        else if ( position < 0 )
        {
            throw new IllegalArgumentException( INDEX_HAVE_TO_BE_GREATER_THAN_OR_EQUAL_0 );
        }

        // menu item element caption location
        if ( getMenuItemElementsCaptionLocation() != null )
        {
            menuItemElement.setCaptionLocation( getMenuItemElementsCaptionLocation() );
        }

        // menu item element drag property
        menuItemElement.setMenuItemElementDraggable( isMenuItemElementsDraggable() );

        // menu item element parent's index
        menuItemElement.setParentMenuItemIndex( getParentMenuItemIndex() );

        // add new menu item element
        addComponent( menuItemElement, position );

        // make sure that component is repainted if has parent
        if ( getParent() != null )
        {
            getParent().markAsDirty();
        }

        return this;
    }

    public CaptionLocation getMenuItemElementsCaptionLocation()
    {
        return menuItemElementsCaptionLocation;
    }

    public void setMenuItemElementsCaptionLocation( CaptionLocation captionLocation )
    {
        for ( MenuItemElement menuItemElement : getMenuItemElements() )
        {
            menuItemElement.setCaptionLocation( captionLocation );
        }
        this.menuItemElementsCaptionLocation = captionLocation;
    }

    public boolean isMenuItemElementsDraggable()
    {
        return menuItemElementsDraggable;
    }

    public void setMenuItemElementsDraggable( boolean menuItemElementsDraggable )
    {
        this.menuItemElementsDraggable = menuItemElementsDraggable;
        for ( MenuItemElement menuItemElement : getMenuItemElements() )
        {
            menuItemElement.setMenuItemElementDraggable( menuItemElementsDraggable );
        }
    }

    public DragStartMode getMenuItemElementsDragMode()
    {
        return menuItemElementsDragMode;
    }

    public void setMenuItemElementsDragMode( DragStartMode menuItemElementsDragMode )
    {
        this.menuItemElementsDragMode = menuItemElementsDragMode;
        for ( MenuItemElement menuItemElement : getMenuItemElements() )
        {
            menuItemElement.setMenuItemElementDragMode( menuItemElementsDragMode );
        }
    }

}
