/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.dussan.vaadin.dmenu.client.events.MenuEvents;
import org.dussan.vaadin.dtabs.client.TabsWidget;

import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Node;
import com.google.gwt.dom.client.NodeList;
import com.google.gwt.dom.client.Style.Display;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.dom.client.Style.Visibility;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.EventListener;
import com.google.gwt.user.client.ui.RootPanel;
import com.vaadin.client.ApplicationConnection;
import com.vaadin.client.LayoutManager;

public class VDMenu2
    extends TabsWidget
    implements HasValueChangeHandlers<Object[]>
{

    private static final String CLASS_NAME = "className";

    private static final String EMPTY_VALUE = "";

    private static final String SPACE_VALUE = " ";

    private static final String COMBOBOX_OPTIONLIST_ID = "VAADIN_COMBOBOX_OPTIONLIST";

    private static final String COMBOBOX_OPTIONLIST_CLASS_NAME = "v-filterselect-suggestpopup";

    private static final String MENU_ITEM_ID = "dmenu-";

    private static final String MENU_SPACING = "v-spacing";

    private static final String MENU_TABSHEET = "v-tabsheet";

    private static final String MENU_DECO = MENU_TABSHEET + "-deco";

    private static final String MENU_TAB_CONTAINER = MENU_TABSHEET + "-tabcontainer";

    private static final String MENU_TABS = MENU_TABSHEET + "-tabs";

    private static final String MENU_PANEL = MENU_TABSHEET + "-tabsheetpanel";

    private static final String MENU_CONTENT = MENU_TABSHEET + "-content";

    private static final String MENU_LAYOUT = "v-horizontallayout-v-dmenu";

    private static final String MENU_SCROLLER_PREV = "v-dmenu-scroller-prev";

    private static final String MENU_SCROLLER_NEXT = "v-dmenu-scroller-next";

    private static final String MENU_ITEM_SLOT = "v-slot-v-dmenu-item";

    private static final String MENU_ITEM_SLOT_CONTEXT = MENU_ITEM_SLOT + "-context";

    private static final String MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT = MENU_ITEM_SLOT_CONTEXT
        + "-items-same-height";

    private static final String MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT = MENU_ITEM_SLOT_CONTEXT
        + "-elements-same-height";

    private static final int MENU_LAYOUT_PADDING = 100;

    private static final int MENU_ZINDEX = 10000;

    private ApplicationConnection connection = null;

    private List<Integer> firstVisibleMenuItemElementIndex = null;

    private Element hiddenMenuItemElementsNotBelongHere = null;

    private Element hiddenMenuItemElementsPrev = null;

    private Element hiddenMenuItemElementsNext = null;

    private boolean menuItemsSameHeight = false;

    private boolean menuItemsSameHeightDefined = false;

    private boolean menuItemElementsSameHeight = false;

    private boolean menuItemAutoChange = false;

    private boolean floatingMenu = false;

    private boolean floatingMenuVisible = false;

    private boolean floatingMenuItemAutoHide = false;

    private boolean floatingMenuItemAutoChange = false;

    private int activeMenuItem = 0;

    private int menuItemElementsCount = 0;

    private int menuPanelMaxHeight = 0;

    private int menuItemMaxHeight = 0;

    private int menuLayoutElementPaddingRight = MENU_LAYOUT_PADDING;

    private int menuLayoutElementPaddingLeft = MENU_LAYOUT_PADDING;

    private String activeMenuTabId = null;

    private final Element scrollerPrev;

    private final Element scrollerNext;

    public VDMenu2()
    {
        firstVisibleMenuItemElementIndex = new ArrayList<Integer>();
        hiddenMenuItemElementsNotBelongHere = DOM.createDiv();
        hiddenMenuItemElementsPrev = DOM.createDiv();
        hiddenMenuItemElementsNext = DOM.createDiv();

        // scroller for previous menu item elements
        scrollerPrev = DOM.createButton();
        scrollerPrev.setTabIndex( -1 );
        scrollerPrev.setPropertyString( CLASS_NAME, MENU_SCROLLER_PREV );
        DOM.sinkEvents( scrollerPrev, Event.ONCLICK );

        Event.setEventListener( scrollerPrev, new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                if ( Event.ONCLICK == event.getTypeInt() )
                {
                    int firstVisibleMenuItemElementCounter = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() );
                    Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );

                    getElement().appendChild( hiddenMenuItemElementsPrev );

                    // append spacer
                    if ( hiddenMenuItemElementsPrev.getFirstChildElement()
                                                   .getClassName()
                                                   .contains( MENU_SPACING ) )
                    {
                        firstVisibleMenuItemElementCounter -= 1;
                        menuLayoutElement.insertFirst( hiddenMenuItemElementsPrev.getFirstChildElement() );
                    }

                    // append menu item element
                    firstVisibleMenuItemElementCounter -= 1;
                    menuLayoutElement.insertFirst( hiddenMenuItemElementsPrev.getFirstChildElement() );

                    // save index of first visible menu item element
                    firstVisibleMenuItemElementIndex.set( getActiveMenuItem(), firstVisibleMenuItemElementCounter );

                    hiddenMenuItemElementsPrev.removeFromParent();

                    // check menu item elements visibility
                    checkMenuItemElementsVisibility();
                }
            }
        } );

        // scroller for next menu item elements
        scrollerNext = DOM.createButton();
        scrollerNext.setTabIndex( -1 );
        scrollerNext.setPropertyString( CLASS_NAME, MENU_SCROLLER_NEXT );
        DOM.sinkEvents( scrollerNext, Event.ONCLICK );

        Event.setEventListener( scrollerNext, new EventListener()
        {
            @Override
            public void onBrowserEvent( Event event )
            {
                if ( Event.ONCLICK == event.getTypeInt() )
                {
                    int firstVisibleMenuItemElementCounter = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() );
                    Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );

                    getElement().appendChild( hiddenMenuItemElementsPrev );

                    // append menu item element
                    firstVisibleMenuItemElementCounter += 1;
                    hiddenMenuItemElementsPrev.insertFirst( menuLayoutElement.getFirstChild() );

                    // append spacer
                    if ( menuLayoutElement.getFirstChildElement() != null && menuLayoutElement.getFirstChildElement()
                                                                                              .getClassName()
                                                                                              .contains( MENU_SPACING ) )
                    {
                        firstVisibleMenuItemElementCounter += 1;
                        hiddenMenuItemElementsPrev.insertFirst( menuLayoutElement.getFirstChild() );
                    }

                    // save index of first visible menu item element
                    firstVisibleMenuItemElementIndex.set( getActiveMenuItem(), firstVisibleMenuItemElementCounter );

                    hiddenMenuItemElementsPrev.removeFromParent();

                    // check menu item elements visibility
                    checkMenuItemElementsVisibility();
                }
            }
        } );
    }

    private static List<Element> getChildElementsByClassName( Element element, String className )
    {
        ArrayList<Element> findedElements = new ArrayList<Element>();
        findChildElementsByClassName( findedElements, element, className );
        return findedElements;
    }

    private static void findChildElementsByClassName( ArrayList<Element> findedChildElements, Element element,
                                                      String className )
    {
        if ( element != null )
        {
            String elementClasses = element.getPropertyString( CLASS_NAME );
            if ( elementClasses != null )
            {
                List<String> classes = new ArrayList<String>( Arrays.asList( elementClasses.split( SPACE_VALUE ) ) );
                if ( classes.contains( className ) )
                {
                    findedChildElements.add( element );
                }
            }

            for ( int i = 0; i < DOM.getChildCount( element ); i++ )
            {
                Element child = DOM.getChild( element, i );
                findChildElementsByClassName( findedChildElements, child, className );
            }
        }
    }

    private boolean isFloatingMenuVisible()
    {
        return floatingMenuVisible;
    }

    private void setFloatingMenuVisible( Element menuPanelElement, Element menuContentElement,
                                         boolean floatingMenuVisible )
    {
        this.floatingMenuVisible = floatingMenuVisible;
        if ( floatingMenuVisible )
        {
            menuPanelElement.getStyle()
                            .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            menuPanelElement.getStyle()
                            .clearVisibility();

            if ( isTabBarBottom() )
            {
                menuContentElement.getStyle()
                                  .clearHeight();
                menuContentElement.getStyle()
                                  .setTop( -getMenuPanelMaxHeight(), Unit.PX );
            }

            // show option list of combo box if exits
            if ( RootPanel.get( COMBOBOX_OPTIONLIST_ID ) != null )
            {
                RootPanel.get( COMBOBOX_OPTIONLIST_ID )
                         .getElement()
                         .getStyle()
                         .clearDisplay();
            }
        }
        else
        {
            menuPanelElement.getStyle()
                            .clearHeight();
            menuPanelElement.getStyle()
                            .setVisibility( Visibility.HIDDEN );

            if ( isTabBarBottom() )
            {
                menuContentElement.getStyle()
                                  .setHeight( 1, Unit.PX );
                menuContentElement.getStyle()
                                  .setTop( -1, Unit.PX );
            }
            else
            {
                menuContentElement.getStyle()
                                  .clearHeight();
                menuContentElement.getStyle()
                                  .clearTop();
            }

            // hide option list of combo box if exits
            if ( RootPanel.get( COMBOBOX_OPTIONLIST_ID ) != null )
            {
                RootPanel.get( COMBOBOX_OPTIONLIST_ID )
                         .getElement()
                         .getStyle()
                         .setDisplay( Display.NONE );
            }
        }

    }

    private String getActiveMenuTabId()
    {
        return activeMenuTabId;
    }

    private void setActiveMenuTabId( String activeMenuTabId )
    {
        this.activeMenuTabId = activeMenuTabId;
    }

    private int getMenuItemMaxHeight()
    {
        return menuItemMaxHeight;
    }

    private void setMenuItemMaxHeight( int menuItemMaxHeight )
    {
        this.menuItemMaxHeight = menuItemMaxHeight;
    }

    private int getMenuPanelMaxHeight()
    {
        return menuPanelMaxHeight;
    }

    private void setMenuPanelMaxHeight( int menuPanelMaxHeight )
    {
        this.menuPanelMaxHeight = menuPanelMaxHeight;
    }

    private void processMenuItemAutoChange( String activeMenuTabId )
    {
        if ( isMenuItemAutoChange() && !getActiveMenuTabId().equals( activeMenuTabId ) )
        {
            setActiveMenuTabId( activeMenuTabId );
            MenuEvents.fireMenuActiveTabChange( this, activeMenuTabId );
        }
    }

    private void processFloatingMenuItemAutoShow( String menuItemId )
    {
        if ( isFloatingMenu() && isFloatingMenuItemAutoShow() && !getActiveMenuTabId().equals( menuItemId ) )
        {
            setActiveMenuTabId( menuItemId );
            MenuEvents.fireMenuItemShow( this, menuItemId );
        }
    }

    public ApplicationConnection getConnection()
    {
        return connection;
    }

    public void setConnection( ApplicationConnection connection )
    {
        this.connection = connection;
    }

    public int getActiveMenuItem()
    {
        return activeMenuItem;
    }

    public void setSelectedMenuItem( int activeMenuItem )
    {
        this.activeMenuItem = activeMenuItem;
    }

    public boolean isMenuItemsSameHeight()
    {
        return menuItemsSameHeight;
    }

    public void setMenuItemsSameHeight( boolean menuItemsSameHeight )
    {
        this.menuItemsSameHeight = menuItemsSameHeight;
    }

    public boolean isMenuItemsSameHeightDefined()
    {
        return menuItemsSameHeightDefined;
    }

    public void setMenuItemsSameHeightDefined( boolean menuItemsSameHeightDefined )
    {
        this.menuItemsSameHeightDefined = menuItemsSameHeightDefined;
    }

    public void setMenuItemsSameHeight()
    {
        if ( isMenuItemsSameHeight() )
        {
            // show all menu item elements
            List<Element> elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT );
            for ( Element element : elements )
            {
                element.getStyle()
                       .clearDisplay();
            }

            // reset menu item elements height
            elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );
            for ( Element element : elements )
            {
                element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT );
                element.getStyle()
                       .clearHeight();
            }

            // get max height of menu item elements
            if ( !isMenuItemsSameHeightDefined() && isMenuItemElementsSameHeight() )
            {
                elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );
                for ( Element element : elements )
                {
                    setMenuItemMaxHeight( Math.max( getMenuItemMaxHeight(), element.getClientHeight() ) );
                }
            }

            // set same height of menu item elements
            if ( isMenuItemElementsSameHeight() )
            {
                for ( int index = 0; index < getMenuItemElementsCount() && index < elements.size(); index++ )
                {
                    elements.get( index )
                            .addClassName( MENU_ITEM_SLOT_CONTEXT_ITEMS_SAME_HEIGHT );
                    elements.get( index )
                            .getStyle()
                            .setHeight( getMenuItemMaxHeight(), Unit.PX );
                }
            }

            // remove menu item elements which not belong to this tab
            elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT );
            for ( int index = elements.size() - 1; getMenuItemElementsCount() <= index; index-- )
            {
                hiddenMenuItemElementsNotBelongHere.appendChild( elements.get( index ) );
            }

            // get menu items max height
            elements = getChildElementsByClassName( getElement(), MENU_PANEL );
            if ( !isMenuItemsSameHeightDefined() )
            {
                for ( Element element : elements )
                {
                    setMenuPanelMaxHeight( Math.max( getMenuPanelMaxHeight(), element.getClientHeight() ) );
                }
            }

            // set menu items same height
            for ( Element element : elements )
            {
                element.getStyle()
                       .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            }
        }
    }

    public boolean isMenuItemElementsSameHeight()
    {
        return menuItemElementsSameHeight;
    }

    public void setMenuItemElementsSameHeight( boolean menuItemElementsSameHeight )
    {
        this.menuItemElementsSameHeight = menuItemElementsSameHeight;
    }

    public void setMenuItemElementsSameHeight()
    {
        if ( !isMenuItemsSameHeight() )
        {
            List<Element> elements = getChildElementsByClassName( getElement(), MENU_ITEM_SLOT_CONTEXT );

            // get max height of menu item elements
            setMenuItemMaxHeight( 0 );
            for ( Element element : elements )
            {
                setMenuItemMaxHeight( Math.max( getMenuItemMaxHeight(), element.getClientHeight() ) );
            }

            // reset height of menu item elements
            for ( Element element : elements )
            {
                if ( element.hasClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT ) )
                {
                    element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .clearHeight();
                }
            }

            if ( isMenuItemElementsSameHeight() )
            {
                // set same height to all menu item elements
                for ( Element element : elements )
                {
                    element.addClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .setHeight( getMenuItemMaxHeight(), Unit.PX );
                }
            }
            else
            {
                // delete height of menu item elements
                for ( Element element : elements )
                {
                    element.removeClassName( MENU_ITEM_SLOT_CONTEXT_ELEMENTS_SAME_HEIGHT );
                    element.getStyle()
                           .clearHeight();
                }
            }

            // set menu panel height
            setMenuPanelMaxHeight( getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 )
                                                                                           .getClientHeight() );
            getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 )
                                                                   .getStyle()
                                                                   .setHeight( getMenuPanelMaxHeight(), Unit.PX );
            if ( isFloatingMenu() && isTabBarBottom() )
            {
                getChildElementsByClassName( getElement(), MENU_CONTENT ).get( 0 )
                                                                         .getStyle()
                                                                         .setTop( -getMenuPanelMaxHeight(), Unit.PX );
            }
        }
    }

    public boolean isMenuItemAutoChange()
    {
        return menuItemAutoChange;
    }

    public void setMenuItemAutoChange( boolean menuItemAutoChange )
    {
        this.menuItemAutoChange = menuItemAutoChange;
    }

    public void setMenuItemAutoChange()
    {
        if ( isMenuItemAutoChange() )
        {
            Element menuTabsElement = getChildElementsByClassName( getElement(), MENU_TABS ).get( 0 );
            DOM.sinkEvents( menuTabsElement, Event.ONMOUSEOVER );
            Event.setEventListener( menuTabsElement, new EventListener()
            {
                @Override
                public void onBrowserEvent( Event event )
                {
                    if ( Event.ONMOUSEOVER == event.getTypeInt() )
                    {
                        String elementEventClassName = EMPTY_VALUE;
                        Element elementEvent = Element.as( event.getEventTarget() );
                        while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                        {
                            if ( elementEvent.getClassName()
                                             .contains( MENU_ITEM_ID ) )
                            {
                                elementEventClassName = elementEvent.getClassName()
                                                                    .split( MENU_ITEM_ID )[1];
                                elementEventClassName = MENU_ITEM_ID + elementEventClassName.split( SPACE_VALUE )[0];
                                processMenuItemAutoChange( elementEventClassName );
                            }
                            elementEvent = elementEvent.getParentElement();
                        }
                    }
                }
            } );
        }
    }

    public boolean isFloatingMenu()
    {
        return floatingMenu;
    }

    public void setFloatingMenu( boolean floatingMenu )
    {
        this.floatingMenu = floatingMenu;
    }

    public void setFloatingMenu()
    {
        if ( isFloatingMenu() )
        {
            final Element menuPanelElement = getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 );
            final Element menuContentElement = getChildElementsByClassName( getElement(), MENU_CONTENT ).get( 0 );
            Element menuTabContainerElement = getChildElementsByClassName( getElement(), MENU_TAB_CONTAINER ).get( 0 );
            Element menuDecoElement = getChildElementsByClassName( getElement(), MENU_DECO ).get( 0 );

            menuContentElement.getStyle()
                              .setPosition( Position.ABSOLUTE );
            menuContentElement.getStyle()
                              .setZIndex( MENU_ZINDEX );
            menuTabContainerElement.getStyle()
                                   .setZIndex( MENU_ZINDEX );
            menuDecoElement.getStyle()
                           .setZIndex( MENU_ZINDEX );
            menuDecoElement.getStyle()
                           .setMarginBottom( 0, Unit.PX );
            menuDecoElement.getStyle()
                           .setBottom( -getMenuPanelMaxHeight() / 2, Unit.PX );
            setFloatingMenuVisible( menuPanelElement, menuContentElement, isFloatingMenuVisible() );

            Element menuTabsElement = getChildElementsByClassName( getElement(), MENU_TABS ).get( 0 );
            DOM.sinkEvents( menuTabsElement, Event.ONCLICK | Event.ONMOUSEOVER );
            Event.setEventListener( menuTabsElement, new EventListener()
            {
                @Override
                public void onBrowserEvent( Event event )
                {
                    String elementEventClassName = EMPTY_VALUE;
                    Element elementEvent = Element.as( event.getEventTarget() );
                    while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                    {
                        if ( elementEvent.getClassName()
                                         .contains( MENU_ITEM_ID ) )
                        {
                            elementEventClassName = elementEvent.getClassName()
                                                                .split( MENU_ITEM_ID )[1];
                            elementEventClassName = MENU_ITEM_ID + elementEventClassName.split( SPACE_VALUE )[0];
                        }
                        elementEvent = elementEvent.getParentElement();
                    }

                    if ( !elementEventClassName.isEmpty() )
                    {
                        switch ( event.getTypeInt() )
                        {
                            case Event.ONCLICK:
                                if ( !elementEventClassName.equals( getActiveMenuTabId() ) )
                                {
                                    setFloatingMenuVisible( menuPanelElement, menuContentElement, true );
                                    processFloatingMenuItemAutoShow( elementEventClassName );
                                }
                                else
                                {
                                    setFloatingMenuVisible( menuPanelElement, menuContentElement,
                                                            !isFloatingMenuVisible() );
                                }
                                setActiveMenuTabId( elementEventClassName );
                                break;
                            case Event.ONMOUSEOVER:
                                if ( isFloatingMenuItemAutoShow() )
                                {
                                    if ( !elementEventClassName.equals( getActiveMenuTabId() ) )
                                    {
                                        setFloatingMenuVisible( menuPanelElement, menuContentElement, true );
                                        processFloatingMenuItemAutoShow( elementEventClassName );
                                    }
                                    else
                                    {
                                        if ( !isFloatingMenuVisible() )
                                        {
                                            setActiveMenuTabId( EMPTY_VALUE );
                                        }
                                        else
                                        {
                                            setActiveMenuTabId( getActiveMenuTabId() );
                                        }
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                }
            } );

            // browser's events
            DOM.sinkEvents( RootPanel.getBodyElement(), Event.ONCLICK | Event.ONKEYDOWN | Event.ONMOUSEOUT );
            Event.setEventListener( RootPanel.getBodyElement(), new EventListener()
            {
                @Override
                public void onBrowserEvent( Event event )
                {
                    if ( isFloatingMenu() && isFloatingMenuVisible() )
                    {
                        if ( ( event.getTypeInt() == Event.ONKEYDOWN && event.getKeyCode() == KeyCodes.KEY_ESCAPE )
                            || ( event.getTypeInt() == Event.ONMOUSEOUT && event.getRelatedEventTarget() == null ) )
                        {
                            setFloatingMenuVisible( menuPanelElement, menuContentElement, false );
                            setActiveMenuTabId( EMPTY_VALUE );
                        }
                        else if ( event.getTypeInt() == Event.ONCLICK || isFloatingMenuItemAutoHide() )
                        {
                            String elementEventClassName = EMPTY_VALUE;
                            Element elementEvent = null;
                            if ( event.getTypeInt() == Event.ONCLICK )
                            {
                                elementEvent = Element.as( event.getEventTarget() );
                            }
                            else
                            {
                                elementEvent = Element.as( event.getRelatedEventTarget() );
                            }

                            // get parent for menu item element
                            while ( elementEvent.hasParentElement() && elementEventClassName.isEmpty() )
                            {
                                if ( elementEvent.hasClassName( MENU_TABSHEET )
                                    || elementEvent.hasClassName( COMBOBOX_OPTIONLIST_CLASS_NAME ) )
                                {
                                    elementEventClassName = elementEvent.getClassName();
                                }
                                elementEvent = elementEvent.getParentElement();
                            }

                            // clicked out of menu area - hide float menu
                            if ( elementEventClassName.isEmpty() )
                            {
                                setFloatingMenuVisible( menuPanelElement, menuContentElement, false );
                                setActiveMenuTabId( EMPTY_VALUE );
                            }
                        }
                    }
                }
            } );

        }
    }

    public boolean isFloatingMenuItemAutoHide()
    {
        return floatingMenuItemAutoHide;
    }

    public void setFloatingMenuItemAutoHide( boolean floatingMenuItemAutoHide )
    {
        this.floatingMenuItemAutoHide = floatingMenuItemAutoHide;
    }

    public boolean isFloatingMenuItemAutoShow()
    {
        return floatingMenuItemAutoChange;
    }

    public void setFloatingMenuItemAutoShow( boolean floatingMenuItemAutoShow )
    {
        this.floatingMenuItemAutoChange = floatingMenuItemAutoShow;
    }

    public int getMenuItemElementsCount()
    {
        return menuItemElementsCount;
    }

    public void setMenuItemElementsCount( int menuItemElementsCount )
    {
        this.menuItemElementsCount = menuItemElementsCount;
    }

    public void resetHiddenMenuItemElements()
    {
        // prepare index for visible item elements
        if ( firstVisibleMenuItemElementIndex.isEmpty() )
        {
            for ( int index = 0; index < getTabCount(); index++ )
            {
                firstVisibleMenuItemElementIndex.add( 0 );
            }
        }

        // remove all child elements from hiddenMenuItemElementsNotBelong
        if ( hiddenMenuItemElementsNotBelongHere == null )
        {
            hiddenMenuItemElementsNotBelongHere = DOM.createDiv();
        }
        else
        {
            hiddenMenuItemElementsNotBelongHere.removeAllChildren();
        }

        // remove all child elements from hiddenMenuItemElementsPrev
        if ( hiddenMenuItemElementsPrev == null )
        {
            hiddenMenuItemElementsPrev = DOM.createDiv();
        }
        else
        {
            hiddenMenuItemElementsPrev.removeAllChildren();
        }

        // remove all child from elements hiddenMenuItemElementsNext
        if ( hiddenMenuItemElementsNext == null )
        {
            hiddenMenuItemElementsNext = DOM.createDiv();
        }
        else
        {
            hiddenMenuItemElementsNext.removeAllChildren();
        }
    }

    public void checkMenuItemElementsVisibility()
    {
        Element menuLayoutElement = getChildElementsByClassName( getElement(), MENU_LAYOUT ).get( 0 );
        Element menuPanelElement = getChildElementsByClassName( getElement(), MENU_PANEL ).get( 0 );
        NodeList<Node> menuLayoutNodes = menuLayoutElement.getChildNodes();

        menuLayoutElementPaddingLeft =
            Math.min( menuLayoutElementPaddingLeft, LayoutManager.get( getConnection() )
                                                                 .getPaddingLeft( menuLayoutElement ) );
        menuLayoutElementPaddingRight =
            Math.min( menuLayoutElementPaddingRight, LayoutManager.get( getConnection() )
                                                                  .getPaddingRight( menuLayoutElement ) );

        // remove unused spacing elements after last menu item element
        if ( 1 < menuLayoutNodes.getLength() && Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) )
                                                       .getClassName()
                                                       .contains( MENU_SPACING )
            && !Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 2 ) )
                       .getClassName()
                       .contains( MENU_SPACING ) )
        {
            while ( Element.as( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) )
                           .getClassName()
                           .contains( MENU_SPACING ) )
            {
                hiddenMenuItemElementsNotBelongHere.appendChild( menuLayoutNodes.getItem( menuLayoutNodes.getLength() - 1 ) );
            }
        }

        getElement().appendChild( hiddenMenuItemElementsPrev );
        getElement().appendChild( hiddenMenuItemElementsNext );

        // remove/append first visible menu items
        if ( hiddenMenuItemElementsPrev.getChildCount() < firstVisibleMenuItemElementIndex.get( getActiveMenuItem() ) )
        {
            for ( int index = hiddenMenuItemElementsPrev.getChildCount(); index < firstVisibleMenuItemElementIndex.get( getActiveMenuItem() ); index++ )
            {
                hiddenMenuItemElementsPrev.insertFirst( menuLayoutElement.getChildNodes()
                                                                         .getItem( 0 ) );
            }
        }
        else if ( firstVisibleMenuItemElementIndex.get( getActiveMenuItem() ) < hiddenMenuItemElementsPrev.getChildCount() )
        {
            for ( int index = firstVisibleMenuItemElementIndex.get( getActiveMenuItem() ); index < hiddenMenuItemElementsPrev.getChildCount(); index++ )
            {
                menuLayoutElement.insertFirst( hiddenMenuItemElementsPrev.getFirstChildElement() );
            }
        }

        // check if menu layout is empty
        if ( menuLayoutElement.getChildCount() == 0 )
        {
            hiddenMenuItemElementsPrev.insertFirst( hiddenMenuItemElementsNext.getFirstChild() );
            firstVisibleMenuItemElementIndex.set( getActiveMenuItem(),
                                                  firstVisibleMenuItemElementIndex.get( getActiveMenuItem() ) + 1 );
        }

        // append scrollers
        menuPanelElement.insertFirst( scrollerPrev );
        menuPanelElement.appendChild( scrollerNext );

        // append menu item elements from hiddenMenuItemElementsNext back to menuLayoutElement
        while ( hiddenMenuItemElementsNext.hasChildNodes() )
        {
            menuLayoutElement.appendChild( hiddenMenuItemElementsNext.getFirstChildElement() );
        }

        // menu item width
        int menuItemWidth = menuPanelElement.getClientWidth();

        // left scroller is visible
        if ( 0 < hiddenMenuItemElementsPrev.getChildCount() )
        {
            menuItemWidth -= scrollerPrev.getOffsetWidth();
        }

        // menu item elements width
        int menuItemElementsWidth = menuLayoutElementPaddingLeft + menuLayoutElementPaddingRight;

        // index of hiddenMenuItemElementsNext
        int hiddenMenuItemElementsNextIndex = menuLayoutNodes.getLength();

        // first go from left to right to get the maximal width of visible menu item elements
        for ( int index = 0; index < menuLayoutNodes.getLength(); index++ )
        {
            Element element = Element.as( menuLayoutNodes.getItem( index ) );
            menuItemElementsWidth += element.getOffsetWidth();

            if ( menuItemWidth < menuItemElementsWidth )
            {
                hiddenMenuItemElementsNextIndex = Math.min( hiddenMenuItemElementsNextIndex, index );

                // remove last spacing element
                if ( element.getClassName()
                            .contains( MENU_SPACING ) )
                {
                    menuItemElementsWidth -= element.getOffsetWidth();
                    element = Element.as( menuLayoutNodes.getItem( --hiddenMenuItemElementsNextIndex ) );
                }

                // right scroller is hidden/visible
                boolean scrollerWidthIsIncluded = hiddenMenuItemElementsNextIndex != ( menuLayoutNodes.getLength() - 1 );

                // right scroller is visible
                if ( scrollerWidthIsIncluded )
                {
                    menuItemWidth -= scrollerNext.getOffsetWidth();
                }

                // then go from right to left to get the minimal width of visible menu item elements
                while ( 0 < hiddenMenuItemElementsNextIndex && menuItemWidth < menuItemElementsWidth )
                {
                    menuItemElementsWidth -= element.getOffsetWidth();
                    element = Element.as( menuLayoutNodes.getItem( --hiddenMenuItemElementsNextIndex ) );

                    if ( !scrollerWidthIsIncluded )
                    {
                        scrollerWidthIsIncluded = true;
                        menuItemWidth -= scrollerNext.getOffsetWidth();
                    }

                    // remove last spacing element
                    if ( element.getClassName()
                                .contains( MENU_SPACING ) )
                    {
                        menuItemElementsWidth -= element.getOffsetWidth();
                        hiddenMenuItemElementsNextIndex -= 1;
                    }
                }

                break;
            }
        }

        // hide invisible menu item elements
        for ( int index = menuLayoutNodes.getLength() - 1; hiddenMenuItemElementsNextIndex < index; index-- )
        {
            hiddenMenuItemElementsNext.insertFirst( menuLayoutNodes.getItem( index ) );
        }

        // set menu items elements height
        if ( isFloatingMenu() && isFloatingMenuVisible() && !isMenuItemsSameHeight() )
        {
            setMenuItemElementsSameHeight();
        }

        // hide/show scrollerPrev
        if ( hiddenMenuItemElementsPrev.getChildCount() == 0 )
        {
            menuLayoutElement.getStyle()
                             .clearPaddingLeft();
            scrollerPrev.getStyle()
                        .setVisibility( Visibility.HIDDEN );
        }
        else
        {
            menuLayoutElement.getStyle()
                             .setPaddingLeft( scrollerPrev.getOffsetWidth() + menuLayoutElementPaddingLeft, Unit.PX );
            scrollerPrev.getStyle()
                        .clearVisibility();
        }

        // hide/show scrollerNext
        if ( hiddenMenuItemElementsNext.getChildCount() == 0 )
        {
            scrollerNext.getStyle()
                        .setVisibility( Visibility.HIDDEN );
        }
        else
        {
            scrollerNext.getStyle()
                        .clearVisibility();
        }

        hiddenMenuItemElementsPrev.removeFromParent();
        hiddenMenuItemElementsNext.removeFromParent();
    }

    @Override
    public HandlerRegistration addValueChangeHandler( ValueChangeHandler<Object[]> handler )
    {
        return this.addHandler( handler, ValueChangeEvent.getType() );
    }

}
