/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dmenu.menuitem;

import java.util.Iterator;

import com.vaadin.ui.Button;
import com.vaadin.ui.CheckBox;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.Component;
import com.vaadin.ui.DateField;
import com.vaadin.ui.DragAndDropWrapper;
import com.vaadin.ui.HasComponents;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Link;
import com.vaadin.ui.MenuBar;
import com.vaadin.ui.OptionGroup;
import com.vaadin.ui.Panel;
import com.vaadin.ui.Table;
import com.vaadin.ui.TextArea;
import com.vaadin.ui.TextField;
import com.vaadin.ui.TreeTable;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.DragAndDropWrapper.DragStartMode;

public class MenuItemElement
    extends Panel
{

    private static final long serialVersionUID = -4754397816991773376L;

    private static final String CONTEXT_IS_EMPTY = "Context is empty.";

    private static final String COMPONENT_SMALL = "small";

    private static final String COMPONENT_TINY = "tiny";

    private static final String MENU_ITEM = "v-dmenu-item";

    private static final String MENU_ITEM_NO_CAPTION = CaptionLocation.NO_CAPTION.getLocation();

    private static final String MENU_ITEM_CAPTION = MENU_ITEM + "-caption-";

    private static final String MENU_ITEM_CONTEXT = MENU_ITEM + "-context";

    private boolean menuItemElementDraggable = false;

    private DragStartMode menuItemElementDragMode = DragStartMode.HTML5;

    private int parentMenuItemIndex = 0;

    private Panel caption = null;

    private CaptionLocation captionLocation = null;

    private DragAndDropWrapper contextDragWrapper = null;

    private VerticalLayout context = null;

    private Component menuItemElementContext = null;

    private MenuItemElement()
    {
        prepareContent();
    }

    public MenuItemElement( String caption )
    {
        this();
        setCaption( caption );
    }

    public MenuItemElement( String caption, Component content )
    {
        this();
        setCaption( caption );
        setContext( content );
    }

    private void prepareContent()
    {
        addStyleName( MENU_ITEM );
        setSizeUndefined();
        context = new VerticalLayout();
        context.setSizeUndefined();
        context.setMargin( false );

        // set menu item element drag enabled/disabled
        contextDragWrapper = new DragAndDropWrapper( context );
        if ( isMenuItemElementDraggable() )
        {
            contextDragWrapper.setDragStartMode( getMenuItemElementDragMode() );
        }
        else
        {
            contextDragWrapper.setDragStartMode( DragStartMode.NONE );
        }

        setContent( contextDragWrapper );
        captionLocation = CaptionLocation.BOTTOM;
    }

    private void addSmallOrTinyStyle( Component component )
    {
        if ( component instanceof CheckBox || component instanceof Link || component instanceof MenuBar )
        {
            component.addStyleName( COMPONENT_SMALL );
        }
        else if ( component instanceof OptionGroup || component instanceof Table || component instanceof TreeTable )
        {
            component.addStyleName( COMPONENT_SMALL );
        }
        else if ( component instanceof Button || component instanceof ComboBox || component instanceof DateField )
        {
            component.addStyleName( COMPONENT_TINY );
        }
        else if ( component instanceof Label || component instanceof TextArea || component instanceof TextField )
        {
            component.addStyleName( COMPONENT_TINY );
        }
        else if ( component instanceof HasComponents )
        {
            for ( Iterator<Component> iterator = ( (HasComponents) component ).iterator(); iterator.hasNext(); )
            {
                addSmallOrTinyStyle( iterator.next() );
            }
        }
    }

    public boolean isMenuItemElementDraggable()
    {
        return menuItemElementDraggable;
    }

    public void setMenuItemElementDraggable( boolean menuItemElementDraggable )
    {
        this.menuItemElementDraggable = menuItemElementDraggable;
        if ( contextDragWrapper != null )
        {
            if ( isMenuItemElementDraggable() )
            {
                contextDragWrapper.setDragStartMode( getMenuItemElementDragMode() );
            }
            else
            {
                contextDragWrapper.setDragStartMode( DragStartMode.NONE );
            }
            markAsDirty();
        }
    }

    public DragStartMode getMenuItemElementDragMode()
    {
        return menuItemElementDragMode;
    }

    public void setMenuItemElementDragMode( DragStartMode menuItemElementDragMode )
    {
        this.menuItemElementDragMode = menuItemElementDragMode;
        if ( contextDragWrapper != null && isMenuItemElementDraggable() )
        {
            contextDragWrapper.setDragStartMode( menuItemElementDragMode );
        }
    }

    public int getParentMenuItemIndex()
    {
        return parentMenuItemIndex;
    }

    public void setParentMenuItemIndex( int parentMenuItemIndex )
    {
        this.parentMenuItemIndex = parentMenuItemIndex;
    }

    public String getCaption()
    {
        return caption.getCaption();
    }

    public void setCaption( String caption )
    {
        if ( caption != null && !caption.isEmpty() )
        {
            this.caption = new Panel( caption );
            this.caption.addStyleName( MENU_ITEM_CAPTION + getCaptionLocation().getLocation() );
            this.caption.setSizeFull();
            this.caption.setHeightUndefined();
        }
        else
        {
            setCaptionLocation( CaptionLocation.NO_CAPTION );
        }
    }

    public Component getContext()
    {
        return getContent();
    }

    public void setContext( Component context )
    {
        if ( context == null )
        {
            throw new IllegalArgumentException( CONTEXT_IS_EMPTY );
        }
        this.context.removeAllComponents();
        context.removeStyleName( MENU_ITEM_NO_CAPTION );

        if ( caption != null && captionLocation == CaptionLocation.TOP )
        {
            this.context.addComponent( caption );
        }

        // all base components have to have small or tiny style
        addSmallOrTinyStyle( context );

        if ( context instanceof HorizontalLayout )
        {
            ( (HorizontalLayout) context ).setMargin( true );
            ( (HorizontalLayout) context ).setSpacing( true );
            ( (HorizontalLayout) context ).setWidthUndefined();
            context.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                context.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( context );
        }
        else if ( context instanceof VerticalLayout )
        {
            ( (VerticalLayout) context ).setMargin( true );
            ( (VerticalLayout) context ).setSpacing( true );
            ( (VerticalLayout) context ).setWidthUndefined();
            context.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                context.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( context );
        }
        else
        {
            VerticalLayout elementContext = new VerticalLayout();
            elementContext.setMargin( true );
            elementContext.setSpacing( true );
            elementContext.setWidthUndefined();
            elementContext.addComponent( context );
            elementContext.addStyleName( MENU_ITEM_CONTEXT );
            if ( getCaptionLocation() == CaptionLocation.NO_CAPTION )
            {
                elementContext.addStyleName( MENU_ITEM_NO_CAPTION );
            }
            this.context.addComponent( elementContext );
        }
        menuItemElementContext = context;

        if ( caption != null && captionLocation == CaptionLocation.BOTTOM )
        {
            this.context.addComponent( caption );
        }
    }

    public CaptionLocation getCaptionLocation()
    {
        return captionLocation;
    }

    public void setCaptionLocation( CaptionLocation captionLocation )
    {
        if ( caption == null || captionLocation == null || captionLocation == CaptionLocation.NO_CAPTION )
        {
            caption = null;
            this.captionLocation = CaptionLocation.NO_CAPTION;
        }
        else if ( caption != null )
        {
            this.captionLocation = captionLocation;
            caption.removeStyleName( MENU_ITEM_CAPTION + CaptionLocation.BOTTOM.getLocation() );
            caption.removeStyleName( MENU_ITEM_CAPTION + CaptionLocation.TOP.getLocation() );
            caption.addStyleName( MENU_ITEM_CAPTION + captionLocation.getLocation() );
        }

        if ( menuItemElementContext != null )
        {
            setContext( menuItemElementContext );
        }
    }

}
