/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery;

import java.io.Serializable;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dmenu.events.MenuItemElementClickEvent;
import org.dussan.vaadin.dmenu.events.MenuItemElementClickHandler;
import org.dussan.vaadin.dmenu.menuitem.MenuItem;
import org.dussan.vaadin.dmenu.menuitem.MenuItemElement;
import org.dussan.vaadin.dquery.base.sql.builder.SqlColumn;
import org.dussan.vaadin.dquery.base.sql.builder.SqlJoin;
import org.dussan.vaadin.dquery.base.sql.builder.SqlTable;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.base.ui.DroppedTable;
import org.dussan.vaadin.dquery.base.ui.JoinedTables;
import org.dussan.vaadin.dquery.client.rpc.QueryServerRpc;
import org.dussan.vaadin.dquery.client.state.QueryState;
import org.dussan.vaadin.dquery.enums.JsonQueryElement;
import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.helper.ManifestHelper;
import org.dussan.vaadin.dquery.json.JsonBuilder;
import org.dussan.vaadin.dquery.json.JsonQuery;
import org.dussan.vaadin.dquery.sql.builder.SqlBuilder;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.ui.DataSourcesMenu;
import org.dussan.vaadin.dquery.ui.DroppedTableContainer;
import org.dussan.vaadin.dquery.ui.SqlMenu;
import org.dussan.vaadin.dquery.ui.SqlTabs;
import org.dussan.vaadin.dquery.ui.windows.JoinedTablesWindow;
import org.dussan.vaadin.dquery.ui.windows.MessageWindow;
import org.dussan.vaadin.dquery.ui.windows.SqlFunctionsWindow;
import org.dussan.vaadin.dquery.ui.windows.SqlParametersWindow;
import org.dussan.vaadin.dquery.ui.windows.TableDataWindow;
import org.dussan.vaadin.dquery.ui.windows.TestQueryWindow;
import org.dussan.vaadin.dquery.utils.ErrorUtil;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.event.DataBoundTransferable;
import com.vaadin.event.FieldEvents.BlurEvent;
import com.vaadin.event.FieldEvents.BlurListener;
import com.vaadin.event.FieldEvents.FocusEvent;
import com.vaadin.event.FieldEvents.FocusListener;
import com.vaadin.event.FieldEvents.TextChangeEvent;
import com.vaadin.event.FieldEvents.TextChangeListener;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.event.dd.DragAndDropEvent;
import com.vaadin.event.dd.DropHandler;
import com.vaadin.event.dd.DropTarget;
import com.vaadin.event.dd.acceptcriteria.AcceptAll;
import com.vaadin.event.dd.acceptcriteria.AcceptCriterion;
import com.vaadin.shared.MouseEventDetails;
import com.vaadin.ui.AbstractSelect.AbstractSelectTargetDetails;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.Component;
import com.vaadin.ui.CustomLayout;
import com.vaadin.ui.DragAndDropWrapper;
import com.vaadin.ui.DragAndDropWrapper.WrapperTargetDetails;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.TableTransferable;
import com.vaadin.ui.TextField;
import com.vaadin.ui.UI;

public abstract class DQuery
    implements Serializable
{

    private static final long serialVersionUID = 8421686217816942853L;

    private static final String VERSION = "Implementation-Version";

    private static final String GIT_VERSION = "Git-Version";

    private static final String IDENTIFIER_IS_EMPTY = "Identifier is empty.";

    private static final String DATASOURCES_ALREADY_HAVE_THIS_IDENTIFIER =
        "Data sources already have this identifier: {0}.";

    private static final String DATASOURCES_NOT_HAVE_THIS_IDENTIFIER = "Data sources not have this identifier: {0}.";

    private boolean showDroppedTableData = true;

    private boolean showQueryDefinition = true;

    private int addedTableLeftPosition = 0;

    private int addedTableTopPosition = 0;

    private String currentDataSourceId = null;

    private JdbcUtil currentJdbc = null;

    private transient Map<String, JdbcUtil> dataSources = null;

    private transient Context context = null;

    private DataSourcesMenu dataSourcesMenu = null;

    private DroppedTableContainer droppedTableContainer = null;

    private SqlTabs sqlTabs = null;

    private SqlMenu sqlMenu = null;

    private JsonBuilder jsonBuilder = null;

    private SqlBuilder sqlBuilder = null;

    /**
     * Creates a new instance.
     */
    public DQuery()
    {
        // set added table start position
        setAddedTablePosition( 50, 50 );

        // context
        context = new Context();

        // json builder
        jsonBuilder = new JsonBuilder();

        // data sources
        dataSources = new LinkedHashMap<>();
        droppedTableContainer = new DroppedTableContainer();

        // data sources menu
        dataSourcesMenu = new DataSourcesMenu();
        dataSourcesMenu.setCenteredTabs( true );
        dataSourcesMenu.setFloatingMenuLockEnabled( true );

        // sql tabs
        sqlTabs = new SqlTabs();
        sqlTabs.setCenteredTabs( true );

        // sql menu
        sqlMenu = new SqlMenu();
    }

    /**
     * Save json query.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     * @param queryString
     *            json query as string
     */
    public abstract void saveQuery( String queryGroup, String queryName, String queryString );

    /**
     * Set header row visible.
     * 
     * @param rowId
     *            row id
     * @param visible
     *            if true header row is visible, otherwise is hidden
     */
    private void setHeaderRowVisible( String rowId, boolean visible )
    {
        // button activity
        switch ( rowId )
        {
            case BaseTable.DESCRIPTION_COMMON:
                sqlMenu.setButtonActive( SqlMenu.BUTTON_COMMON, visible );
                break;
            case BaseTable.DESCRIPTION_SORTING:
                sqlMenu.setButtonActive( SqlMenu.BUTTON_SORTING, visible );
                break;
            case BaseTable.DESCRIPTION_TOTAL:
                sqlMenu.setButtonActive( SqlMenu.BUTTON_TOTAL, visible );
                break;
            default:
                break;
        }

        // header row visibility
        sqlTabs.setHeaderRowVisible( rowId, visible );
        if ( SharedUtil.isNotNull( context ) )
        {
            context.getState( true )
                   .setInvisibleHeaderRows( sqlTabs.getInvisibleHeaderRows() );
            context.markAsDirty();
        }
    }

    /**
     * Refresh sql editor context.
     */
    private void refreshSqlEditorContext()
    {
        // sql builder
        sqlBuilder = new SqlBuilder();

        // table columns
        for ( Object column : sqlTabs.getHeaderFieldsTable()
                                     .getVisibleColumns() )
        {
            String columnId = column.toString();
            if ( !BaseTable.COLUMN_LAST_FINAL.equals( column ) && !sqlTabs.isFieldColumnEmpty( columnId ) )
            {
                SqlColumn sqlColumn = new SqlColumn();
                sqlColumn.setColumn( sqlTabs.getHeaderFieldsTable()
                                            .getCellValue( columnId, BaseTable.DESCRIPTION_FIELD ) )
                         .as( sqlTabs.getHeaderFieldsTable()
                                     .getCellValue( columnId, BaseTable.DESCRIPTION_NAME ) )
                         .from( sqlTabs.getHeaderFieldsTable()
                                       .getCellValue( columnId, BaseTable.DESCRIPTION_TABLE ) );

                // common function
                if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_COMMON ) )
                {
                    String function = sqlTabs.getHeaderFieldsTable()
                                             .getCellValue( columnId, BaseTable.DESCRIPTION_COMMON );
                    if ( SharedUtil.isNotNullAndNotEmpty( function ) )
                    {
                        sqlColumn.setCommonFunction( function, sqlTabs.getCommonFunctions()
                                                                      .getType( function ) );
                    }
                }

                // total function
                if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL ) )
                {
                    String function = sqlTabs.getHeaderFieldsTable()
                                             .getCellValue( columnId, BaseTable.DESCRIPTION_TOTAL );
                    if ( SharedUtil.isNotNullAndNotEmpty( function ) )
                    {
                        sqlColumn.setTotalFunction( function, sqlTabs.getTotalFunctions()
                                                                     .getType( function ) );
                    }
                }

                // sorting
                if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_SORTING ) )
                {
                    String sorting = sqlTabs.getHeaderFieldsTable()
                                            .getCellValue( columnId, BaseTable.DESCRIPTION_SORTING );
                    if ( SharedUtil.isNotNullAndNotEmpty( sorting ) )
                    {
                        sqlColumn.setOrderBy( sorting );
                    }
                }

                // where conditions
                sqlBuilder.setTableGrouping( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL ) );
                if ( SharedUtil.isNotNullAndNotEmpty( sqlColumn.toShortString() ) && !sqlColumn.toShortString()
                                                                                               .contains( BaseTable.FIELDS_ALL ) )
                {
                    for ( Object rowId : sqlTabs.getCriteriaFieldsTable()
                                                .getItemIds() )
                    {
                        String whereCondition = sqlTabs.getCriteriaFieldsTable()
                                                       .getCellValue( columnId, rowId.toString() );
                        if ( SharedUtil.isNotNullAndNotEmpty( whereCondition ) )
                        {
                            String columnValue = sqlColumn.getColumnAlias();
                            if ( sqlBuilder.isTableGrouping() )
                            {
                                columnValue = sqlColumn.toShortString();
                            }
                            sqlBuilder.addColumnWhereCondition( columnId, rowId.toString(), columnValue,
                                                                whereCondition );
                        }
                    }
                }

                // save column
                boolean saveColumn = Boolean.parseBoolean( sqlTabs.getHeaderFieldsTable()
                                                                  .getCellValue( columnId,
                                                                                 BaseTable.DESCRIPTION_SHOW ) );
                if ( saveColumn )
                {
                    sqlBuilder.addColumn( sqlColumn );
                }
            }
        }

        // table and its alias
        for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                  .iterator(); iterator.hasNext(); )
        {
            Component component = iterator.next();
            if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
            {
                DroppedTable table = (DroppedTable) component;
                String tableName = table.getTableName();

                // check if table is query element form json query object
                if ( tableName.contains( StringUtil.DOT_VALUE )
                    && tableName.split( StringUtil.STRING_DOT_SPLITTER ).length == 2 )
                {
                    String queryGroup = tableName.split( StringUtil.STRING_DOT_SPLITTER )[0];
                    String queryName = tableName.split( StringUtil.STRING_DOT_SPLITTER )[1];
                    if ( jsonBuilder.getJsonQueries()
                                    .containsKey( queryGroup ) )
                    {
                        JsonQuery jsonQuery = jsonBuilder.getJsonQueries()
                                                         .get( queryGroup )
                                                         .get( queryName );
                        if ( SharedUtil.isNotNull( jsonQuery ) )
                        {
                            tableName = jsonQuery.get( JsonQueryElement.QUERY_AS_COLUMN )
                                                 .toString();
                            sqlBuilder.setQueryDataSourceId( jsonQuery.get( JsonQueryElement.DATA_SOURCE_ID )
                                                                      .toString() );
                        }
                    }
                }

                SqlTable sqlTable = new SqlTable();
                sqlTable.setTableName( tableName )
                        .as( table.getTableAlias() );
                sqlBuilder.addTable( sqlTable );
            }
        }

        // table joins
        for ( JoinedTables joinedTables : droppedTableContainer.getJoinedTables() )
        {
            SqlTable fromTable = new SqlTable().as( joinedTables.getFromTable() );
            SqlTable toTable = new SqlTable().as( joinedTables.getToTable() );
            for ( SqlTable table : sqlBuilder.getTables() )
            {
                if ( table.getTableAlias()
                          .equals( fromTable.getTableAlias() ) )
                {
                    fromTable = new SqlTable( table.getTableName(), table.getTableAlias() );
                }
                if ( table.getTableAlias()
                          .equals( toTable.getTableAlias() ) )
                {
                    toTable = new SqlTable( table.getTableName(), table.getTableAlias() );
                }
            }

            SqlJoin sqlJoin = new SqlJoin( fromTable, toTable, joinedTables.getTablesJoin() );
            for ( int index = 0; index < Math.min( joinedTables.getFromFields().length,
                                                   joinedTables.getToFields().length ); index++ )
            {
                sqlJoin.setFromToColumn( joinedTables.getFromFields()[index], joinedTables.getToFields()[index] );
            }
            sqlBuilder.addJoin( sqlJoin );
        }

        // sql editor context
        sqlTabs.getSqlQueryEditor()
               .setReadOnly( false );
        sqlTabs.getSqlQueryEditor()
               .setValue( sqlBuilder.toString() );
        sqlTabs.getSqlQueryEditor()
               .setCursorRowCol( 0, 0 );
        sqlTabs.getSqlQueryEditor()
               .setReadOnly( true );

        // test current query
        sqlMenu.setQueryButtonEnabled( !sqlTabs.getSqlQueryEditor()
                                               .isEmpty() );
    }

    /**
     * Get dQuery version.
     * 
     * @return dQuery version
     */
    public static String getVersion()
    {
        if ( !SharedUtil.isNull( ManifestHelper.getManifest() ) )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( VERSION );
        }
        return null;
    }

    /**
     * Get dQuery git version.
     * 
     * @return dQuery git version
     */
    public static String getGitVersion()
    {
        if ( !SharedUtil.isNull( ManifestHelper.getManifest() ) )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( GIT_VERSION );
        }
        return null;
    }

    /**
     * Set data sources item visible.
     * 
     * @param visible
     *            if true data sources item is visible, otherwise is hidden
     */
    public void setDataSourcesItemVisible( boolean visible )
    {
        dataSourcesMenu.setDataSourcesItemVisible( visible );
    }

    /**
     * Get component context.
     * 
     * @return component context
     */
    public Context getContext()
    {
        saveCurrentDataSource();
        context.prepare();
        return context;
    }

    /**
     * Get added table left position. Table left position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @return added table left position
     */
    public int getAddedTableLeftPosition()
    {
        return addedTableLeftPosition;
    }

    /**
     * Set added table left position. Table left position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @param left
     *            added table left position
     */
    public void setAddedTableLeftPosition( int left )
    {
        addedTableLeftPosition = left;
    }

    /**
     * Get added table top position. Table top position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @return added table top position
     */
    public int getAddedTableTopPosition()
    {
        return addedTableTopPosition;
    }

    /**
     * Set added table top position. Table top position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @param top
     *            added table top position
     */
    public void setAddedTableTopPosition( int top )
    {
        addedTableTopPosition = top;
    }

    /**
     * Set added table left and top positions. Table left and top positions are a start position for the tables which is
     * added to the table container (middle part of dQuery UI) from data source menu with double click on menu item
     * element.
     * 
     * @param left
     *            added table left position
     * @param top
     *            added table top position
     */
    public void setAddedTablePosition( int left, int top )
    {
        setAddedTableLeftPosition( left );
        setAddedTableTopPosition( top );
    }

    /**
     * Get current data source.
     * 
     * @return current data source
     */
    public JdbcUtil getCurrentDataSource()
    {
        return currentJdbc;
    }

    /**
     * Save current data source.
     */
    public void saveCurrentDataSource()
    {
        if ( SharedUtil.isNullOrEmpty( currentJdbc.getDescription() ) )
        {
            currentJdbc.setDescription( currentDataSourceId );
        }
        dataSources.put( currentDataSourceId, currentJdbc );
    }

    /**
     * Get data source for specified database.
     * 
     * @param database
     *            database name
     * @return jdbc util
     */
    public JdbcUtil getDataSourceForDatabase( String database )
    {
        JdbcUtil jdbcUtil = null;
        if ( !SharedUtil.isNullOrEmpty( database ) )
        {
            for ( Entry<String, JdbcUtil> entry : dataSources.entrySet() )
            {
                String key = entry.getKey();
                JdbcUtil dataSource = entry.getValue();
                try
                {
                    String catalog = dataSource.getConnection()
                                               .getCatalog();
                    String schema = dataSource.getConnection()
                                              .getSchema();
                    if ( ( SharedUtil.isNotNullAndNotEmpty( catalog ) && catalog.equals( database ) )
                        || ( SharedUtil.isNotNullAndNotEmpty( schema ) && schema.equals( database ) ) )
                    {
                        jdbcUtil = dataSources.get( key );
                        break;
                    }
                }
                catch ( SQLException e )
                {
                    Logger.getLogger( DQuery.class.getName() )
                          .log( Level.INFO, e.getLocalizedMessage(), e );
                    jdbcUtil = null;
                }
            }
        }

        // can not find data source for database
        if ( SharedUtil.isNull( jdbcUtil ) )
        {
            new ErrorUtil( MessageFormat.format( ErrorUtil.CANNOT_FIND_DATASOURCE_FOR_DATABASE, database ) ).show();
        }

        return jdbcUtil;
    }

    /**
     * Get data source.
     * 
     * @param dataSourceId
     *            id of the data source
     * @return jdbc util
     */
    public JdbcUtil getDataSource( String dataSourceId )
    {
        if ( SharedUtil.isNullOrEmpty( dataSourceId ) )
        {
            throw new IllegalArgumentException( IDENTIFIER_IS_EMPTY );
        }
        else if ( SharedUtil.isNull( dataSources.get( dataSourceId ) ) )
        {
            throw new IllegalArgumentException( MessageFormat.format( DATASOURCES_NOT_HAVE_THIS_IDENTIFIER,
                                                                      dataSourceId ) );
        }

        currentJdbc = dataSources.get( dataSourceId );
        return currentJdbc;
    }

    /**
     * Add data source.
     * 
     * @param dataSourceId
     *            id of added data source
     * @return jdbc helper of added data source
     */
    public JdbcUtil addDataSource( String dataSourceId )
    {
        if ( !SharedUtil.isNull( dataSources.get( dataSourceId ) ) )
        {
            throw new IllegalArgumentException( MessageFormat.format( DATASOURCES_ALREADY_HAVE_THIS_IDENTIFIER,
                                                                      dataSourceId ) );
        }
        else if ( !SharedUtil.isNull( currentJdbc ) )
        {
            saveCurrentDataSource();
        }

        currentDataSourceId = dataSourceId;
        currentJdbc = new JdbcUtil();
        dataSources.put( currentDataSourceId, currentJdbc );
        return currentJdbc;
    }

    /**
     * Check if data source is enabled.
     * 
     * @param dataSourceId
     *            data source id
     * @return true data source is enabled, otherwise false
     */
    public boolean isDataSourceEnabled( String dataSourceId )
    {
        return dataSourcesMenu.isDataSourceEnabled( dataSourceId );
    }

    /**
     * Set data source enabled. User can later change a state of the data source if data source menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param enabled
     *            if true data source is enabled, otherwise is disabled
     */
    public void setDataSourceEnabled( String dataSourceId, boolean enabled )
    {
        dataSourcesMenu.setDataSourceEnabled( dataSourceId, enabled );
    }

    /**
     * Check if data source table is disabled.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @return true data source table is disabled, otherwise false
     */
    public boolean isDataSourceTableDisabled( String dataSourceId, String tableName )
    {
        return dataSourcesMenu.isDataSourceTableDisabled( dataSourceId, tableName );
    }

    /**
     * Set data source table disabled. User can later change a state of the data source table if data source menu is
     * visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @param disabled
     *            if true data source table is disabled, otherwise is enabled
     */
    public void setDataSourceTableDisabled( String dataSourceId, String tableName, boolean disabled )
    {
        dataSourcesMenu.setDataSourceTableDisabled( dataSourceId, tableName, disabled );
    }

    /**
     * Check if can show dropped table data.
     * 
     * @return true if can show dropped table data, otherwise false
     */
    public boolean isShowDroppedTableData()
    {
        return showDroppedTableData;
    }

    /**
     * Show dropped table data.
     * 
     * @param showDroppedTableData
     *            if true show dropped table data, otherwise not
     */
    public void setShowDroppedTableData( boolean showDroppedTableData )
    {
        this.showDroppedTableData = showDroppedTableData;
    }

    /**
     * Check if can show query definition.
     * 
     * @return true if can show query definition, otherwise false
     */
    public boolean isShowQueryDefinition()
    {
        return showQueryDefinition;
    }

    /**
     * Show query definition.
     * 
     * @param showQueryDefinition
     *            if true show query definition, otherwise not
     */
    public void setShowQueryDefinition( boolean showQueryDefinition )
    {
        this.showQueryDefinition = showQueryDefinition;
    }

    /**
     * Check if show query data is enabled.
     * 
     * @return true if show query data is enabled, otherwise false
     */
    public boolean isShowQueryDataEnabled()
    {
        return sqlMenu.isShowQueryDataEnabled();
    }

    /**
     * Set show query data enabled.
     * 
     * @param showQueryData
     *            if true query data show is enabled, otherwise not
     */
    public void setShowQueryDataEnabled( boolean showQueryData )
    {
        sqlMenu.setShowQueryDataEnabled( showQueryData );
    }

    /**
     * Check if create a new group is enabled.
     * 
     * @return true if create a new group is enabled, otherwise false
     */
    public boolean isCreateNewQueryGroupEnabled()
    {
        return sqlMenu.isCreateNewQueryGroupEnabled();
    }

    /**
     * Set create a new group enabled.
     * 
     * @param createNewQueryGroupEnabled
     *            enable create a new query group
     */
    public void setCreateNewQueryGroupEnabled( boolean createNewQueryGroupEnabled )
    {
        sqlMenu.setCreateNewQueryGroupEnabled( createNewQueryGroupEnabled );
    }

    /**
     * Get common functions.
     * 
     * @return common functions
     */
    public String[] getCommonFunctions()
    {
        return sqlTabs.getCommonFunctions()
                      .getFunctions();
    }

    /**
     * Add common functions.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void addCommonFunction( String function, String description )
    {
        sqlTabs.getCommonFunctions()
               .addFunction( function, description );
    }

    /**
     * Get common function description.
     * 
     * @param function
     *            function name
     * @return function description
     */
    public String getCommonFunctionDescription( String function )
    {
        return sqlTabs.getCommonFunctions()
                      .getDescription( function );
    }

    /**
     * Set common function description.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void setCommonFunctionDescription( String function, String description )
    {
        sqlTabs.getCommonFunctions()
               .setDescription( function, description );
    }

    /**
     * Get common function's parameter values.
     * 
     * @param function
     *            function name
     * @return function parameter values
     */
    public Object[] getCommonFunctionParameterValues( String function )
    {
        return sqlTabs.getCommonFunctions()
                      .getParameterValues( function );
    }

    /**
     * Set common function's parameter values.
     * 
     * @param function
     *            function name
     * @param parameterValues
     *            function parameter values
     */
    public void setCommonFunctionParameterValues( String function, Object[] parameterValues )
    {
        sqlTabs.getCommonFunctions()
               .setParameterValues( function, parameterValues );
    }

    /**
     * Get common function parameters.
     * 
     * @param function
     *            function name
     * @return function parameters
     */
    public String[] getCommonFunctionParameters( String function )
    {
        return sqlTabs.getCommonFunctions()
                      .getParameters( function );
    }

    /**
     * Set common function parameters.
     * 
     * @param function
     *            function name
     * @param parameters
     *            function parameters
     */
    public void setCommonFunctionParameters( String function, String... parameters )
    {
        sqlTabs.getCommonFunctions()
               .setParameters( function, parameters );
    }

    /**
     * Get total functions.
     * 
     * @return total functions
     */
    public String[] getTotalFunctions()
    {
        return sqlTabs.getTotalFunctions()
                      .getFunctions();
    }

    /**
     * Add total functions.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void addTotalFunction( String function, String description )
    {
        sqlTabs.getTotalFunctions()
               .addFunction( function, description );
    }

    /**
     * Get total function description.
     * 
     * @param function
     *            function name
     * @return function description
     */
    public String getTotalFunctionDescription( String function )
    {
        return sqlTabs.getTotalFunctions()
                      .getDescription( function );
    }

    /**
     * Set total function description.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void setTotalFunctionDescription( String function, String description )
    {
        sqlTabs.getTotalFunctions()
               .setDescription( function, description );
    }

    /**
     * Get total function's parameter values.
     * 
     * @param function
     *            function name
     * @return function parameter values
     */
    public Object[] getTotalFunctionParameterValues( String function )
    {
        return sqlTabs.getTotalFunctions()
                      .getParameterValues( function );
    }

    /**
     * Set total function's parameter values.
     * 
     * @param function
     *            function name
     * @param parameterValues
     *            function parameter values
     */
    public void setTotalFunctionParameterValues( String function, Object[] parameterValues )
    {
        sqlTabs.getTotalFunctions()
               .setParameterValues( function, parameterValues );
    }

    /**
     * Get total function parameters.
     * 
     * @param function
     *            function name
     * @return function parameters
     */
    public String[] getTotalFunctionParameters( String function )
    {
        return sqlTabs.getTotalFunctions()
                      .getParameters( function );
    }

    /**
     * Set total function parameters.
     * 
     * @param function
     *            function name
     * @param parameters
     *            function parameters
     */
    public void setTotalFunctionParameters( String function, String... parameters )
    {
        sqlTabs.getCommonFunctions()
               .setParameters( function, parameters );
    }

    /**
     * Get sources page length.
     * 
     * @return sources page length
     */
    public int getSourcesPageLength()
    {
        return dataSourcesMenu.getSourcesPageLength();
    }

    /**
     * Set sources page length.
     * 
     * @param sourcesPageLength
     *            sources page length
     */
    public void setSourcesPageLength( int sourcesPageLength )
    {
        dataSourcesMenu.setSourcesPageLength( sourcesPageLength );
    }

    /**
     * Get criteria page length.
     * 
     * @return criteria page length
     */
    public int getCriteriaPageLength()
    {
        return sqlTabs.getCriteriaPageLength();
    }

    /**
     * Set criteria page length.
     * 
     * @param criteriaPageLength
     *            criteria page length
     */
    public void setCriteriaPageLength( int criteriaPageLength )
    {
        sqlTabs.setCriteriaPageLength( criteriaPageLength );
    }

    /**
     * Add json query to data source menu.
     * 
     * @param query
     *            json query
     * @param showQuery
     *            if true json query's structure is showing in table container and field columns, otherwise not
     */
    public void addQuery( String query, boolean showQuery )
    {
        if ( showQuery && SharedUtil.isNotNull( context ) )
        {
            context.clearTableContainerAndFieldColumns();
        }

        JsonQuery jsonQuery = null;
        JSONObject jsonObject = null;
        try
        {
            jsonObject = new JSONObject( query );

            // create json query
            String queryGroup = jsonObject.getString( JsonQueryElement.GROUP.toString() );
            String queryName = jsonObject.getString( JsonQueryElement.NAME.toString() );
            if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
            {
                jsonQuery = new JsonQuery( queryGroup, queryName );
            }

            if ( SharedUtil.isNotNullAndNotEmpty( jsonQuery ) )
            {
                sqlMenu.setQueryGroup( queryGroup );
                sqlMenu.setQueryName( queryName );
                sqlMenu.getTextField()
                       .setValue( sqlMenu.getQueryGroupWithQueryName() );
                sqlMenu.getTextField()
                       .setEnabled( true );

                // tables
                if ( jsonObject.has( JsonQueryElement.TABLES.toString() ) )
                {
                    JSONArray jsonTables = jsonObject.getJSONArray( JsonQueryElement.TABLES.toString() );
                    for ( int index = 0; index < jsonTables.length(); index++ )
                    {
                        JSONObject jsonTable = (JSONObject) jsonTables.get( index );

                        // base table data
                        String dataSourceId = jsonTable.getString( JsonQuery.DATA_SOURCE_ID );
                        String databaseName = jsonTable.getString( JsonQuery.DATABASE );
                        String tableName = jsonTable.getString( JsonQuery.TABLE );
                        String tableAlias = jsonTable.getString( JsonQuery.TABLE_ALIAS );
                        String tableType = jsonTable.getString( JsonQuery.TABLE_TYPE );

                        // table position
                        JSONObject jsonTablePosition = jsonTable.getJSONObject( JsonQuery.TABLE_POSITION );
                        int left = jsonTablePosition.getInt( JsonQuery.TABLE_POSITION_LEFT );
                        int top = jsonTablePosition.getInt( JsonQuery.TABLE_POSITION_TOP );

                        // table data
                        jsonQuery.addTable( left, top, dataSourceId, databaseName, tableName, tableAlias, tableType );

                        // drop table
                        String id = dataSourceId;
                        if ( tableType.equals( TableType.QUERY.toString() ) )
                        {
                            id = databaseName;
                        }
                        for ( MenuItem menuItem : dataSourcesMenu.getMenuItems() )
                        {
                            if ( id.equals( menuItem.getData() ) )
                            {
                                for ( MenuItemElement itemElement : menuItem.getMenuItemElements() )
                                {
                                    Object data = itemElement.getData();
                                    if ( SharedUtil.isNotNullAndNotEmpty( data ) && data.toString()
                                                                                        .contains( StringUtil.DOT_VALUE ) )
                                    {
                                        // format: data source id, database name, table name, table columns
                                        String[] datas = data.toString()
                                                             .split( StringUtil.STRING_DOT_SPLITTER );
                                        if ( dataSourceId.equals( datas[0] ) && databaseName.equals( datas[1] )
                                            && tableName.equals( datas[2] ) )
                                        {
                                            context.dropTable( left, top, dataSourceId, databaseName, tableName,
                                                               datas[3], tableType );
                                            break;
                                        }
                                    }
                                }
                                break;
                            }
                        }
                    }
                }

                // header and criteria fields
                if ( jsonObject.has( JsonQueryElement.FIELDS.toString() ) )
                {
                    Map<Integer, Map<String, String>> columnsValues = new HashMap<>();
                    JSONArray jsonFields = jsonObject.getJSONArray( JsonQueryElement.FIELDS.toString() );
                    for ( int index = 0; index < jsonFields.length(); index++ )
                    {
                        String columnId = null;
                        Map<String, String> columnValues = new HashMap<>();
                        JSONObject jsonField = (JSONObject) jsonFields.get( index );
                        for ( Iterator<String> keys = jsonField.keys(); keys.hasNext(); )
                        {
                            String key = keys.next();
                            if ( key.equals( JsonQuery.TABLE_FIELD ) )
                            {
                                columnId = jsonField.getString( key );
                            }
                            else
                            {
                                switch ( key )
                                {
                                    case BaseTable.DESCRIPTION_COMMON:
                                    case BaseTable.DESCRIPTION_SORTING:
                                    case BaseTable.DESCRIPTION_TOTAL:
                                        setHeaderRowVisible( key, true );
                                        columnValues.put( key, jsonField.getString( key ) );
                                        break;
                                    default:
                                        columnValues.put( key, jsonField.getString( key ) );
                                        break;
                                }
                                columnValues.put( key, jsonField.getString( key ) );
                            }
                        }
                        columnsValues.put( Integer.parseInt( columnId ), columnValues );
                        jsonQuery.addTableField( columnId, columnValues.keySet()
                                                                       .toArray(),
                                                 columnValues.values()
                                                             .toArray() );
                    }

                    // values for header and criteria table
                    for ( int index = 0; index < columnsValues.size(); index++ )
                    {
                        Map<String, String> columnValues = columnsValues.get( index );

                        // add column with base values
                        String columnId =
                            sqlTabs.addFieldColumn( columnValues.get( BaseTable.DESCRIPTION_TABLE ),
                                                    columnValues.get( BaseTable.DESCRIPTION_FIELD ),
                                                    columnValues.get( BaseTable.DESCRIPTION_NAME ), true );

                        // move column to the right place
                        sqlTabs.addFieldColumnBefore( columnId, sqlTabs.getHeaderFieldsTable()
                                                                       .getVisibleColumns()[index].toString() );

                        // add other values
                        for ( Entry<String, String> entry : columnValues.entrySet() )
                        {
                            String key = entry.getKey();
                            String columnValue = entry.getValue();
                            if ( !SharedUtil.isInteger( key ) )
                            {
                                sqlTabs.getHeaderFieldsTable()
                                       .setCellValue( columnId, key, columnValue );
                            }
                            else
                            {
                                sqlTabs.getCriteriaFieldsTable()
                                       .setCellValue( columnId, key, columnValue );
                            }
                        }
                    }
                }

                // table joins
                if ( jsonObject.has( JsonQueryElement.JOINS.toString() ) )
                {
                    JSONArray jsonJoins = jsonObject.getJSONArray( JsonQueryElement.JOINS.toString() );
                    for ( int index = 0; index < jsonJoins.length(); index++ )
                    {
                        JSONObject jsonJoin = (JSONObject) jsonJoins.get( index );
                        TablesJoin tablesJoin =
                            TablesJoin.getJoinFromString( jsonJoin.getString( JsonQuery.TABLE_JOINS_TYPE ) );
                        String fromTable = jsonJoin.getString( JsonQuery.TABLE_JOINS_FROM );
                        String[] fromFields =
                            StringUtil.asStringArray( jsonJoin.getJSONArray( JsonQuery.TABLE_JOINS_FROM_FIELDS ) );
                        String toTable = jsonJoin.getString( JsonQuery.TABLE_JOINS_TO );
                        String[] toFields =
                            StringUtil.asStringArray( jsonJoin.getJSONArray( JsonQuery.TABLE_JOINS_TO_FIELDS ) );

                        // save table joins
                        jsonQuery.addTableJoins( fromTable, fromFields, toTable, toFields, tablesJoin );

                        // add to table container
                        for ( int i = 0; i < Math.min( fromFields.length, toFields.length ); i++ )
                        {
                            droppedTableContainer.setJoinedTables( fromTable, fromFields[i], toTable, toFields[i],
                                                                   tablesJoin );
                        }
                    }
                    context.getState( true )
                           .setJoinedTables( droppedTableContainer.getJoinedTablesAsStringList() );
                }

                // add json query to json query builders
                jsonBuilder.addQuery( jsonQuery );

                // refresh sql editor
                refreshSqlEditorContext();
            }
        }
        catch ( JSONException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getLocalizedMessage(), e );
            new ErrorUtil( MessageFormat.format( ErrorUtil.JSON_QUERY_IMPORT, e.getLocalizedMessage() ) ).show();
        }
    }

    /**
     * Context of the dQuery component.
     */
    public class Context
        extends CustomLayout
    {

        private static final long serialVersionUID = 4728885766134972348L;

        private static final String TABLES = "tables";

        private static final String CONTENT = "content";

        private static final String SQL_MENU = "sql-menu";

        private static final String FIELDS = "fields";

        private static final String TR_START = "<tr><td class='";

        private static final String TR_MIDDLE = "'><div location='";

        private static final String TR_END = "'></div></td></tr>";

        private static final String HTML = "<table style='width:100%;height:100%;border-collapse:collapse;'>" //
            + TR_START + TABLES + TR_MIDDLE + TABLES + TR_END //
            + TR_START + CONTENT + "'><div style='height:100%;' location='" + CONTENT + TR_END //
            + TR_START + SQL_MENU + TR_MIDDLE + SQL_MENU + TR_END //
            + TR_START + FIELDS + TR_MIDDLE + FIELDS + TR_END //
            + "</table>";

        /**
         * Creates a new empty instance.
         */
        public Context()
        {
            setSizeFull();

            registerRpc( new QueryServerRpc()
            {
                private static final long serialVersionUID = -7489427457610395747L;

                @Override
                public void changeJoinedTables( String joinedTablesId, Integer joinedFieldsIndex )
                {
                    UI.getCurrent()
                      .addWindow( new JoinedTablesWindow( joinedTablesId, joinedFieldsIndex )
                      {
                          private static final long serialVersionUID = -7082985030501860887L;

                          @Override
                          protected void setJoinedTablesJoinType( String fromTable, String toTable,
                                                                  TablesJoin tablesJoin, int joinedFieldsIndex )
                          {
                              if ( tablesJoin.equals( TablesJoin.REMOVE ) )
                              {
                                  droppedTableContainer.removeJoinedTablesJoin( fromTable, toTable, joinedFieldsIndex );
                              }
                              else
                              {
                                  droppedTableContainer.setJoinedTablesJoinType( fromTable, toTable, tablesJoin );
                              }
                              context.getState( true )
                                     .setJoinedTables( droppedTableContainer.getJoinedTablesAsStringList() );

                              // enable clear table container
                              sqlMenu.setClearButtonEnabled( true );

                              // enable save query
                              sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                              // refresh sql editor
                              refreshSqlEditorContext();
                          }
                      } );
                }

                @Override
                public void tableMoved( String movedTableId, Integer left, Integer top )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( movedTableId ) )
                    {
                        for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                                  .iterator(); iterator.hasNext(); )
                        {
                            Component component = iterator.next();
                            if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
                            {
                                DroppedTable droppedTable = (DroppedTable) component;
                                if ( droppedTable.getCaption()
                                                 .equals( movedTableId ) )
                                {
                                    droppedTable.setData( MessageFormat.format( DroppedTableContainer.TABLE_POSITION_DATA,
                                                                                left, top ) );

                                    // enable save query
                                    sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );
                                    break;
                                }
                            }
                        }
                    }
                }
            } );
        }

        /**
         * Get the shared state.
         */
        @Override
        protected QueryState getState()
        {
            return (QueryState) super.getState();
        }

        /**
         * Get the shared state.
         */
        @Override
        protected QueryState getState( boolean markAsDirty )
        {
            return (QueryState) super.getState( markAsDirty );
        }

        /**
         * Drop table.
         * 
         * @param left
         *            dropped table left position
         * @param top
         *            dropped table top position
         * @param dataSourceId
         *            id of the data source for the dropped table
         * @param databaseName
         *            database name for the dropped table
         * @param tableName
         *            table name for the dropped table
         * @param tableColumns
         *            table columns
         * @param tableType
         *            table type: table or query
         */
        private void dropTable( int left, int top, String dataSourceId, String databaseName, String tableName,
                                String tableColumns, String tableType )
        {
            // check if added query is itself definition
            if ( MessageFormat.format( BaseTable.TABLE_NAME, databaseName, tableName )
                              .equals( MessageFormat.format( BaseTable.TABLE_NAME, sqlMenu.getQueryGroup(),
                                                             sqlMenu.getQueryName() ) ) )
            {
                new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY ).show();
                return;
            }

            DroppedTable droppedTable = new DroppedTable();
            droppedTable.prepare( getDataSource( dataSourceId ), dataSourceId, databaseName, tableName,
                                  MessageFormat.format( DroppedTableContainer.TABLE_DROP_NAME, tableName,
                                                        droppedTableContainer.getSameNameIndex( tableName ) ),
                                  tableColumns, tableType );
            droppedTable.showTableData( showDroppedTableData );
            droppedTable.showQueryDefinition( showQueryDefinition && tableType.equals( TableType.QUERY.toString() )
                && jsonBuilder.getJsonQueries()
                              .containsKey( databaseName ) );

            // add table field to tabs header field table
            droppedTable.addItemClickListener( new ItemClickListener()
            {
                private static final long serialVersionUID = -2054759352120240782L;

                @Override
                public void itemClick( ItemClickEvent event )
                {
                    if ( event.isDoubleClick() && SharedUtil.isNotNullAndNotEmpty( event.getItemId() ) )
                    {
                        // insert new field column
                        // format: database source id, database name, table name, column name, table alias
                        String itemId = event.getItemId()
                                             .toString();
                        String[] data = itemId.split( StringUtil.STRING_DOT_SPLITTER );
                        String newColumnId = sqlTabs.addFieldColumn( data[4], data[3], data[3], true );
                        String emptyColumnId = sqlTabs.getFirstEmptyFieldColumn();

                        if ( SharedUtil.isNotNullAndNotEmpty( newColumnId ) )
                        {
                            // new field column is added to the fields table
                            if ( SharedUtil.isNotNullAndNotEmpty( emptyColumnId ) )
                            {
                                sqlTabs.reverseFieldColumns( newColumnId, emptyColumnId );
                                sqlTabs.addMissingFieldColumns();
                            }

                            // unselect row in dropped table
                            ( (Table) event.getComponent() ).unselect( itemId );

                            // enable clear table container
                            sqlMenu.setClearButtonEnabled( true );

                            // enable save query
                            sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                            // enable/disable function on group of fields
                            sqlMenu.setGroupButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                            // refresh sql editor
                            refreshSqlEditorContext();
                        }
                        else
                        {
                            new ErrorUtil( ErrorUtil.CANNOT_ADD_NEW_COLUMN ).show();
                        }
                    }
                }
            } );

            // remove table from table container
            droppedTable.addClickListener( new ClickListener()
            {
                private static final long serialVersionUID = -1948012368378356027L;

                @Override
                public void buttonClick( ClickEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getButton() ) && SharedUtil.isNotNullAndNotEmpty( event.getButton()
                                                                                                            .getData() ) )
                    {
                        Button button = event.getButton();
                        String tableData = button.getData()
                                                 .toString();
                        switch ( button.getCaption() )
                        {
                            case DroppedTable.BUTTON_DATA:
                                if ( tableData.contains( StringUtil.DOT_VALUE )
                                    && 4 <= tableData.split( StringUtil.STRING_DOT_SPLITTER ).length )
                                {
                                    // format: database source id, database name, table name, table alias
                                    String dataSourceId = tableData.split( StringUtil.STRING_DOT_SPLITTER )[0];
                                    String database = tableData.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                    String table = tableData.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                    String tableAlias = tableData.split( StringUtil.STRING_DOT_SPLITTER )[3];

                                    // data query: SELECT * FROM database.table
                                    String query = MessageFormat.format( TableDataWindow.ROW_SELECT, database, table );

                                    // check if table is query element form json query object
                                    // queryGroup is database, queryName is table, data query is from saved query
                                    if ( jsonBuilder.getJsonQueries()
                                                    .containsKey( database ) )
                                    {
                                        JsonQuery jsonQuery = jsonBuilder.getJsonQueries()
                                                                         .get( database )
                                                                         .get( table );
                                        if ( SharedUtil.isNotNull( jsonQuery ) )
                                        {
                                            query = jsonQuery.get( JsonQueryElement.QUERY )
                                                             .toString();
                                        }
                                    }

                                    UI.getCurrent()
                                      .addWindow( new TableDataWindow( dataSources.get( dataSourceId ), tableAlias,
                                                                       query ) );
                                }
                                break;
                            case DroppedTable.BUTTON_REMOVE:
                                sqlTabs.removeFieldColumns( tableData );
                                droppedTableContainer.removeJoinedTables( tableData );
                                sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                                // remove table from table container
                                droppedTableContainer.getContainer()
                                                     .removeComponent( button.getParent()
                                                                             .getParent()
                                                                             .getParent() );

                                // enable clear table container
                                sqlMenu.setClearButtonEnabled( droppedTableContainer.getContainer()
                                                                                    .getComponentCount() != 0 );

                                // enable save query
                                sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                                // enable/disable function on group of fields
                                sqlMenu.setGroupButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                                // remove table joins form removed table
                                getState( true ).setJoinedTables( droppedTableContainer.getJoinedTablesAsStringList() );
                                break;
                            case DroppedTable.BUTTON_QUERY:
                                if ( tableData.contains( StringUtil.DOT_VALUE )
                                    && 4 <= tableData.split( StringUtil.STRING_DOT_SPLITTER ).length )
                                {
                                    // format: database source id, database name, table name, table alias
                                    String jsonQueryGroup = tableData.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                    String jsonQueryName = tableData.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                    JsonQuery jsonQuery = jsonBuilder.getQueries( jsonQueryGroup )
                                                                     .get( jsonQueryName );
                                    if ( SharedUtil.isNotNull( jsonQuery ) )
                                    {
                                        addQuery( jsonQuery.toString(), true );
                                        break;
                                    }
                                }
                                break;
                            default:
                                break;
                        }

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            // create tables join
            droppedTable.addDropHandler( new DropHandler()
            {

                private static final long serialVersionUID = -2389935918885846115L;

                @Override
                public AcceptCriterion getAcceptCriterion()
                {
                    return AcceptAll.get();
                }

                @Override
                public void drop( DragAndDropEvent event )
                {
                    if ( event.getTransferable() instanceof DataBoundTransferable )
                    {
                        DataBoundTransferable dragData = (DataBoundTransferable) event.getTransferable();
                        AbstractSelectTargetDetails dropData = (AbstractSelectTargetDetails) event.getTargetDetails();
                        if ( SharedUtil.isNotNull( dragData ) && SharedUtil.isNotNull( dropData ) )
                        {
                            Object dragId = dragData.getItemId();
                            Object dropId = dropData.getItemIdOver();
                            if ( SharedUtil.isNotNullAndNotEmpty( dragId )
                                && SharedUtil.isNotNullAndNotEmpty( dropId ) )
                            {
                                // format: database source id, database name, table name, column name, table alias
                                String[] dragIdSplitted = dragId.toString()
                                                                .split( StringUtil.STRING_DOT_SPLITTER );
                                String[] dropIdSplitted = dropId.toString()
                                                                .split( StringUtil.STRING_DOT_SPLITTER );
                                droppedTableContainer.setJoinedTables( dragIdSplitted[4], dragIdSplitted[3],
                                                                       dropIdSplitted[4], dropIdSplitted[3],
                                                                       TablesJoin.INNER );

                                sqlMenu.setClearButtonEnabled( true );
                                sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );
                                getState( true ).setJoinedTables( droppedTableContainer.getJoinedTablesAsStringList() );
                            }
                            ( (Table) dragData.getSourceComponent() ).unselect( dragId );
                        }

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            // add dropped table to table container
            droppedTable.setData( MessageFormat.format( DroppedTableContainer.TABLE_POSITION_DATA, left, top ) );
            droppedTableContainer.addTable( droppedTable,
                                            MessageFormat.format( DroppedTableContainer.TABLE_POSITION, left, top ) );

            // enable clear table container
            sqlMenu.setClearButtonEnabled( true );

        }

        /**
         * Remove json query.
         * 
         * @param queryGroup
         *            query group
         * @param queryName
         *            query name
         */
        private void removeJsonQuery( final String queryGroup, final String queryName )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
            {
                UI.getCurrent()
                  .addWindow( new MessageWindow( MessageWindow.WARNING,
                                                 MessageFormat.format( MessageWindow.QUERY_DELETE, queryGroup,
                                                                       queryName ) )
                  {
                      private static final long serialVersionUID = -5167834406049144385L;

                      @Override
                      public void onOkClick()
                      {
                          jsonBuilder.removeQuery( queryGroup, queryName );
                          dataSourcesMenu.removeQueryItemElement( queryGroup, queryName );
                          clearTableContainerAndFieldColumns();
                      }
                  } );
            }
        }

        /**
         * Save query in json format.
         * 
         * @param queryGroup
         *            query group
         * @param queryName
         *            query name
         */
        private void saveJsonQuery( String queryGroup, String queryName )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
            {
                JsonQuery jsonQuery = new JsonQuery( queryGroup, queryName );

                // tables
                for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                          .iterator(); iterator.hasNext(); )
                {
                    Component component = iterator.next();
                    if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
                    {
                        DroppedTable table = (DroppedTable) component;

                        // check if saved query include itself definition
                        if ( table.getTableName()
                                  .equals( MessageFormat.format( BaseTable.TABLE_NAME, sqlMenu.getQueryGroup(),
                                                                 sqlMenu.getQueryName() ) ) )
                        {
                            new ErrorUtil( ErrorUtil.CANNOT_SAVE_QUERY ).show();
                            return;
                        }

                        // add table
                        jsonQuery.addTable( table.getLeft(), table.getTop(), table.getDataSourceId(),
                                            table.getDatabaseName(), table.getTableName(), table.getTableAlias(),
                                            table.getTableType() );
                    }
                }

                // joins
                for ( JoinedTables joinedTables : droppedTableContainer.getJoinedTables() )
                {
                    jsonQuery.addTableJoins( joinedTables.getFromTable(), joinedTables.getFromFields(),
                                             joinedTables.getToTable(), joinedTables.getToFields(),
                                             joinedTables.getTablesJoin() );
                }

                // fields
                Integer column = 0;
                for ( Object visibleColumn : sqlTabs.getHeaderFieldsTable()
                                                    .getVisibleColumns() )
                {
                    String columnId = visibleColumn.toString();
                    if ( !BaseTable.COLUMN_LAST_FINAL.equals( columnId ) && !sqlTabs.isFieldColumnEmpty( columnId ) )
                    {
                        List<Object> rowIds = new ArrayList<>();
                        List<Object> rowValues = new ArrayList<>();

                        // header fields
                        for ( Object row : sqlTabs.getHeaderFieldsTable()
                                                  .getItemIds() )
                        {
                            String rowId = row.toString();
                            if ( sqlTabs.isHeaderRowVisible( rowId ) )
                            {
                                rowIds.add( rowId );
                                rowValues.add( sqlTabs.getHeaderFieldsTable()
                                                      .getCellValue( columnId, rowId ) );
                            }
                        }

                        // criteria fields
                        for ( Object row : sqlTabs.getCriteriaFieldsTable()
                                                  .getItemIds() )
                        {
                            rowIds.add( row.toString() );
                            rowValues.add( sqlTabs.getCriteriaFieldsTable()
                                                  .getCellValue( columnId, row.toString() ) );
                        }
                        jsonQuery.addTableField( column.toString(), rowIds.toArray(), rowValues.toArray() );
                        column += 1;
                    }
                }

                // query
                jsonQuery.addQuery( sqlBuilder.toString() );

                // save json query
                jsonBuilder.addQuery( jsonQuery );
                dataSourcesMenu.addQueryItemElement( jsonQuery );
                saveQuery( queryGroup, queryName, jsonQuery.toString() );
            }
        }

        /**
         * Clear table container and field columns.
         */
        private void clearTableContainerAndFieldColumns()
        {
            for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                      .iterator(); iterator.hasNext(); )
            {
                Component component = iterator.next();
                if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
                {
                    String tableName = ( (DroppedTable) component ).getTableName();
                    if ( SharedUtil.isNotNullAndNotEmpty( tableName ) && tableName.contains( StringUtil.DOT_VALUE )
                        && tableName.split( StringUtil.STRING_DOT_SPLITTER ).length == 2 )
                    {
                        // remove all field columns
                        sqlTabs.removeFieldColumns( tableName.split( StringUtil.STRING_DOT_SPLITTER )[1] );

                        // remove all joins
                        droppedTableContainer.removeJoinedTables( tableName.split( StringUtil.STRING_DOT_SPLITTER )[1] );
                    }
                }
            }

            // remove all components from table container
            droppedTableContainer.getContainer()
                                 .removeAllComponents();

            // send refreshed data for joined tables to the client
            context.getState( true )
                   .setJoinedTables( droppedTableContainer.getJoinedTablesAsStringList() );

            // clear sql editor
            sqlTabs.getSqlQueryEditor()
                   .setReadOnly( false );
            sqlTabs.getSqlQueryEditor()
                   .clear();
            sqlTabs.getSqlQueryEditor()
                   .setReadOnly( true );

            // clear query name
            sqlMenu.resetQueryName();

            // disable buttons
            sqlMenu.setClearButtonEnabled( false );
            sqlMenu.setGroupButtonEnabled( false );
            sqlMenu.setQueryButtonEnabled( false );
            sqlMenu.setRemoveButtonEnabled( false );
            sqlMenu.setSaveButtonEnabled( false );
        }

        /**
         * Run query.
         */
        private void runQuery()
        {
            if ( SharedUtil.isNotNullAndNotEmpty( sqlBuilder.toString() ) )
            {
                // get jdbc util
                JdbcUtil jdbcUtil;
                if ( sqlBuilder.getTables()
                               .isEmpty() )
                {
                    jdbcUtil = getDataSource( dataSources.keySet()
                                                         .iterator()
                                                         .next() );
                }
                else if ( SharedUtil.isNotNullAndNotEmpty( sqlBuilder.getQueryDataSourceId() ) )
                {
                    jdbcUtil = getDataSource( sqlBuilder.getQueryDataSourceId() );
                }
                else
                {
                    String database = sqlBuilder.getTables()
                                                .get( 0 )
                                                .getTableName()
                                                .split( StringUtil.STRING_DOT_SPLITTER )[0];
                    jdbcUtil = getDataSourceForDatabase( database );
                }

                // run query
                if ( SharedUtil.isNotNull( jdbcUtil ) )
                {
                    UI.getCurrent()
                      .addWindow( new TestQueryWindow( jdbcUtil, sqlBuilder.toString() ) );
                }
                else
                {
                    new ErrorUtil( ErrorUtil.CANNOT_RUN_QUERY ).show();
                }
            }
        }

        /**
         * Prepare data source menu.
         */
        private void prepareDataSourcesMenu()
        {
            dataSourcesMenu.prepare( dataSources );
            dataSourcesMenu.addHandler( new MenuItemElementClickHandler()
            {
                @Override
                public void onMenuItemElementClick( MenuItemElementClickEvent event )
                {
                    // drop table
                    if ( event.isDoubleClick() )
                    {
                        Object data = event.getMenuItemElement()
                                           .getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
                        {
                            // format: database source id, database name, table name, table columns, table type
                            String[] dataTableName = data.toString()
                                                         .split( StringUtil.STRING_DOT_SPLITTER );
                            String dataSourceId = dataTableName[0];
                            String databaseName = dataTableName[1];
                            String tableName = dataTableName[2];
                            String tableColumns = dataTableName[3];
                            String tableType = dataTableName[4];
                            if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId )
                                && SharedUtil.isNotNullAndNotEmpty( databaseName )
                                && SharedUtil.isNotNullAndNotEmpty( tableName )
                                && SharedUtil.isNotNullAndNotEmpty( tableColumns )
                                && SharedUtil.isNotNullAndNotEmpty( tableType ) )
                            {
                                dropTable( addedTableLeftPosition, addedTableTopPosition, dataSourceId, databaseName,
                                           tableName, tableColumns, tableType );
                            }
                        }
                    }
                }
            } );
        }

        /**
         * Prepare dropped table container.
         */
        private void prepareDroppedTableContainer()
        {
            droppedTableContainer.prepare();
            droppedTableContainer.setDropHandler( new DropHandler()
            {
                private static final long serialVersionUID = 4406196213826842664L;

                @Override
                public AcceptCriterion getAcceptCriterion()
                {
                    return AcceptAll.get();
                }

                @Override
                public void drop( DragAndDropEvent event )
                {
                    // drop table
                    if ( SharedUtil.isNotNull( event.getTransferable() ) )
                    {
                        Component component = event.getTransferable()
                                                   .getSourceComponent();
                        if ( SharedUtil.isNotNull( component.getParent() )
                            && component.getParent() instanceof MenuItemElement )
                        {
                            Object dataObject = ( (MenuItemElement) component.getParent() ).getData();
                            if ( SharedUtil.isNotNullAndNotEmpty( dataObject ) )
                            {
                                // format: database source id, database name, table name, field name, table alias, table
                                // columns
                                String dataTableName = dataObject.toString();
                                WrapperTargetDetails targetDetails = (WrapperTargetDetails) event.getTargetDetails();
                                MouseEventDetails eventDetails = targetDetails.getMouseEvent();

                                int left = eventDetails.getClientX() - targetDetails.getAbsoluteLeft();
                                int top = eventDetails.getClientY() - targetDetails.getAbsoluteTop();

                                String dataSourceId = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[0];
                                String databaseName = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                String tableName = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                String tableColumns = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[3];
                                String tableType = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[4];

                                if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId )
                                    && SharedUtil.isNotNullAndNotEmpty( databaseName )
                                    && SharedUtil.isNotNullAndNotEmpty( tableName )
                                    && SharedUtil.isNotNullAndNotEmpty( tableColumns ) )
                                {
                                    dropTable( left, top, dataSourceId, databaseName, tableName, tableColumns,
                                               tableType );
                                }
                            }
                        }
                    }
                }
            } );
        }

        /**
         * Prepare sql menu.
         */
        private void prepareSqlMenu()
        {
            sqlMenu.prepare();
            sqlMenu.addClickListener( new ClickListener()
            {
                private static final long serialVersionUID = -6740871942258241472L;

                @Override
                public void buttonClick( ClickEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof Button )
                    {
                        boolean setSaveButtonEnabled = true;
                        Button button = (Button) event.getComponent();
                        Object data = button.getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
                        {
                            // button action
                            switch ( data.toString() )
                            {
                                case SqlMenu.BUTTON_CLEAR:
                                    UI.getCurrent()
                                      .addWindow( new MessageWindow( MessageWindow.WARNING,
                                                                     MessageWindow.CLEAR_TABLE_CONTAINER_AND_FIELD_COLUMNS )
                                      {
                                          private static final long serialVersionUID = -8196232907668242224L;

                                          @Override
                                          public void onOkClick()
                                          {
                                              clearTableContainerAndFieldColumns();
                                          }
                                      } );
                                    break;
                                case SqlMenu.BUTTON_COMMON:
                                    setHeaderRowVisible( BaseTable.DESCRIPTION_COMMON,
                                                         !sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_COMMON ) );
                                    break;
                                case SqlMenu.BUTTON_DELETE:
                                    removeJsonQuery( sqlMenu.getQueryGroup(), sqlMenu.getQueryName() );
                                    break;
                                case SqlMenu.BUTTON_GROUP:
                                    UI.getCurrent()
                                      .addWindow( new SqlFunctionsWindow( sqlTabs ) );
                                    break;
                                case SqlMenu.BUTTON_QUERY:
                                    runQuery();
                                    break;
                                case SqlMenu.BUTTON_SAVE:
                                    saveJsonQuery( sqlMenu.getQueryGroup(), sqlMenu.getQueryName() );
                                    setSaveButtonEnabled = false;
                                    break;
                                case SqlMenu.BUTTON_SORTING:
                                    setHeaderRowVisible( BaseTable.DESCRIPTION_SORTING,
                                                         !sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_SORTING ) );
                                    break;
                                case SqlMenu.BUTTON_TOTAL:
                                    setHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL,
                                                         !sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL ) );
                                    break;
                                default:
                                    break;
                            }

                            // enable save query
                            sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() && setSaveButtonEnabled );

                            // enable/disable group button
                            if ( !sqlTabs.isAllFieldColumnsEmpty()
                                && ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_COMMON )
                                    || sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_SORTING )
                                    || sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL ) ) )
                            {
                                sqlMenu.setGroupButtonEnabled( true );
                            }
                            else
                            {
                                sqlMenu.setGroupButtonEnabled( false );
                            }

                            // refresh sql editor
                            refreshSqlEditorContext();
                        }
                    }
                }
            } );

            sqlMenu.addValueChangeListener( new ValueChangeListener()
            {
                private static final long serialVersionUID = 4999340710848144792L;

                @Override
                public void valueChange( ValueChangeEvent event )
                {
                    if ( SharedUtil.isNotNull( event ) && SharedUtil.isNotNull( event.getProperty() ) )
                    {
                        TextField textField = (TextField) event.getProperty();
                        if ( SharedUtil.isNotNull( textField ) )
                        {
                            if ( SharedUtil.isNotNullAndNotEmpty( textField.getValue() ) )
                            {
                                sqlMenu.setQueryName( textField.getValue() );
                                textField.setValue( sqlMenu.getQueryGroupWithQueryName() );
                            }
                            else
                            {
                                sqlMenu.resetQueryName();
                            }
                        }

                        // enable remove query
                        sqlMenu.setRemoveButtonEnabled( !textField.isEmpty() );

                        // enable save query
                        sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );
                    }
                }
            } );
        }

        /**
         * Prepare sql tabs.
         */
        private void prepareSqlTabs()
        {
            sqlTabs.addBlurListener( new BlurListener()
            {
                private static final long serialVersionUID = 2245396496450080414L;

                @Override
                public void blur( BlurEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getComponent() ) )
                    {
                        if ( event.getComponent() instanceof ComboBox )
                        {
                            String description = null;
                            ComboBox comboBox = (ComboBox) event.getComponent();
                            Object data = comboBox.getData();
                            Object value = comboBox.getValue();
                            if ( SharedUtil.isNotNullAndNotEmpty( data ) && SharedUtil.isNotNullAndNotEmpty( value ) )
                            {
                                switch ( data.toString() )
                                {
                                    case BaseTable.DESCRIPTION_COMMON:
                                        description = sqlTabs.getCommonFunctions()
                                                             .getDescription( value.toString() );
                                        break;
                                    case BaseTable.DESCRIPTION_SORTING:
                                        description = sqlTabs.getSortingFunctions()
                                                             .getDescription( value.toString() );
                                        break;
                                    case BaseTable.DESCRIPTION_TOTAL:
                                        description = sqlTabs.getTotalFunctions()
                                                             .getDescription( value.toString() );
                                        break;
                                    default:
                                        break;
                                }
                            }
                            comboBox.setDescription( description );
                        }
                        else if ( SharedUtil.isNotNull( event.getComponent()
                                                             .getParent() ) //
                            && event.getComponent()
                                    .getParent() instanceof HorizontalLayout )
                        {
                            HorizontalLayout layout = (HorizontalLayout) event.getComponent()
                                                                              .getParent();
                            if ( layout.getComponentCount() == 2 )
                            {
                                // trim value in text field
                                if ( layout.getComponent( 0 ) instanceof TextField )
                                {
                                    TextField textField = (TextField) layout.getComponent( 0 );
                                    textField.setValue( StringUtil.trim( textField.getValue() ) );
                                    if ( 0 < textField.getValue()
                                                      .length() )
                                    {
                                        sqlTabs.addMissingFieldColumns();
                                    }
                                }

                                // disable fields column in header and criteria tables
                                if ( layout.getComponent( 1 ) instanceof Button )
                                {
                                    Button button = (Button) layout.getComponent( 1 );
                                    Object columnId = button.getData();
                                    if ( !button.isVisible() && SharedUtil.isNotNullAndNotEmpty( columnId ) )
                                    {
                                        sqlTabs.setFieldColumnEnabled( columnId.toString(), false );
                                    }
                                }

                                // enable/disable function on group of fields
                                sqlMenu.setGroupButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );
                            }
                        }

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            sqlTabs.addClickListener( new ClickListener()
            {
                private static final long serialVersionUID = -34202109096977621L;

                @Override
                public void buttonClick( ClickEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof Button
                        && SharedUtil.isNotNull( event.getComponent()
                                                      .getParent() )
                        && event.getComponent()
                                .getParent() instanceof HorizontalLayout )
                    {
                        HorizontalLayout layout = (HorizontalLayout) event.getComponent()
                                                                          .getParent();
                        if ( layout.getComponent( 0 ) instanceof TextField )
                        {
                            Object columnId = ( (Button) event.getComponent() ).getData();
                            if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                            {
                                sqlTabs.removeFieldColumn( columnId.toString() );
                            }
                        }
                        else if ( layout.getComponent( 0 ) instanceof ComboBox )
                        {
                            Object columnId = ( (Button) event.getComponent() ).getData();
                            if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                            {
                                ComboBox comboBox = (ComboBox) layout.getComponent( 0 );
                                Object value = comboBox.getValue();
                                if ( SharedUtil.isNotNullAndNotEmpty( value ) && sqlTabs.getCommonFunctions()
                                                                                        .hasParameters( value.toString() ) )
                                {
                                    UI.getCurrent()
                                      .addWindow( new SqlParametersWindow( sqlTabs, value.toString(),
                                                                           columnId.toString() )
                                      {
                                          private static final long serialVersionUID = -4502056994959979919L;

                                          @Override
                                          public void saveNewParameters( CommonFunctions commonFunctions,
                                                                         String function )
                                          {
                                              // nothing to do
                                          }
                                      } );
                                }
                            }
                        }
                        else if ( layout.getComponent( 0 ) instanceof Label )
                        {
                            Object data = ( (Button) event.getComponent() ).getData();
                            if ( SharedUtil.isNotNullAndNotEmpty( data ) )
                            {
                                switch ( data.toString() )
                                {
                                    case BaseTable.DESCRIPTION_CRITERIA_LESS:
                                        sqlTabs.setMoreCriteriaRowsVisibled( false );
                                        break;
                                    case BaseTable.DESCRIPTION_CRITERIA_MORE:
                                        sqlTabs.setMoreCriteriaRowsVisibled( true );
                                        break;
                                    default:
                                        break;
                                }
                                layout.getComponent( 1 )
                                      .setEnabled( sqlTabs.getCriteriaMinPageLength() < sqlTabs.getCriteriaPageLength() );
                                layout.getComponent( 2 )
                                      .setEnabled( sqlTabs.getCriteriaPageLength() < sqlTabs.getCriteriaMaxPageLength() );
                            }
                        }

                        // enable save query
                        sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                        // enable/disable function on group of fields
                        sqlMenu.setGroupButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            sqlTabs.addDropHandler( new DropHandler()
            {
                private static final long serialVersionUID = 876785811093238175L;

                @Override
                public AcceptCriterion getAcceptCriterion()
                {
                    return AcceptAll.get();
                }

                @Override
                public void drop( DragAndDropEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getTransferable() )
                        && event.getTransferable() instanceof DataBoundTransferable
                        && SharedUtil.isNotNull( event.getTargetDetails() )
                        && SharedUtil.isNotNull( event.getTargetDetails()
                                                      .getTarget() ) )
                    {
                        DataBoundTransferable draggableObject = (DataBoundTransferable) event.getTransferable();
                        DropTarget droppedObject = event.getTargetDetails()
                                                        .getTarget();
                        if ( draggableObject instanceof TableTransferable
                            && droppedObject instanceof DragAndDropWrapper )
                        {
                            Table draggableTable = (Table) draggableObject.getSourceComponent();
                            Object draggableItemId = draggableObject.getItemId();
                            Object droppedData = ( (DragAndDropWrapper) droppedObject ).getData();
                            if ( SharedUtil.isNotNull( draggableTable )
                                && SharedUtil.isNotNullAndNotEmpty( draggableItemId )
                                && SharedUtil.isNotNullAndNotEmpty( droppedData ) )
                            {
                                String columnId = droppedData.toString()
                                                             .split( StringUtil.STRING_DOT_SPLITTER )[0];
                                String[] itemIds = StringUtil.asStringArray( draggableTable.getItemIds()
                                                                                           .toArray() );
                                for ( String itemId : itemIds )
                                {
                                    if ( SharedUtil.isNotNullAndNotEmpty( itemId )
                                        && ( draggableTable.isSelected( itemId ) || itemId.equals( draggableItemId ) ) )
                                    {

                                        String[] itemData = itemId.split( StringUtil.STRING_DOT_SPLITTER );

                                        // insert new field column
                                        // format: database source id, database name, table name, field name, table
                                        // alias
                                        String newColumnId =
                                            sqlTabs.addFieldColumn( itemData[4], itemData[3], itemData[3], true );

                                        if ( SharedUtil.isNotNullAndNotEmpty( newColumnId ) )
                                        {
                                            if ( !newColumnId.equals( columnId ) )
                                            {
                                                if ( !sqlTabs.isFieldColumnEmpty( columnId ) )
                                                {
                                                    sqlTabs.addFieldColumnBefore( newColumnId, columnId );
                                                }
                                                else
                                                {
                                                    sqlTabs.reverseFieldColumns( newColumnId, columnId );
                                                }
                                            }
                                            sqlTabs.addMissingFieldColumns();
                                            columnId = sqlTabs.getNextFieldColumnId( newColumnId );

                                            // unselect row in draggable table
                                            draggableTable.unselect( itemId );
                                        }
                                        else
                                        {
                                            new ErrorUtil( ErrorUtil.CANNOT_ADD_NEW_COLUMN ).show();
                                            break;
                                        }
                                    }
                                }
                            }
                        }

                        // enable clear table container
                        sqlMenu.setClearButtonEnabled( true );

                        // enable save query
                        sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                        // enable/disable function on group of fields
                        sqlMenu.setGroupButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            sqlTabs.addFocusListener( new FocusListener()
            {
                private static final long serialVersionUID = -4220158327597345370L;

                @Override
                public void focus( FocusEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof ComboBox )
                    {
                        ( (ComboBox) event.getComponent() ).setDescription( null );
                    }
                }
            } );

            sqlTabs.addTextChangeListener( new TextChangeListener()
            {
                private static final long serialVersionUID = 3493664093363280359L;

                @Override
                public void textChange( TextChangeEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getComponent() ) && SharedUtil.isNotNull( event.getComponent()
                                                                                                    .getParent() )
                        && event.getComponent()
                                .getParent() instanceof HorizontalLayout )
                    {

                        HorizontalLayout parent = (HorizontalLayout) event.getComponent()
                                                                          .getParent();
                        if ( parent.getComponentCount() == 2 && parent.getComponent( 1 ) instanceof Button )
                        {
                            Button button = (Button) parent.getComponent( 1 );
                            button.setVisible( SharedUtil.isNotNullAndNotEmpty( StringUtil.trim( event.getText() ) ) );
                            if ( button.isVisible() && SharedUtil.isNotNullAndNotEmpty( button.getData() ) )
                            {
                                String columnId = button.getData()
                                                        .toString();

                                sqlTabs.setFieldColumnEnabled( columnId, true );

                                // combo boxes for common functions, total functions and sorting
                                String commonValue = sqlTabs.getHeaderFieldsTable()
                                                            .getCellValue( columnId, BaseTable.DESCRIPTION_COMMON );
                                String sortValue = sqlTabs.getHeaderFieldsTable()
                                                          .getCellValue( columnId, BaseTable.DESCRIPTION_SORTING );
                                String totalValue = sqlTabs.getHeaderFieldsTable()
                                                           .getCellValue( columnId, BaseTable.DESCRIPTION_TOTAL );
                                if ( SharedUtil.isNull( commonValue ) && SharedUtil.isNull( sortValue )
                                    && SharedUtil.isNull( totalValue ) )
                                {
                                    // add values to common combo box
                                    sqlTabs.getHeaderFieldsTable()
                                           .setCellValues( columnId, BaseTable.DESCRIPTION_COMMON,
                                                           sqlTabs.getCommonFunctions()
                                                                  .getFunctions() );

                                    // add values to sorting combo box
                                    sqlTabs.getHeaderFieldsTable()
                                           .setCellValues( columnId, BaseTable.DESCRIPTION_SORTING,
                                                           sqlTabs.getSortingFunctions()
                                                                  .getFunctions() );

                                    // add values to total combo box
                                    sqlTabs.getHeaderFieldsTable()
                                           .setCellValues( columnId, BaseTable.DESCRIPTION_TOTAL,
                                                           sqlTabs.getTotalFunctions()
                                                                  .getFunctions() );

                                    // enable to show
                                    sqlTabs.getHeaderFieldsTable()
                                           .setCellValue( columnId, BaseTable.DESCRIPTION_SHOW,
                                                          Boolean.TRUE.toString() );
                                }
                            }

                            // enable save query
                            sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );
                        }

                        // refresh sql editor
                        refreshSqlEditorContext();
                    }
                }
            } );

            sqlTabs.addValueChangeListener( new ValueChangeListener()
            {
                private static final long serialVersionUID = -3939080926028623701L;

                @Override
                public void valueChange( ValueChangeEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getProperty() ) && event.getProperty() instanceof ComboBox )
                    {
                        ComboBox comboBox = (ComboBox) event.getProperty();
                        Object data = comboBox.getData();
                        Object value = comboBox.getValue();
                        if ( SharedUtil.isNotNullAndNotEmpty( data ) && SharedUtil.isNotNullAndNotEmpty( value ) )
                        {
                            // enable button for setting function's parameter values
                            String description = data.toString();
                            if ( description.equals( BaseTable.DESCRIPTION_COMMON )
                                && SharedUtil.isNotNull( comboBox.getParent() )
                                && comboBox.getParent() instanceof HorizontalLayout )
                            {
                                HorizontalLayout layout = (HorizontalLayout) comboBox.getParent();
                                if ( layout.getComponentCount() == 2 && layout.getComponent( 1 ) instanceof Button )
                                {
                                    Button button = (Button) layout.getComponent( 1 );
                                    if ( SharedUtil.isNotNull( button.getData() ) )
                                    {
                                        button.setEnabled( sqlTabs.getCommonFunctions()
                                                                  .hasParameters( value.toString() ) );
                                    }
                                }
                            }
                        }
                    }

                    // enable save query
                    sqlMenu.setSaveButtonEnabled( !sqlTabs.isAllFieldColumnsEmpty() );

                    // refresh sql editor
                    refreshSqlEditorContext();
                }
            } );

            sqlTabs.prepare();
        }

        /**
         * Context preparation.
         */
        private void prepare()
        {
            // prepare context components
            prepareDataSourcesMenu();
            prepareDroppedTableContainer();
            prepareSqlMenu();
            prepareSqlTabs();

            // context layout
            setTemplateContents( HTML );
            addComponent( dataSourcesMenu, TABLES );
            addComponent( droppedTableContainer, CONTENT );
            addComponent( sqlMenu, SQL_MENU );
            addComponent( sqlTabs, FIELDS );

            // visibility of header rows
            setHeaderRowVisible( BaseTable.DESCRIPTION_COMMON, false );
            setHeaderRowVisible( BaseTable.DESCRIPTION_SORTING, true );
            setHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL, false );
        }
    }

}
