/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.ui;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.dussan.vaadin.dquery.enums.CellType;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.data.Item;
import com.vaadin.data.Property;
import com.vaadin.event.FieldEvents.BlurListener;
import com.vaadin.event.FieldEvents.FocusListener;
import com.vaadin.event.FieldEvents.TextChangeListener;
import com.vaadin.event.dd.DropHandler;
import com.vaadin.server.FontAwesome;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.CheckBox;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.Component;
import com.vaadin.ui.DragAndDropWrapper;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Table;
import com.vaadin.ui.TextField;
import com.vaadin.ui.themes.ValoTheme;

/**
 * Base table instance.
 */
public class BaseTable
    extends Table
{

    private static final long serialVersionUID = 617187403643030713L;

    private static final String BUTTON = "-button";

    private static final String BUTTON_REMOVE_COLUMN = "Remove column";

    private static final String BUTTON_CHANGE_FUNCTION_PARAMETERS = "Change the function parameters";

    private static final String CRITERIA_DESCRIPTION = "criteria-description";

    private static final String CRITERIA_DESCRIPTION_BUTTON = CRITERIA_DESCRIPTION + BUTTON;

    private static final String CRITERIA_DESCRIPTION_LABEL = CRITERIA_DESCRIPTION + "-label";

    private static final String FIELD_NEW = "NEW FIELD";

    private static final String HEADER_FIELD = "header-field";

    private static final String HEADER_FIELD_LABEL = HEADER_FIELD + "-label";

    private static final String HEADER_FIELD_BUTTON = HEADER_FIELD + BUTTON;

    private static final String HEADER_PARAMETER = "parameter-field";

    private static final String HEADER_PARAMETER_BUTTON = HEADER_PARAMETER + BUTTON;

    private static final String HEADER_PARAMETER_COMBO = HEADER_PARAMETER + "-combo";

    private static final String TABLE_DESCRIPTION = "table-description";

    private static final String TABLE_SOURCE = "table-source";

    private static final int COLUMN_LAST_WIDTH = 100;

    public static final int DESCRIPTION_COMMON_ID = 3;

    public static final int DESCRIPTION_SORTING_ID = 5;

    public static final int DESCRIPTION_TOTAL_ID = 4;

    public static final String COLUMN_DESCRIPTION = "__DESCRIPTION_COLUMN__";

    public static final String COLUMN_LAST_FINAL = "__LAST_COLUMN_FINAL__";

    public static final String COLUMN_NAME = "COLUMN_NAME";

    public static final String DESCRIPTION_COMMON = "COMMON";

    public static final String DESCRIPTION_CRITERIA = "CRITERIA";

    public static final String DESCRIPTION_CRITERIA_OR = "OR";

    public static final String DESCRIPTION_CRITERIA_LESS = "Show less criteria rows";

    public static final String DESCRIPTION_CRITERIA_MORE = "Show more criteria rows";

    public static final String DESCRIPTION_FIELD = "FIELD";

    public static final String DESCRIPTION_FIELD_STYLE = "field-description";

    public static final String DESCRIPTION_NAME = "NAME";

    public static final String DESCRIPTION_SHOW = "SHOW";

    public static final String DESCRIPTION_SORTING = "SORTING";

    public static final String DESCRIPTION_TABLE = "TABLE";

    public static final String DESCRIPTION_TOTAL = "TOTAL";

    public static final String FIELDS_ALL = "*";

    public static final String HEADER_NAME = "{0}{1}";

    public static final String TABLE_NAME = "{0}.{1}";

    public static final String TABLE_NO_SCROLL = "no-scroll";

    public static final String TABLE_ROW_FIELD = "field";

    public static final String TABLE_ROW_FIELD_HIDDEN = TABLE_ROW_FIELD + "-hidden";

    public static final String TABLE_ROW_FIELD_ODD = TABLE_ROW_FIELD + "-odd";

    private int columns = 0;

    private int columnWidth = 0;

    private BlurListener blurListener = null;

    private ClickListener clickListener = null;

    private DropHandler dropHandler = null;

    private FocusListener focusListener = null;

    private TextChangeListener textChangeListener = null;

    private ValueChangeListener valueChangeListener = null;

    /**
     * Creates a new instance.
     */
    public BaseTable()
    {
        // Auto-generated constructor stub
    }

    /**
     * Creates a new empty base table with or without description column.
     * 
     * @param hasDescriptions
     *            specifies when table include description column
     */
    public BaseTable( boolean hasDescriptions )
    {
        addStyleName( ValoTheme.TABLE_NO_HEADER );
        addStyleName( ValoTheme.TABLE_NO_HORIZONTAL_LINES );
        addStyleName( ValoTheme.TABLE_BORDERLESS );
        addStyleName( ValoTheme.TABLE_COMPACT );
        addStyleName( ValoTheme.TABLE_SMALL );
        addStyleName( TABLE_SOURCE );
        setCacheRate( 100 );
        setColumnCollapsingAllowed( true );
        setImmediate( true );

        // cell style for description table
        if ( hasDescriptions )
        {
            removeStyleName( TABLE_SOURCE );
            addStyleName( TABLE_DESCRIPTION );
            setCellStyleGenerator( new CellStyleGenerator()
            {
                private static final long serialVersionUID = -4818757050175929609L;

                public String getStyle( Table source, Object itemId, Object propertyId )
                {
                    if ( SharedUtil.isNotNull( propertyId ) )
                    {
                        return DESCRIPTION_FIELD_STYLE;
                    }
                    return null;
                }
            } );
        }
    }

    /**
     * Add text field blur listener.
     * 
     * @param blurListener
     *            text field blur listener
     */
    public void addBlurListener( BlurListener blurListener )
    {
        this.blurListener = blurListener;
    }

    /**
     * Add button click listener.
     * 
     * @param clickListener
     *            button click listener
     */
    public void addClickListener( Button.ClickListener clickListener )
    {
        this.clickListener = clickListener;
    }

    /**
     * Add drop handler.
     * 
     * @param dropHandler
     *            drop handler
     */
    public void addDropHandler( DropHandler dropHandler )
    {
        this.dropHandler = dropHandler;
    }

    /**
     * Add text change listener.
     * 
     * @param textChangeListener
     *            text change listener
     */
    public void addTextChangeListener( TextChangeListener textChangeListener )
    {
        this.textChangeListener = textChangeListener;
    }

    public void addValueChangeListener( ValueChangeListener valueChangeListener )
    {
        this.valueChangeListener = valueChangeListener;
    }

    /**
     * Add focus listener.
     * 
     * @param focusListener
     *            value change listener
     */
    public void addFocusListener( FocusListener focusListener )
    {
        this.focusListener = focusListener;
    }

    /**
     * Get number of visible columns.
     * 
     * @return number of visible columns.
     */
    public int getColumns()
    {
        return columns;
    }

    /**
     * Set number of visible columns.
     * 
     * @param columns
     *            number of visible columns
     */
    public void setColumns( int columns )
    {
        this.columns = columns;
    }

    /**
     * Set column width.
     * 
     * @param columnWidth
     *            column width
     */
    public void setColumnWidth( int columnWidth )
    {
        this.columnWidth = columnWidth;
    }

    /**
     * Get cell component.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @return cell component
     */
    @SuppressWarnings( "rawtypes" )
    public Component getCellComponent( String columnId, String rowId )
    {
        Item rowItem = getItem( rowId );
        if ( SharedUtil.isNotNull( rowItem ) )
        {
            Property cellProperty = rowItem.getItemProperty( columnId );
            if ( SharedUtil.isNotNull( cellProperty ) )
            {
                return (Component) cellProperty.getValue();
            }
        }
        return null;
    }

    /**
     * Set cell component.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param cellType
     *            cell type
     */
    @SuppressWarnings( { "unchecked", "rawtypes" } )
    public void setCellComponent( String columnId, String rowId, CellType cellType )
    {
        Item rowItem = getItem( rowId );
        if ( SharedUtil.isNotNull( rowItem ) )
        {
            Property cellProperty = rowItem.getItemProperty( columnId );
            if ( SharedUtil.isNotNull( cellProperty ) )
            {
                Component cellComponent = null;
                switch ( cellType )
                {
                    case CHECK_BOX:
                        cellComponent = new CheckBox( null, false );
                        cellComponent.setEnabled( false );
                        cellComponent.addStyleName( ValoTheme.CHECKBOX_SMALL );
                        // event listeners
                        if ( SharedUtil.isNotNull( valueChangeListener ) )
                        {
                            ( (CheckBox) cellComponent ).addValueChangeListener( valueChangeListener );
                        }
                        break;
                    case COMBO_BOX:
                        cellComponent = new ComboBox( null );
                        ( (ComboBox) cellComponent ).setData( rowId );
                        ( (ComboBox) cellComponent ).setNewItemsAllowed( false );
                        ( (ComboBox) cellComponent ).setNullSelectionAllowed( false );
                        ( (ComboBox) cellComponent ).setPageLength( 0 );
                        ( (ComboBox) cellComponent ).setScrollToSelectedItem( true );
                        ( (ComboBox) cellComponent ).setTextInputAllowed( false );
                        cellComponent.setEnabled( false );
                        cellComponent.setWidth( 100, Unit.PERCENTAGE );
                        cellComponent.addStyleName( ValoTheme.COMBOBOX_BORDERLESS );
                        cellComponent.addStyleName( ValoTheme.COMBOBOX_TINY );
                        // event listeners
                        if ( SharedUtil.isNotNull( blurListener ) )
                        {
                            ( (ComboBox) cellComponent ).addBlurListener( blurListener );
                        }
                        if ( SharedUtil.isNotNull( focusListener ) )
                        {
                            ( (ComboBox) cellComponent ).addFocusListener( focusListener );
                        }
                        if ( SharedUtil.isNotNull( valueChangeListener ) )
                        {
                            ( (ComboBox) cellComponent ).addValueChangeListener( valueChangeListener );
                        }
                        break;
                    case COMBO_BOX_PARAMETER:
                        Button parametersButton = new Button();
                        parametersButton.setData( columnId );
                        parametersButton.setDescription( BUTTON_CHANGE_FUNCTION_PARAMETERS );
                        parametersButton.setEnabled( false );
                        parametersButton.setIcon( FontAwesome.DOT_CIRCLE_O );
                        parametersButton.addStyleName( HEADER_PARAMETER_BUTTON );
                        parametersButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                        parametersButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );

                        ComboBox comboBox = new ComboBox( null );
                        comboBox.setData( rowId );
                        comboBox.setNewItemsAllowed( false );
                        comboBox.setNullSelectionAllowed( false );
                        comboBox.setPageLength( 0 );
                        comboBox.setScrollToSelectedItem( true );
                        comboBox.setTextInputAllowed( false );
                        comboBox.setWidth( 100, Unit.PERCENTAGE );
                        comboBox.addStyleName( HEADER_PARAMETER_COMBO );
                        comboBox.addStyleName( ValoTheme.COMBOBOX_BORDERLESS );
                        comboBox.addStyleName( ValoTheme.COMBOBOX_TINY );

                        cellComponent = new HorizontalLayout( comboBox, parametersButton );
                        cellComponent.addStyleName( HEADER_PARAMETER );
                        cellComponent.addStyleName( ValoTheme.PANEL_BORDERLESS );
                        cellComponent.setEnabled( false );
                        cellComponent.setWidth( 100, Unit.PERCENTAGE );
                        ( (HorizontalLayout) cellComponent ).setExpandRatio( comboBox, 10 );

                        // event listeners
                        if ( SharedUtil.isNotNull( clickListener ) )
                        {
                            parametersButton.addClickListener( clickListener );
                        }
                        if ( SharedUtil.isNotNull( blurListener ) )
                        {
                            comboBox.addBlurListener( blurListener );
                        }
                        if ( SharedUtil.isNotNull( focusListener ) )
                        {
                            comboBox.addFocusListener( focusListener );
                        }
                        if ( SharedUtil.isNotNull( valueChangeListener ) )
                        {
                            comboBox.addValueChangeListener( valueChangeListener );
                        }
                        break;
                    case HEADER:
                        Button removeButton = new Button();
                        removeButton.setData( columnId );
                        removeButton.setDescription( BUTTON_REMOVE_COLUMN );
                        removeButton.setIcon( FontAwesome.REMOVE );
                        removeButton.setVisible( false );
                        removeButton.addStyleName( HEADER_FIELD_BUTTON );
                        removeButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                        removeButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );

                        TextField textField = new TextField( StringUtil.EMPTY_VALUE );
                        textField.setCaption( null );
                        textField.setInputPrompt( FIELD_NEW );
                        textField.setWidth( 100, Unit.PERCENTAGE );
                        textField.addStyleName( HEADER_FIELD_LABEL );
                        textField.addStyleName( ValoTheme.TEXTFIELD_BORDERLESS );
                        textField.addStyleName( ValoTheme.TEXTFIELD_TINY );

                        HorizontalLayout headerLayout = new HorizontalLayout( textField, removeButton );
                        headerLayout.addStyleName( HEADER_FIELD );
                        headerLayout.addStyleName( ValoTheme.PANEL_BORDERLESS );
                        headerLayout.setWidth( 100, Unit.PERCENTAGE );
                        headerLayout.setExpandRatio( textField, 10 );

                        cellComponent = new DragAndDropWrapper( headerLayout );
                        cellComponent.setSizeFull();
                        ( (DragAndDropWrapper) cellComponent ).setData( MessageFormat.format( TABLE_NAME, columnId,
                                                                                              rowId ) );
                        // event listeners
                        if ( SharedUtil.isNotNull( clickListener ) )
                        {
                            removeButton.addClickListener( clickListener );
                        }
                        if ( SharedUtil.isNotNull( blurListener ) )
                        {
                            textField.addBlurListener( blurListener );
                        }
                        if ( SharedUtil.isNotNull( textChangeListener ) )
                        {
                            textField.addTextChangeListener( textChangeListener );
                        }
                        if ( SharedUtil.isNotNull( dropHandler ) )
                        {
                            ( (DragAndDropWrapper) cellComponent ).setDropHandler( dropHandler );
                        }
                        break;
                    case LABEL:
                    case LABEL_RIGHT_ALIGNED:
                        cellComponent = new Label( StringUtil.EMPTY_VALUE );
                        cellComponent.setEnabled( false );
                        cellComponent.addStyleName( ValoTheme.LABEL_BOLD );
                        cellComponent.addStyleName( ValoTheme.LABEL_TINY );
                        if ( cellType == CellType.LABEL_RIGHT_ALIGNED )
                        {
                            cellComponent.addStyleName( ValoTheme.TEXTAREA_ALIGN_RIGHT );
                        }
                        break;
                    case LABEL_WITH_BUTTONS:
                        Button downButton = new Button();
                        downButton.setData( DESCRIPTION_CRITERIA_LESS );
                        downButton.setDescription( DESCRIPTION_CRITERIA_LESS );
                        downButton.setIcon( FontAwesome.CHEVRON_DOWN );
                        downButton.addStyleName( CRITERIA_DESCRIPTION_BUTTON );
                        downButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                        downButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
                        downButton.addStyleName( ValoTheme.BUTTON_TINY );

                        Button upButton = new Button();
                        upButton.setData( DESCRIPTION_CRITERIA_MORE );
                        upButton.setDescription( DESCRIPTION_CRITERIA_MORE );
                        upButton.setIcon( FontAwesome.CHEVRON_UP );
                        upButton.addStyleName( CRITERIA_DESCRIPTION_BUTTON );
                        upButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                        upButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
                        upButton.addStyleName( ValoTheme.BUTTON_TINY );

                        Label label = new Label( StringUtil.EMPTY_VALUE );
                        label.setEnabled( false );
                        label.addStyleName( CRITERIA_DESCRIPTION_LABEL );
                        label.addStyleName( ValoTheme.LABEL_BOLD );
                        label.addStyleName( ValoTheme.LABEL_TINY );

                        cellComponent = new HorizontalLayout( label, downButton, upButton );
                        cellComponent.addStyleName( CRITERIA_DESCRIPTION );
                        cellComponent.addStyleName( ValoTheme.PANEL_BORDERLESS );
                        cellComponent.setWidth( 100, Unit.PERCENTAGE );
                        ( (HorizontalLayout) cellComponent ).setExpandRatio( label, 10 );

                        // event listeners
                        if ( SharedUtil.isNotNull( clickListener ) )
                        {
                            downButton.addClickListener( clickListener );
                            upButton.addClickListener( clickListener );
                        }
                        break;
                    case TEXT_FIELD:
                        cellComponent = new TextField( StringUtil.EMPTY_VALUE );
                        cellComponent.setEnabled( false );
                        cellComponent.setSizeFull();
                        cellComponent.setHeightUndefined();
                        cellComponent.addStyleName( ValoTheme.TEXTFIELD_BORDERLESS );
                        cellComponent.addStyleName( ValoTheme.TEXTFIELD_TINY );
                        ( (TextField) cellComponent ).setData( rowId );
                        // event listeners
                        if ( SharedUtil.isNotNull( valueChangeListener ) )
                        {
                            ( (TextField) cellComponent ).addValueChangeListener( valueChangeListener );
                        }
                        if ( SharedUtil.isNotNull( focusListener ) )
                        {
                            ( (TextField) cellComponent ).addFocusListener( focusListener );
                        }
                        break;
                    default:
                        // unknown cell type
                        return;
                }
                cellProperty.setValue( cellComponent );
            }
        }
    }

    /**
     * Set cell component.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param component
     *            component object
     */
    private void setCellComponent( String columnId, String rowId, Component component )
    {
        CellType cellType = null;
        if ( component instanceof DragAndDropWrapper )
        {
            cellType = CellType.HEADER;
        }
        else if ( component instanceof CheckBox )
        {
            cellType = CellType.CHECK_BOX;
        }
        else if ( component instanceof ComboBox )
        {
            cellType = CellType.COMBO_BOX;
        }
        else if ( component instanceof HorizontalLayout )
        {
            HorizontalLayout layout = (HorizontalLayout) component;
            if ( layout.getComponent( 0 ) instanceof ComboBox )
            {
                cellType = CellType.COMBO_BOX_PARAMETER;
            }
            else if ( layout.getComponent( 0 ) instanceof Label )
            {
                cellType = CellType.LABEL_WITH_BUTTONS;
            }
        }
        else if ( component instanceof Label )
        {
            cellType = CellType.LABEL;
        }
        else if ( component instanceof TextField )
        {
            cellType = CellType.TEXT_FIELD;
        }
        else
        {
            // unknown cell type
            return;
        }
        setCellComponent( columnId, rowId, cellType );
    }

    /**
     * Set all column's cells with same component.
     * 
     * @param columnId
     *            column id
     */
    @SuppressWarnings( "rawtypes" )
    private void setColumnComponents( String columnId )
    {
        Collection rowIds = getItemIds();
        String formattedColumnId = getContainerPropertyIds().iterator()
                                                            .next()
                                                            .toString();
        for ( Iterator rowId = rowIds.iterator(); rowId.hasNext(); )
        {
            String idRow = rowId.next()
                                .toString();
            Item rowItem = getItem( idRow );
            Property cellProperty = rowItem.getItemProperty( formattedColumnId );
            if ( SharedUtil.isNotNull( cellProperty ) )
            {
                setCellComponent( columnId, idRow, (Component) cellProperty.getValue() );
            }
        }
    }

    /**
     * Get cell value.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @return cell value
     */
    public String getCellValue( String columnId, String rowId )
    {
        Component component = getCellComponent( columnId, rowId );
        if ( SharedUtil.isNotNull( component ) )
        {
            // get cell value object
            Object cellValue = null;
            if ( component instanceof CheckBox )
            {
                cellValue = ( (CheckBox) component ).getValue();
            }
            else if ( component instanceof ComboBox )
            {
                cellValue = ( (ComboBox) component ).getValue();
            }
            else if ( component instanceof DragAndDropWrapper )
            {
                DragAndDropWrapper wrapper = (DragAndDropWrapper) component;
                HorizontalLayout layout = (HorizontalLayout) wrapper.iterator()
                                                                    .next();
                TextField textField = (TextField) layout.getComponent( 0 );
                cellValue = textField.getValue();
            }
            else if ( component instanceof HorizontalLayout )
            {
                HorizontalLayout layout = (HorizontalLayout) component;
                if ( layout.getComponent( 0 ) instanceof ComboBox )
                {
                    cellValue = ( (ComboBox) layout.getComponent( 0 ) ).getValue();
                }
                else if ( layout.getComponent( 0 ) instanceof Label )
                {
                    cellValue = ( (Label) layout.getComponent( 0 ) ).getValue();
                }
            }
            else if ( component instanceof Label )
            {
                cellValue = ( (Label) component ).getValue();
            }
            else if ( component instanceof TextField )
            {
                cellValue = ( (TextField) component ).getValue();
            }

            // return cell value only if it's not null
            if ( SharedUtil.isNotNull( cellValue ) )
            {
                return StringUtil.trim( cellValue.toString() );
            }
        }
        return null;
    }

    /**
     * Set cell value.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param cellValue
     *            cell value
     */
    public void setCellValue( String columnId, String rowId, String value )
    {
        String cellValue = value;
        if ( SharedUtil.isNull( cellValue ) )
        {
            cellValue = StringUtil.EMPTY_VALUE;
        }

        Component component = getCellComponent( columnId, rowId );
        if ( SharedUtil.isNotNull( component ) )
        {
            cellValue = StringUtil.trim( cellValue );
            if ( component instanceof CheckBox )
            {
                ( (CheckBox) component ).setValue( Boolean.valueOf( cellValue ) );
            }
            else if ( component instanceof ComboBox )
            {
                ComboBox comboBox = (ComboBox) component;
                comboBox.addItems( cellValue );
                comboBox.setValue( cellValue );
                if ( SharedUtil.isNullOrEmpty( cellValue ) )
                {
                    comboBox.setDescription( null );
                }
            }
            else if ( component instanceof DragAndDropWrapper )
            {
                DragAndDropWrapper wrapper = (DragAndDropWrapper) component;
                HorizontalLayout layout = (HorizontalLayout) wrapper.iterator()
                                                                    .next();
                TextField textField = (TextField) layout.getComponent( 0 );
                Button button = (Button) layout.getComponent( 1 );
                textField.setValue( cellValue );
                button.setVisible( SharedUtil.isNotNullAndNotEmpty( textField.getValue() ) );
                setColumnEnabled( columnId, button.isVisible() );
            }
            else if ( component instanceof HorizontalLayout )
            {
                HorizontalLayout layout = (HorizontalLayout) component;
                if ( layout.getComponent( 0 ) instanceof ComboBox )
                {
                    ComboBox comboBox = (ComboBox) layout.getComponent( 0 );
                    comboBox.addItems( cellValue );
                    comboBox.setValue( cellValue );
                    if ( SharedUtil.isNullOrEmpty( cellValue ) )
                    {
                        comboBox.setDescription( null );
                    }
                }
                else if ( layout.getComponent( 0 ) instanceof Label )
                {
                    ( (Label) layout.getComponent( 0 ) ).setValue( cellValue );
                }
            }
            else if ( component instanceof Label )
            {
                ( (Label) component ).setValue( cellValue );
            }
            else if ( component instanceof TextField )
            {
                ( (TextField) component ).setValue( cellValue );
            }
        }
    }

    /**
     * Set cell values.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param value
     *            cell values
     */
    public void setCellValues( String columnId, String rowId, String... values )
    {
        if ( SharedUtil.isNotNull( values ) )
        {
            for ( String value : values )
            {
                setCellValue( columnId, rowId, value );
            }
            setCellValue( columnId, rowId, values[0] );
        }
    }

    /**
     * Set cell description (tooltip).
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param cellValue
     *            description for cell
     */
    public void setCellDescription( String columnId, String rowId, String description )
    {
        String cellValue = description;
        if ( SharedUtil.isNull( cellValue ) )
        {
            cellValue = StringUtil.EMPTY_VALUE;
        }

        Component component = getCellComponent( columnId, rowId );
        if ( SharedUtil.isNotNull( component ) )
        {
            if ( component instanceof CheckBox )
            {
                ( (CheckBox) component ).setDescription( cellValue );
            }
            else if ( component instanceof ComboBox )
            {
                ( (ComboBox) component ).setDescription( cellValue );
            }
            else if ( component instanceof DragAndDropWrapper )
            {
                DragAndDropWrapper wrapper = (DragAndDropWrapper) component;
                HorizontalLayout layout = (HorizontalLayout) wrapper.iterator()
                                                                    .next();
                ( (TextField) layout.getComponent( 0 ) ).setDescription( cellValue );
            }
            else if ( component instanceof HorizontalLayout )
            {
                HorizontalLayout layout = (HorizontalLayout) component;
                ( (ComboBox) layout.getComponent( 0 ) ).setDescription( cellValue );
            }
            else if ( component instanceof Label )
            {
                ( (Label) component ).setDescription( cellValue );
            }
            else if ( component instanceof TextField )
            {
                ( (TextField) component ).setDescription( cellValue );
            }
        }
    }

    /**
     * Reset cell content.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     */
    public void resetCell( String columnId, String rowId )
    {
        setCellComponent( columnId, rowId, getCellComponent( columnId, rowId ) );
        setCellDescription( columnId, rowId, StringUtil.EMPTY_VALUE );
        getCellComponent( columnId, rowId ).setEnabled( true );
    }

    /**
     * <b>field column name:</b> column name + '_' + index value
     * <p>
     * If have field columns with same name, index present the max value of index values which are prepend to column
     * names, increased by 1. Index with value 0 is not append to column name.
     * 
     * @param name
     *            column name
     * @return the new index value
     */
    public String getSameNameIndex( String name )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( name ) && !name.equals( FIELDS_ALL ) )
        {
            // collect field header names
            List<String> names = new ArrayList<>();
            for ( Iterator<?> iterator = getContainerPropertyIds().iterator(); iterator.hasNext(); )
            {
                String headerId = iterator.next()
                                          .toString();
                String headerName = getCellValue( headerId, BaseTable.DESCRIPTION_NAME );
                if ( SharedUtil.isNotNullAndNotEmpty( headerName ) )
                {
                    names.add( headerName );
                }
            }

            // search/count same field header names
            int index = 0;
            String formattedName = name;
            while ( names.contains( formattedName ) )
            {
                formattedName = MessageFormat.format( BaseTable.HEADER_NAME, name,
                                                      MessageFormat.format( StringUtil.COUNTER_STRING, ++index ) );
            }

            if ( index != 0 )
            {
                return MessageFormat.format( StringUtil.COUNTER_STRING, index );
            }
        }
        return StringUtil.EMPTY_VALUE;
    }

    /**
     * Check if field column has no values in rows: name, field or table.
     * 
     * @param columnId
     *            column id
     * @return true if column has no values; false otherwise
     */
    public boolean isColumnEmpty( String columnId )
    {
        if ( !COLUMN_LAST_FINAL.equals( columnId ) )
        {
            return SharedUtil.isNullOrEmpty( getCellValue( columnId, DESCRIPTION_NAME ) )
                && SharedUtil.isNullOrEmpty( getCellValue( columnId, DESCRIPTION_FIELD ) )
                && SharedUtil.isNullOrEmpty( getCellValue( columnId, DESCRIPTION_TABLE ) );
        }
        return false;
    }

    /**
     * Add new empty column.
     * 
     * @param columnId
     *            column id
     */
    public void addEmptyColumn( String columnId )
    {
        addContainerProperty( columnId, Component.class, null, null, null, Align.CENTER );
        // move final column to end of the table
        if ( getContainerPropertyIds().contains( COLUMN_LAST_FINAL ) )
        {
            removeContainerProperty( COLUMN_LAST_FINAL );
            addContainerProperty( COLUMN_LAST_FINAL, Component.class, null );
            setColumnWidth( COLUMN_LAST_FINAL, COLUMN_LAST_WIDTH );
        }
    }

    /**
     * Add new column with default components.
     * 
     * @param columnId
     *            column id
     */
    public void addColumn( String columnId )
    {
        addEmptyColumn( columnId );
        setColumnComponents( columnId );
        setColumnWidth( columnId, columnWidth );
    }

    /**
     * Add new column to field's table with values from database and table definitions.
     * 
     * @param databaseName
     *            database name
     * @param tableName
     *            table name
     * @param fieldName
     *            field name
     * @param name
     *            column name
     * @param show
     *            if true the new column will be included in generated sql clause
     * @return the column id of newly added column
     */
    public String addColumn( String tableName, String fieldName, String name, boolean show )
    {
        List<Object> columnIds = new ArrayList<>( Arrays.asList( getVisibleColumns() ) );
        Collections.reverse( columnIds );
        columnIds.remove( 0 );
        for ( Object columnId : columnIds )
        {
            String id = columnId.toString();
            if ( !isColumnCollapsed( columnId ) && isColumnEmpty( id ) )
            {
                setColumnData( id, BaseTable.DESCRIPTION_NAME,
                               MessageFormat.format( BaseTable.HEADER_NAME, name, getSameNameIndex( name ) ),
                               BaseTable.DESCRIPTION_FIELD, fieldName, BaseTable.DESCRIPTION_TABLE, tableName,
                               BaseTable.DESCRIPTION_SHOW, Boolean.toString( show ) );
                setColumnCollapsed( id, false );
                setColumnEnabled( id, true );
                return id;
            }
        }
        return null;
    }

    /**
     * Remove column.
     * 
     * @param columnId
     *            column id
     */
    public void removeColumn( String columnId )
    {
        if ( 0 < columns && getContainerPropertyIds().contains( columnId ) )
        {
            setColumnEnabled( columnId, false );
            List<Object> visibleColumns = new ArrayList<>( Arrays.asList( getVisibleColumns() ) );
            visibleColumns.remove( columnId );
            visibleColumns.add( columns - 1, columnId );
            setVisibleColumns( visibleColumns.toArray() );
        }
    }

    /**
     * Move column to new location.
     * 
     * @param columnId
     *            column which will be moved to new location
     * @param insertBeforeColumnId
     *            column on which places will be inserted new column
     */
    public void moveColumn( String columnId, String insertBeforeColumnId )
    {
        if ( getContainerPropertyIds().contains( columnId )
            && getContainerPropertyIds().contains( insertBeforeColumnId ) )
        {
            List<Object> visibleColumns = new ArrayList<>( Arrays.asList( getVisibleColumns() ) );
            visibleColumns.remove( visibleColumns.indexOf( columnId ) );
            visibleColumns.add( visibleColumns.indexOf( insertBeforeColumnId ), columnId );
            setVisibleColumns( visibleColumns.toArray() );
        }
    }

    /**
     * Reverse column with other column.
     * 
     * @param columnId
     *            column which will be moved to new location
     * @param insertBeforeColumnId
     *            column on which places will be inserted new column
     */
    public void reverseColumns( String columnId, String reversedColumnId )
    {
        if ( getContainerPropertyIds().contains( columnId ) && getContainerPropertyIds().contains( reversedColumnId ) )
        {
            List<Object> visibleColumns = new ArrayList<>( Arrays.asList( getVisibleColumns() ) );
            int reversedIndex = visibleColumns.indexOf( reversedColumnId );
            visibleColumns.remove( visibleColumns.indexOf( reversedColumnId ) );
            visibleColumns.add( visibleColumns.indexOf( columnId ), reversedColumnId );
            visibleColumns.remove( visibleColumns.indexOf( columnId ) );
            visibleColumns.add( reversedIndex, columnId );
            setVisibleColumns( visibleColumns.toArray() );
        }
    }

    /**
     * Enable or disable all components in field column for interaction.
     * 
     * @param columnId
     *            column id
     * @param enabled
     *            if true component is enable for interaction, false otherwise
     */
    public void setColumnEnabled( String columnId, boolean enabled )
    {
        for ( Object rowId : getItemIds() )
        {
            // clear cell value if column is disabled
            if ( !enabled )
            {
                resetCell( columnId, rowId.toString() );
            }

            // enable/disable cell
            if ( !BaseTable.DESCRIPTION_NAME.equals( rowId ) )
            {
                ( (Component) getItem( rowId ).getItemProperty( columnId )
                                              .getValue() ).setEnabled( enabled );
            }
        }
    }

    /**
     * Set column new data with specified id.
     * 
     * @param columnId
     *            id of column to which is data added
     * @param columnData
     *            column data in pair: row id with cell value
     */
    public void setColumnData( String columnId, String... columnData )
    {
        for ( int index = 0; index < columnData.length; index += 2 )
        {
            if ( ( index + 1 ) < columnData.length && SharedUtil.isNotNullAndNotEmpty( columnData[index] )
                && SharedUtil.isNotNullAndNotEmpty( columnData[index + 1] ) )
            {
                setCellValue( columnId, columnData[index], columnData[index + 1] );
            }
        }
    }

    /**
     * Add empty row.
     * 
     * @param rowId
     *            row id
     */
    public void addEmptyRow( String rowId )
    {
        addItem( rowId );
    }

    /**
     * Remove row.
     * 
     * @param rowId
     *            row id
     */
    public void removeRow( String rowId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( rowId ) )
        {
            removeItem( rowId );
        }
    }

}
