/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.base.ui.DroppedTable;
import org.dussan.vaadin.dquery.base.ui.JoinedTables;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.ui.AbsoluteLayout;
import com.vaadin.ui.DragAndDropWrapper;

public class DroppedTableContainer
    extends DragAndDropWrapper
{

    private static final long serialVersionUID = -7565468160234691704L;

    public static final String TABLE_DROP = "table-drop";

    public static final String TABLE_DROP_CAPTION = "v-panel-caption";

    public static final String TABLE_DROP_BUTTON = TABLE_DROP + "-button";

    public static final String TABLE_DROP_BUTTON_LAST = TABLE_DROP_BUTTON + "-last";

    public static final String TABLE_DROP_LABEL = TABLE_DROP + "-label";

    public static final String TABLE_DROP_LOCATION = TABLE_DROP + "-location";

    public static final String TABLE_POSITION = "left: {0}px; top: {1}px;";

    public static final String TABLE_POSITION_DATA = "{0}.{1}";

    public static final String TABLE_DATA_ID = "{0}.{1}.{2}.{3}.{4}";

    public static final String TABLE_ROW_ID = "{0}.{1}.{2}.{3}.{4}";

    public static final String TABLE_DATA_NAME = "{0}.{1}";

    public static final String TABLE_DROP_NAME = "{0}{1}";

    private AbsoluteLayout droppedTablesLayout = null;

    private List<JoinedTables> joinedTables = null;

    /**
     * Creates a new empty instance.
     */
    @SuppressWarnings( "deprecation" )
    public DroppedTableContainer()
    {
        // Auto-generated constructor stub
    }

    /**
     * Prepare the table container.
     */
    public void prepare()
    {
        // layout for drop tables
        droppedTablesLayout = new AbsoluteLayout();
        droppedTablesLayout.setSizeFull();
        droppedTablesLayout.addStyleName( TABLE_DROP_LOCATION );

        joinedTables = new ArrayList<>();
        setCompositionRoot( droppedTablesLayout );
        setSizeFull();
    }

    /**
     * Get the table container.
     * 
     * @return table container
     */
    public AbsoluteLayout getContainer()
    {
        return droppedTablesLayout;
    }

    /**
     * Remove joined table and all links to that table.
     * 
     * @param table
     *            table to remove
     */
    public void removeJoinedTables( String table )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( table ) )
        {
            List<JoinedTables> toRemove = new ArrayList<>();
            for ( JoinedTables joinedTable : joinedTables )
            {
                if ( joinedTable.getFromTable()
                                .equals( table )
                    || joinedTable.getToTable()
                                  .equals( table ) )
                {
                    toRemove.add( joinedTable );
                }
            }
            joinedTables.removeAll( toRemove );
        }
    }

    /**
     * Remove join between tables.
     * 
     * @param fromTable
     *            table name from which join will be removed
     * @param toTable
     *            table name from which join will be removed
     * @param joinedFieldsIndex
     *            index of joined fields
     */
    public void removeJoinedTablesJoin( String fromTable, String toTable, int joinedFieldsIndex )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( fromTable ) && SharedUtil.isNotNullAndNotEmpty( toTable ) )
        {
            for ( JoinedTables joinedTable : joinedTables )
            {
                if ( ( joinedTable.getFromTable()
                                  .equals( fromTable )
                    && joinedTable.getToTable()
                                  .equals( toTable ) )
                    || ( joinedTable.getFromTable()
                                    .equals( toTable )
                        && joinedTable.getToTable()
                                      .equals( fromTable ) ) )
                {
                    joinedTable.removeFromToFields( joinedFieldsIndex );
                    if ( joinedTable.getFromFields().length == 0 && joinedTable.getToFields().length == 0 )
                    {
                        joinedTables.remove( joinedTable );
                        break;
                    }
                }
            }
        }
    }

    /**
     * Get joined tables as string array.
     * 
     * @return joined tables as string array
     */
    public List<String> getJoinedTablesAsStringList()
    {
        List<String> list = new ArrayList<>();
        for ( JoinedTables joinedTable : joinedTables )
        {
            list.add( joinedTable.toString() );
        }
        return list;
    }

    /**
     * Get joined tables.
     * 
     * @return joined tables
     */
    public JoinedTables[] getJoinedTables()
    {
        return joinedTables.toArray( new JoinedTables[joinedTables.size()] );
    }

    /**
     * Get joined tables for specified table.
     * 
     * @param tableName
     *            table name for which search joined tables
     * @return joined tables for specified table
     */
    public JoinedTables[] getJoinedTables( String tableName )
    {
        List<JoinedTables> joins = new ArrayList<>();
        for ( JoinedTables joinedTable : joinedTables )
        {
            if ( joinedTable.getFromTable()
                            .equals( tableName )
                || joinedTable.getToTable()
                              .equals( tableName ) )
            {
                joins.add( joinedTable );
            }
        }
        return joins.toArray( new JoinedTables[joins.size()] );
    }

    /**
     * Set joined tables.
     * 
     * @param fromTable
     *            table name from which join will be established
     * @param fromField
     *            field name from which join will be established
     * @param toTable
     *            table name to which join will be established
     * @param toField
     *            field name to which join will be established
     * @param tablesJoin
     *            type of tables join
     */
    public void setJoinedTables( String fromTable, String fromField, String toTable, String toField,
                                 TablesJoin tablesJoin )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( fromTable ) && SharedUtil.isNotNullAndNotEmpty( toTable )
            && !fromTable.equals( toTable ) && SharedUtil.isNotNullAndNotEmpty( fromField )
            && SharedUtil.isNotNullAndNotEmpty( toField ) && !BaseTable.FIELDS_ALL.equals( fromField )
            && !BaseTable.FIELDS_ALL.equals( toField ) )
        {
            int joinedTablesIndex = -1;
            for ( JoinedTables joinedTable : joinedTables )
            {
                if ( ( joinedTable.getFromTable()
                                  .equals( fromTable )
                    && joinedTable.getToTable()
                                  .equals( toTable ) )
                    || ( joinedTable.getFromTable()
                                    .equals( toTable )
                        && joinedTable.getToTable()
                                      .equals( fromTable ) ) )
                {
                    joinedTablesIndex = joinedTables.indexOf( joinedTable );
                    break;
                }
            }

            if ( 0 <= joinedTablesIndex )
            {
                if ( joinedTables.get( joinedTablesIndex )
                                 .getFromTable()
                                 .equals( fromTable ) )
                {
                    joinedTables.get( joinedTablesIndex )
                                .setFromToFields( fromField, toField );
                }
                else
                {
                    joinedTables.get( joinedTablesIndex )
                                .setFromToFields( toField, fromField );
                }
            }
            else
            {
                joinedTables.add( new JoinedTables( fromTable, fromField, toTable, toField, tablesJoin ) );
            }
        }
    }

    /**
     * Set joined tables.
     * 
     * @param fromTable
     *            table name from which join will be established
     * @param toTable
     *            table name to which join will be established
     * @param tablesJoin
     *            type of tables join
     * @param fromToFields
     *            a pair of fields names which will be included in join
     */
    public void setJoinedTables( String fromTable, String toTable, TablesJoin tablesJoin, String... fromToFields )
    {
        JoinedTables joinedTable = new JoinedTables( fromTable, toTable, tablesJoin, fromToFields );
        for ( int index = 0; index < joinedTable.getFromFields().length; index++ )
        {
            setJoinedTables( joinedTable.getFromTable(), joinedTable.getFromFields()[index], joinedTable.getToTable(),
                             joinedTable.getToFields()[index], joinedTable.getTablesJoin() );
        }
    }

    /**
     * Set join type to joined tables.
     * 
     * @param fromTable
     *            table name from which join will be established
     * @param toTable
     *            table name to which join will be established
     * @param tablesJoin
     *            type of tables join
     */
    public void setJoinedTablesJoinType( String fromTable, String toTable, TablesJoin tablesJoin )
    {
        for ( JoinedTables joinedTable : joinedTables )
        {
            if ( ( joinedTable.getFromTable()
                              .equals( fromTable )
                && joinedTable.getToTable()
                              .equals( toTable ) )
                || ( joinedTable.getFromTable()
                                .equals( toTable )
                    && joinedTable.getToTable()
                                  .equals( fromTable ) ) )
            {
                joinedTable.setTablesJoin( tablesJoin );
                break;
            }
        }
    }

    /**
     * Add dropped table to the table container.
     * 
     * @param table
     *            dropped table
     * @param cssPosition
     *            position of the dropped table
     */
    public void addTable( DroppedTable table, String cssPosition )
    {
        if ( SharedUtil.isNotNull( table ) && SharedUtil.isNotNullAndNotEmpty( cssPosition ) )
        {
            droppedTablesLayout.addComponent( table, cssPosition );
        }
    }

    /**
     * Get index of tables with same name.
     * 
     * @param tableName
     *            table name
     * @return index of tables with same name
     */
    public String getSameNameIndex( String tableName )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( tableName ) )
        {
            List<String> names = new ArrayList<>();
            for ( Iterator<?> iterator = droppedTablesLayout.iterator(); iterator.hasNext(); )
            {
                DroppedTable table = (DroppedTable) iterator.next();
                String caption = table.getCaption();
                if ( SharedUtil.isNotNullAndNotEmpty( caption ) )
                {
                    names.add( caption );
                }
            }

            // search/count same field header names
            int index = 0;
            String formattedName = tableName;
            while ( names.contains( formattedName ) )
            {
                index += 1;
                formattedName = MessageFormat.format( TABLE_DROP_NAME, tableName,
                                                      MessageFormat.format( StringUtil.COUNTER_STRING, index ) );
            }

            if ( index != 0 )
            {
                return MessageFormat.format( StringUtil.COUNTER_STRING, index );
            }
        }
        return StringUtil.EMPTY_VALUE;
    }

}
