/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.sql.builder;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class SqlWhere
    implements Serializable
{

    private static final long serialVersionUID = -7856159055040351015L;

    private static final String AND = " AND ";

    private static final String AND_CONDITION = "{0} {1}";

    private static final String OR = " OR ";

    private static final String OR_CONDITION = "( {0} )";

    private transient Map<String, Map<String, String[]>> where = null;

    /**
     * Creates a new instance.
     */
    public SqlWhere()
    {
        where = new LinkedHashMap<>();
    }

    /**
     * Add where condition.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param columnValue
     *            column value
     * @param columnCondition
     *            column condition
     */
    public void addWhereCondition( String columnId, String rowId, String columnValue, String columnCondition )
    {
        Map<String, String[]> row = where.get( rowId );
        if ( SharedUtil.isNull( row ) )
        {
            row = new LinkedHashMap<>();
        }
        row.put( columnId, new String[] { columnValue, columnCondition } );
        where.put( rowId, row );
    }

    /**
     * Get sql string for specified where condition.
     */
    @Override
    public String toString()
    {
        if ( !where.isEmpty() )
        {
            StringBuilder rowsCondition = new StringBuilder();
            for ( Entry<String, Map<String, String[]>> rowEntry : where.entrySet() )
            {
                Map<String, String[]> row = rowEntry.getValue();
                StringBuilder rowCondition = new StringBuilder();
                for ( Entry<String, String[]> columnEntry : row.entrySet() )
                {
                    String[] column = columnEntry.getValue();
                    if ( SharedUtil.isNotNullAndNotEmpty( rowCondition ) )
                    {
                        rowCondition.append( AND );
                    }
                    rowCondition.append( MessageFormat.format( AND_CONDITION, column[0], column[1] ) );
                }

                if ( SharedUtil.isNotNullAndNotEmpty( rowsCondition ) )
                {
                    rowsCondition.append( OR );
                }
                rowsCondition.append( MessageFormat.format( OR_CONDITION, rowCondition ) );
            }
            return rowsCondition.toString();
        }
        return StringUtil.EMPTY_VALUE;
    }

}
