/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.sql.builder;

import java.io.Serializable;
import java.text.MessageFormat;

import org.dussan.vaadin.dquery.base.sql.functions.SqlFunctions;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.enums.SqlFunctionType;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class SqlColumn
    implements Serializable
{

    private static final long serialVersionUID = -5819700629280380287L;

    private static final String COLUMN = "{0}.{1}";

    private static final String FUNCTION_DEFAULT = "{0}( {1} )";

    private static final String FUNCTION_WITH_PARAMETERS = "{0}( {1}, {2} )";

    private static final String FUNCTION_WITHOUT_PARAMETERS = "{0}()";

    private static final String STRING = "{0} AS {1}";

    private String column = null;

    private String columnAlias = null;

    private String commonFunction = null;

    private String fromTable = null;

    private String orderBy = null;

    private String totalFunction = null;

    private SqlFunctionType commonFunctionType = null;

    private SqlFunctionType totalFunctionType = null;

    /**
     * Creates a new instance.
     */
    public SqlColumn()
    {
        // Auto-generated constructor stub
    }

    /**
     * Creates a new instance with column, column alias and table name.
     */
    public SqlColumn( String column, String columnAlias, String fromTable )
    {
        setColumn( column ).as( columnAlias )
                           .from( fromTable );
    }

    /**
     * Get column name.
     * 
     * @return column name
     */
    public String getColumn()
    {
        return column;
    }

    /**
     * Set column name.
     * 
     * @param column
     *            column name
     * @return
     */
    public SqlColumn setColumn( String column )
    {
        this.column = column;
        this.columnAlias = column;
        return this;
    }

    /**
     * Get column alias.
     * 
     * @return column alias
     */
    public String getColumnAlias()
    {
        return columnAlias;
    }

    /**
     * Set column alias.
     * 
     * @param columnAlias
     *            column alias
     */
    public void setColumnAlias( String columnAlias )
    {
        this.columnAlias = columnAlias;
    }

    /**
     * Set column alias.
     * 
     * @param columnAlias
     *            column alias
     * @return sql column object
     */
    public SqlColumn as( String columnAlias )
    {
        setColumnAlias( columnAlias );
        return this;
    }

    /**
     * Get table name to which column belongs.
     * 
     * @return table name to which column belongs
     */
    public String getFromTable()
    {
        return fromTable;
    }

    /**
     * Set table name to which column belongs.
     * 
     * @param fromTable
     *            table name to which column belongs
     */
    public void setFromTable( String fromTable )
    {
        this.fromTable = fromTable;
    }

    /**
     * Set table name to which column belongs.
     * 
     * @param fromTable
     *            table name to which column belongs
     * @return sql column object
     */
    public SqlColumn from( String fromTable )
    {
        setFromTable( fromTable );
        return this;
    }

    /**
     * Get common function.
     * 
     * @return common function
     */
    public String getCommonFunction()
    {
        return commonFunction;
    }

    /**
     * Set common function.
     * 
     * @param function
     *            common function
     * @param commonFunctionType
     *            common function type
     */
    public void setCommonFunction( String commonFunction, SqlFunctionType commonFunctionType )
    {
        this.commonFunction = commonFunction;
        this.commonFunctionType = commonFunctionType;
    }

    /**
     * Set common function.
     * 
     * @param function
     *            common function
     * @param commonFunctionType
     *            common function type
     * @return sql common object
     */
    public SqlColumn commonFunction( String commonFunction, SqlFunctionType commonFunctionType )
    {
        this.commonFunction = commonFunction;
        this.commonFunctionType = commonFunctionType;
        return this;
    }

    /**
     * Get common function type.
     * 
     * @return common function type
     */
    public SqlFunctionType getCommonFunctionType()
    {
        return commonFunctionType;
    }

    /**
     * Set common function type.
     * 
     * @param commonFunctionType
     *            common function type
     */
    public void setCommonFunctionType( SqlFunctionType commonFunctionType )
    {
        this.commonFunctionType = commonFunctionType;
    }

    /**
     * Get total function.
     * 
     * @return total function
     */
    public String getTotalFunction()
    {
        return totalFunction;
    }

    /**
     * Set total function.
     * 
     * @param function
     *            total function
     * @param totalFunctionType
     *            total function type
     */
    public void setTotalFunction( String totalFunction, SqlFunctionType totalFunctionType )
    {
        this.totalFunction = totalFunction;
        this.totalFunctionType = totalFunctionType;
    }

    /**
     * Set total function.
     * 
     * @param function
     *            total function
     * @param totalFunctionType
     *            total function type
     * @return sql column object
     */
    public SqlColumn totalFunction( String totalFunction, SqlFunctionType totalFunctionType )
    {
        this.totalFunction = totalFunction;
        this.totalFunctionType = totalFunctionType;
        return this;
    }

    /**
     * Get total function type.
     * 
     * @return total function type
     */
    public SqlFunctionType getTotalFunctionType()
    {
        return totalFunctionType;
    }

    /**
     * Set total function type.
     * 
     * @param totalFunctionType
     *            total function type
     */
    public void setTotalFunctionType( SqlFunctionType totalFunctionType )
    {
        this.totalFunctionType = totalFunctionType;
    }

    /**
     * Get order by.
     * 
     * @return order by
     */
    public String getOrderBy()
    {
        return orderBy;
    }

    /**
     * Set order by.
     * 
     * @param orderBy
     *            order by
     */
    public void setOrderBy( String orderBy )
    {
        this.orderBy = orderBy;
    }

    /**
     * Set order by.
     * 
     * @param orderBy
     *            order by
     * @return sql column object
     */
    public SqlColumn orderBy( String orderBy )
    {
        this.orderBy = orderBy;
        return this;
    }

    /**
     * Get sql string for specified column without alias.
     * 
     * @return sql string for specified column without alias
     */
    public String toShortString()
    {
        if ( SharedUtil.isNotNullAndNotEmpty( this.column )
            || ( SharedUtil.isNullOrEmpty( this.column ) ) && SharedUtil.isNotNullAndNotEmpty( commonFunction )
                && commonFunctionType.equals( SqlFunctionType.WITHOUT_PARAMETERS ) )
        {
            // column name
            String column = this.column;
            if ( SharedUtil.isNotNullAndNotEmpty( fromTable ) )
            {
                column = MessageFormat.format( COLUMN, fromTable, column );
            }

            if ( !this.column.equals( BaseTable.FIELDS_ALL ) )
            {
                // common function
                if ( SharedUtil.isNotNullAndNotEmpty( commonFunction ) )
                {
                    String functionName = commonFunction.split( SqlFunctions.LEFT_PARENTHESIS_ESCAPED )[0];
                    if ( commonFunctionType.equals( SqlFunctionType.WITHOUT_PARAMETERS ) )
                    {
                        column = MessageFormat.format( FUNCTION_WITHOUT_PARAMETERS, functionName );
                    }
                    else
                    {
                        String[] functionParameters =
                            commonFunction.split( SqlFunctions.LEFT_PARENTHESIS_ESCAPED )[1].split( SqlFunctions.RIGHT_PARENTHESIS_ESCAPED );
                        if ( functionParameters.length == 0 )
                        {
                            column = MessageFormat.format( FUNCTION_DEFAULT, functionName, column );
                        }
                        else
                        {
                            column = MessageFormat.format( FUNCTION_WITH_PARAMETERS, functionName, column,
                                                           StringUtil.trim( functionParameters[0] ) );
                        }
                    }
                }

                // total function
                if ( SharedUtil.isNotNullAndNotEmpty( totalFunction )
                    && !totalFunctionType.equals( SqlFunctionType.NULL ) )
                {
                    String functionName = totalFunction.split( SqlFunctions.LEFT_PARENTHESIS_ESCAPED )[0];
                    if ( totalFunctionType.equals( SqlFunctionType.WITHOUT_PARAMETERS ) )
                    {
                        column = MessageFormat.format( FUNCTION_DEFAULT, functionName, column );
                    }
                    else
                    {
                        String[] functionParameters =
                            totalFunction.split( SqlFunctions.LEFT_PARENTHESIS_ESCAPED )[1].split( SqlFunctions.RIGHT_PARENTHESIS_ESCAPED );
                        if ( functionParameters.length == 0 )
                        {
                            column = MessageFormat.format( FUNCTION_DEFAULT, functionName, column );
                        }
                        else
                        {
                            column = MessageFormat.format( FUNCTION_WITH_PARAMETERS, functionName, column,
                                                           StringUtil.trim( functionParameters[0] ) );
                        }
                    }
                }
            }

            // column with alias
            return StringUtil.trim( column );
        }
        return null;
    }

    /**
     * Get sql string for specified column.
     */
    @Override
    public String toString()
    {
        String shortString = toShortString();
        if ( SharedUtil.isNotNullAndNotEmpty( shortString ) )
        {
            if ( !this.column.equals( BaseTable.FIELDS_ALL ) )
            {
                // column with alias
                return StringUtil.trim( MessageFormat.format( STRING, shortString, columnAlias ) );
            }
            return StringUtil.trim( shortString );
        }
        return StringUtil.EMPTY_VALUE;
    }

}
