/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.sql.builder;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class SqlJoin
    implements Serializable
{

    private static final long serialVersionUID = -2722988674472388057L;

    private static final String AND = " AND ";

    private static final String JOIN = "{0}.{1}={2}.{3}";

    private static final String STRING = "{0} JOIN {1} AS {2}\n\tON ( {3} )";

    private SqlTable fromTable = null;

    private SqlTable toTable = null;

    private List<String> fromColumns = null;

    private List<String> toColumns = null;

    private TablesJoin tablesJoin = null;

    /**
     * Creates a new instance.
     */
    public SqlJoin()
    {
        fromColumns = new ArrayList<>();
        toColumns = new ArrayList<>();
        tablesJoin = TablesJoin.INNER;
    }

    /**
     * Creates a new instance with tables and their joins.
     */
    public SqlJoin( SqlTable fromTable, SqlTable toTable, TablesJoin tablesJoin )
    {
        this();
        this.fromTable = fromTable;
        this.toTable = toTable;
        this.tablesJoin = tablesJoin;
    }

    /**
     * Get join from table.
     * 
     * @return sql join object
     */
    public SqlTable getFromTable()
    {
        return fromTable;
    }

    /**
     * Set join from table.
     * 
     * @param fromTable
     *            from table name
     * @return sql join object
     */
    public SqlJoin setFromTable( SqlTable fromTable )
    {
        this.fromTable = fromTable;
        return this;
    }

    /**
     * Alias for from table.
     * 
     * @param fromTable
     *            from table name
     * @return sql join object
     */
    public SqlJoin from( SqlTable fromTable )
    {
        setFromTable( fromTable );
        return this;
    }

    /**
     * Get join to table.
     * 
     * @return sql join object
     */
    public SqlTable getToTable()
    {
        return toTable;
    }

    /**
     * Set join to table.
     * 
     * @param toTable
     * @return sql join object
     */
    public SqlJoin setToTable( SqlTable toTable )
    {
        this.toTable = toTable;
        return this;
    }

    /**
     * Alias for to table.
     * 
     * @param toTable
     * @return sql join object
     */
    public SqlJoin to( SqlTable toTable )
    {
        setToTable( toTable );
        return this;
    }

    /**
     * Get tables join.
     * 
     * @return tables join
     */
    public TablesJoin getTablesJoin()
    {
        return tablesJoin;
    }

    /**
     * Set tables join.
     * 
     * @param tablesJoin
     *            tables join
     * @return sql join object
     */
    public SqlJoin setTablesJoin( TablesJoin tablesJoin )
    {
        this.tablesJoin = tablesJoin;
        return this;
    }

    /**
     * Alias for set tables join.
     * 
     * @param tablesJoin
     *            tables join
     * @return sql join object
     */
    public SqlJoin join( TablesJoin tablesJoin )
    {
        setTablesJoin( tablesJoin );
        return this;
    }

    /**
     * @param fromColumn
     * @param toColumn
     * @return sql join object
     */
    public SqlJoin setFromToColumn( String fromColumn, String toColumn )
    {
        fromColumns.add( StringUtil.trim( fromColumn ) );
        toColumns.add( StringUtil.trim( toColumn ) );
        return this;
    }

    /**
     * Get sql string for specified join.
     */
    @Override
    public String toString()
    {
        StringBuilder fromToColumns = new StringBuilder();

        // create tables join
        for ( int index = 0; index < Math.min( fromColumns.size(), toColumns.size() ); index++ )
        {
            if ( !fromToColumns.toString()
                               .isEmpty() )
            {
                fromToColumns.append( AND );
            }
            fromToColumns.append( MessageFormat.format( JOIN, fromTable.getTableAlias(), fromColumns.get( index ),
                                                        toTable.getTableAlias(), toColumns.get( index ) ) );
        }

        // formated string for tables join
        if ( !fromToColumns.toString()
                           .isEmpty() )
        {
            return MessageFormat.format( STRING, tablesJoin.toString(), fromTable.getTableName(),
                                         fromTable.getTableAlias(), fromToColumns.toString() );
        }

        return null;
    }

}
