/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.ui;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class JoinedTables
    implements Serializable
{

    private static final long serialVersionUID = -8564778844383922563L;

    private static final String JOINED_TABLES = "{0}:{1}:{2}:{3}";

    private static final String JOINED_FIELDS = "{0}-{1}";

    private String fromTable = null;

    private String toTable = null;

    private transient List<String> fromFields = null;

    private transient List<String> toFields = null;

    private TablesJoin tablesJoin = null;

    /**
     * Creates a new instance and initialize all variables.
     */
    public JoinedTables()
    {
        fromTable = new String();
        toTable = new String();
        fromFields = new ArrayList<String>();
        toFields = new ArrayList<String>();
        tablesJoin = TablesJoin.INNER;
    }

    /**
     * Creates a new instance and set variables value.
     * 
     * @param fromTable
     *            table name from which join will be established
     * @param fromField
     *            field name from which join will be established
     * @param toTable
     *            table name to which join will be established
     * @param toField
     *            field name to which join will be established
     * @param tablesJoin
     *            type of tables join
     */
    public JoinedTables( String fromTable, String fromField, String toTable, String toField, TablesJoin tablesJoin )
    {
        this();
        this.fromTable = fromTable;
        this.fromFields.add( fromField );
        this.toTable = toTable;
        this.toFields.add( toField );
        this.tablesJoin = tablesJoin;
    }

    /**
     * Creates a new instance and set variables value.
     * 
     * @param fromTable
     *            table name from which join will be established
     * @param toTable
     *            table name to which join will be established
     * @param tablesJoin
     *            type of tables join
     * @param fromToFields
     *            a pair of fields names which will be included in join
     */
    public JoinedTables( String fromTable, String toTable, TablesJoin tablesJoin, String... fromToFields )
    {
        this();
        this.fromTable = fromTable;
        this.toTable = toTable;
        this.tablesJoin = tablesJoin;
        for ( int index = 0; index < fromToFields.length; index += 2 )
        {
            if ( ( index + 1 ) < fromToFields.length )
            {
                this.fromFields.add( fromToFields[index] );
                this.toFields.add( fromToFields[index + 1] );
            }
        }
    }

    /**
     * Get table name from which join will be established.
     * 
     * @return table name from which join will be established
     */
    public String getFromTable()
    {
        return fromTable;
    }

    /**
     * Set table name from which join will be established.
     * 
     * @param fromTable
     *            table name from which join will be established
     */
    public void setFromTable( String fromTable )
    {
        this.fromTable = fromTable;
    }

    /**
     * Get table name to which join will be established.
     * 
     * @return table name to which join will be established
     */
    public String getToTable()
    {
        return toTable;
    }

    /**
     * Set table name to which join will be established.
     * 
     * @param toTable
     *            table name to which join will be established
     */
    public void setToTable( String toTable )
    {
        this.toTable = toTable;
    }

    /**
     * Get fields from which join be established.
     * 
     * @return fields from which join be established.
     */
    public String[] getFromFields()
    {
        return fromFields.toArray( new String[fromFields.size()] );
    }

    /**
     * Get fields to which join be be established.
     * 
     * @return fields to which join be established
     */
    public String[] getToFields()
    {
        return toFields.toArray( new String[toFields.size()] );
    }

    /**
     * Set a pair of fields names which will be included in join.
     * 
     * @param fromToFields
     *            a pair of fields names which will be included in join
     */

    public void setFromToFields( String... fromToFields )
    {
        int indexAll = fromToFields.length - fromToFields.length % 2;
        for ( int index = 0; index < indexAll; index += 2 )
        {
            String from = fromToFields[index];
            String to = fromToFields[index + 1];
            if ( !fromFields.contains( from ) && !toFields.contains( to ) )
            {
                fromFields.add( from );
                toFields.add( to );
            }
            else if ( fromFields.contains( from ) && !toFields.contains( to ) )
            {
                toFields.set( fromFields.indexOf( from ), to );
            }
            else if ( toFields.contains( to ) && !fromFields.contains( from ) )
            {
                fromFields.set( toFields.indexOf( to ), from );
            }
        }
    }

    /**
     * Remove join.
     * 
     * @param index
     *            index of from field and to field
     */
    public void removeFromToFields( int index )
    {
        fromFields.remove( index );
        toFields.remove( index );
    }

    /**
     * Get tables join type.
     * 
     * @return tables join type.
     */
    public TablesJoin getTablesJoin()
    {
        return tablesJoin;
    }

    /**
     * Set tables join type.
     * 
     * @param tablesJoin
     *            tables join type.
     */
    public void setTablesJoin( TablesJoin tablesJoin )
    {
        this.tablesJoin = tablesJoin;
    }

    /**
     * Convert JoinedTables object to string.
     */
    @Override
    public String toString()
    {
        StringBuilder builder = new StringBuilder();
        for ( int index = 0; index < fromFields.size(); index++ )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( builder.toString() ) )
            {
                builder.append( StringUtil.STRING_COLON_SPLITTER );
            }
            builder.append( MessageFormat.format( JOINED_FIELDS, fromFields.get( index ), toFields.get( index ) ) );
        }

        // format: fromTable, toTable, tablesJoin, fromToFields
        return MessageFormat.format( JOINED_TABLES, fromTable, toTable, tablesJoin.getValue(), builder.toString() );
    }

}
