/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dmenu.DMenu;
import org.dussan.vaadin.dmenu.menuitem.MenuItem;
import org.dussan.vaadin.dmenu.menuitem.MenuItemElement;
import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.enums.JsonQueryElement;
import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.json.JsonQuery;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONObject;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.ui.ListSelect;

public class DataSourcesMenu
    extends DMenu
{

    private static final long serialVersionUID = -4568749185695417903L;

    private static final String DATABASE_NAME_IS_NOT_SPECIFIED_IN_THE_DATABASE_URL =
        "Database name is not specified in the database url: {0}";

    private static final String COLUMN_NAME = "COLUMN_NAME";

    private static final String DATA_QUERY_ID = "{0}.{1}.{2}.{3}." + TableType.QUERY.toString();

    private static final String DATA_TABLE_ID = "{0}.{1}.{2}.{3}." + TableType.TABLE.toString();

    private static final String TABLE_ID = "{0}.{1}";

    private static final String TABLE_NAME = "TABLE_NAME";

    private static final String TABLES_AREA = "tables-area";

    private static final String TABLES = "Tables: {0}";

    public static final String DATA_SOURCE = "Data source";

    public static final String DATA_SOURCES = "DATA SOURCES";

    public static final String DATA_SOURCE_WITH_DATABASE = "{0} ( {1} )";

    private boolean dataSourcesItemVisible = false;

    private int sourcesPageLength = 0;

    private Map<String, Boolean> dataSourceEnabled = null;

    private Map<String, Boolean> dataSourceTableDisabled = null;

    private Map<String, JdbcUtil> dataSources = null;

    /**
     * Creates a new empty instance.
     */
    public DataSourcesMenu()
    {
        dataSourceEnabled = new LinkedHashMap<>();
        dataSourceTableDisabled = new LinkedHashMap<>();
        dataSources = new LinkedHashMap<>();
        setFloatingMenu( false );
        setSourcesPageLength( 4 );
    }

    /**
     * Check if data sources item is visible.
     * 
     * @return true if data sources item is visible, otherwise false
     */
    public boolean isDataSourcesItemVisible()
    {
        return dataSourcesItemVisible;
    }

    /**
     * Set data sources item visible.
     * 
     * @param visible
     *            if true data sources item is visible, otherwise is hidden
     */
    public void setDataSourcesItemVisible( boolean visible )
    {
        dataSourcesItemVisible = visible;
        for ( int index = 0; index < getComponentCount(); index++ )
        {
            MenuItem menuItem = getMenuItem( index );
            if ( SharedUtil.isNotNullAndNotEmpty( menuItem.getData() ) && DATA_SOURCES.equals( menuItem.getData() ) )
            {
                getTab( index ).setVisible( visible );
                break;
            }
        }
    }

    /**
     * Get sources page length.
     * 
     * @return sources page length
     */
    public int getSourcesPageLength()
    {
        return sourcesPageLength;
    }

    /**
     * Set sources page length.
     * 
     * @param visibleSourcesRows
     *            sources page length
     */
    public void setSourcesPageLength( int sourcesPageLength )
    {
        this.sourcesPageLength = sourcesPageLength;
    }

    /**
     * Get data sources.
     * 
     * @return data sources
     */
    public Map<String, JdbcUtil> getDataSources()
    {
        return dataSources;
    }

    /**
     * Set data sources.
     * 
     * @param dataSources
     *            data sources
     */
    public void setDataSources( Map<String, JdbcUtil> dataSources )
    {
        this.dataSources.putAll( dataSources );
    }

    /**
     * Check if data source is enabled.
     * 
     * @param dataSourceId
     *            data source id
     * @return true data source is enabled, otherwise false
     */
    public boolean isDataSourceEnabled( String dataSourceId )
    {
        if ( dataSourceEnabled.containsKey( dataSourceId ) )
        {
            return dataSourceEnabled.get( dataSourceId );
        }
        return false;
    }

    /**
     * Set data source enabled. User can later change a state of the data source if data source menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param enabled
     *            if true data source is enabled, otherwise is disabled
     */
    public void setDataSourceEnabled( String dataSourceId, boolean enabled )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            dataSourceEnabled.put( dataSourceId, enabled );
            for ( int index = 0; index < getComponentCount(); index++ )
            {
                MenuItem menuItem = getMenuItem( index );
                if ( getTab( index ).isVisible() != enabled && SharedUtil.isNotNullAndNotEmpty( menuItem.getData() )
                    && dataSourceId.equals( menuItem.getData() ) )
                {
                    getTab( index ).setVisible( enabled );
                    break;
                }
            }
        }
    }

    /**
     * Check if data source table is disabled. User can later change a state of the data source table if data source
     * menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @return true data source table is disabled, otherwise false
     */
    public boolean isDataSourceTableDisabled( String dataSourceId, String tableName )
    {
        String tableId = MessageFormat.format( TABLE_ID, dataSourceId, tableName );
        if ( dataSourceTableDisabled.containsKey( tableId ) )
        {
            return dataSourceTableDisabled.get( tableId );
        }
        return false;
    }

    /**
     * Set data source table disabled.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @param disabled
     *            if true data source table is disabled, otherwise is enabled
     */
    public void setDataSourceTableDisabled( String dataSourceId, String tableName, boolean disabled )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && SharedUtil.isNotNullAndNotEmpty( tableName ) )
        {
            dataSourceTableDisabled.put( MessageFormat.format( TABLE_ID, dataSourceId, tableName ), disabled );
            for ( int index = 0; index < getComponentCount(); index++ )
            {
                MenuItem menuItem = getMenuItem( index );
                if ( SharedUtil.isNotNullAndNotEmpty( menuItem.getData() )
                    && DATA_SOURCES.equals( menuItem.getData() ) )
                {
                    for ( MenuItemElement menuItemElement : menuItem.getMenuItemElements() )
                    {
                        if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() )
                            && dataSourceId.equals( menuItemElement.getData() ) )
                        {
                            if ( disabled )
                            {
                                ( (ListSelect) menuItemElement.getContext() ).unselect( tableName );
                            }
                            else
                            {
                                ( (ListSelect) menuItemElement.getContext() ).select( tableName );
                            }
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }

    /**
     * Get table item element.
     * 
     * @param metaData
     *            meta data
     * @param databaseName
     *            database name
     * @param tableName
     *            table name
     * @return table item element
     */
    private MenuItemElement getTableItemElement( DatabaseMetaData metaData, String dataSourceId, String databaseName,
                                                 String tableName )
    {
        try
        {
            // collect table column names
            ListSelect tableColumnsList = new ListSelect();
            tableColumnsList.setRows( getSourcesPageLength() );
            tableColumnsList.setMultiSelect( true );
            tableColumnsList.setNullSelectionAllowed( false );
            tableColumnsList.setSizeFull();

            ResultSet tableColumns = metaData.getColumns( null, null, tableName, null );
            while ( tableColumns.next() )
            {
                String columnName = tableColumns.getString( COLUMN_NAME );
                tableColumnsList.addItem( columnName );
                tableColumnsList.select( columnName );
            }
            tableColumns.close();

            // DATATABLE_ID format: data source id, database name, table name, table columns
            MenuItemElement tableItemElement = new MenuItemElement( tableName, tableColumnsList );
            tableItemElement.setData( MessageFormat.format( DATA_TABLE_ID, dataSourceId, databaseName, tableName,
                                                            Arrays.toString( tableColumnsList.getItemIds()
                                                                                             .toArray() ) ) );
            return tableItemElement;
        }
        catch ( SQLException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getLocalizedMessage(), e );
            return null;
        }
    }

    /**
     * Get tables item.
     * 
     * @param metaData
     *            meta data
     * @param catalog
     *            catalog name
     * @param schema
     *            schema name
     * @param dataSourceId
     *            id of data source
     * @param dataSourceDescription
     *            description of data source
     * @param databaseName
     *            database name
     * @return tables item
     */
    private MenuItem getTablesItem( DatabaseMetaData metaData, String catalog, String schema, String dataSourceId,
                                    String dataSourceDescription, String databaseName )
    {
        try
        {
            MenuItem tablesItem =
                new MenuItem( MessageFormat.format( DATA_SOURCE_WITH_DATABASE, dataSourceDescription, databaseName ) );

            ResultSet tables = metaData.getTables( catalog, schema, null, null );
            while ( tables.next() )
            {
                // add table menu item element to table menu item
                MenuItemElement tableItemElement =
                    getTableItemElement( metaData, dataSourceId, databaseName, tables.getString( TABLE_NAME ) );
                if ( SharedUtil.isNotNull( tableItemElement ) )
                {
                    tablesItem.addMenuItemElement( tableItemElement );
                }
            }
            tables.close();

            tablesItem.setData( dataSourceId );
            tablesItem.setMenuItemElementsDraggable( true );
            return tablesItem;
        }
        catch ( SQLException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getLocalizedMessage(), e );
            return null;
        }
    }

    /**
     * Prepare data source menu.
     * 
     * @param dataSources
     *            hash map of data sources
     */
    public void prepare( Map<String, JdbcUtil> dataSources )
    {
        addStyleName( TABLES_AREA );
        this.dataSources.putAll( dataSources );

        ListSelect dataSourcesList = new ListSelect();
        dataSourcesList.setWidth( "150px" );
        dataSourcesList.setRows( getSourcesPageLength() );
        dataSourcesList.setImmediate( true );
        dataSourcesList.setMultiSelect( true );
        dataSourcesList.setNullSelectionAllowed( false );

        // collect data sources
        final MenuItem dataSourcesItem = new MenuItem( DATA_SOURCES );
        dataSourcesItem.setData( DATA_SOURCES );
        for ( String dataSourceId : dataSources.keySet() )
        {
            if ( SharedUtil.isNull( dataSourceEnabled.get( dataSourceId ) ) )
            {
                dataSourceEnabled.put( dataSourceId, true );
            }

            String description = dataSources.get( dataSourceId )
                                            .getDescription();
            dataSourcesList.addItem( description );
            dataSourcesList.select( description );
        }
        dataSourcesItem.addMenuItemElement( new MenuItemElement( DATA_SOURCE, dataSourcesList ) );

        // data source list value change event listener
        dataSourcesList.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = 1967058519175394515L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                Set<?> values = (Set<?>) event.getProperty()
                                              .getValue();
                for ( MenuItemElement menuItemElement : dataSourcesItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() ) )
                    {
                        String dataSourceId = menuItemElement.getData()
                                                             .toString();
                        String description = getDataSources().get( dataSourceId )
                                                             .getDescription();
                        menuItemElement.setVisible( values.contains( description ) );
                        setDataSourceEnabled( dataSourceId, values.contains( description ) );
                    }
                }
            }
        } );

        // add data sources menu item to main menu
        addMenuItem( dataSourcesItem );

        // data source tables
        for ( String dataSourceId : dataSources.keySet() )
        {
            ListSelect tablesList = new ListSelect();
            tablesList.setRows( getSourcesPageLength() );
            tablesList.setSizeFull();
            tablesList.setImmediate( true );
            tablesList.setMultiSelect( true );
            tablesList.setNullSelectionAllowed( false );

            JdbcUtil dataSource = dataSources.get( dataSourceId );
            String dataSourceDescription = dataSource.getDescription();
            try
            {
                Connection connection = dataSource.getConnection();
                DatabaseMetaData metaData = connection.getMetaData();

                // catalog or schema cannot be empty
                if ( connection.getCatalog() == null && connection.getSchema() == null )
                {
                    throw new IllegalArgumentException( MessageFormat.format( DATABASE_NAME_IS_NOT_SPECIFIED_IN_THE_DATABASE_URL,
                                                                              dataSource.getUrl() ) );
                }

                // database name
                String databaseName = connection.getCatalog();
                if ( connection.getCatalog() == null )
                {
                    databaseName = connection.getSchema();
                }

                MenuItem tablesItem = getTablesItem( metaData, connection.getCatalog(), connection.getSchema(),
                                                     dataSourceId, dataSourceDescription, databaseName );
                if ( SharedUtil.isNotNull( tablesItem ) )
                {
                    // add tables menu item to main menu
                    addMenuItem( tablesItem );

                    // collect table names from tables menu item
                    for ( MenuItemElement itemElement : tablesItem.getMenuItemElements() )
                    {
                        boolean tableSelect = true;
                        String tableName = itemElement.getCaption();
                        String tableId = MessageFormat.format( TABLE_ID, dataSourceId, tableName );

                        // data source table disabled
                        if ( dataSourceTableDisabled.containsKey( tableId ) )
                        {
                            tableSelect = dataSourceTableDisabled.get( tableId );
                        }

                        // data source table selected
                        tablesList.addItem( tableName );
                        itemElement.setVisible( tableSelect );
                        if ( tableSelect )
                        {
                            tablesList.select( tableName );
                        }
                        else
                        {
                            tablesList.unselect( tableName );
                        }
                    }

                    // add data sources menu item element with table names to data sources menu item
                    MenuItemElement menuItemElement =
                        new MenuItemElement( MessageFormat.format( TABLES, dataSourceDescription ), tablesList );
                    menuItemElement.setData( dataSourceId );
                    dataSourcesItem.addMenuItemElement( menuItemElement );
                }

            }
            catch ( SQLException e )
            {
                Logger.getLogger( DQuery.class.getName() )
                      .log( Level.INFO, e.getLocalizedMessage(), e );
            }
            finally
            {
                dataSource.close();
            }

            // enable menu item with data source id
            setDataSourceEnabled( dataSourceId, dataSourceEnabled.get( dataSourceId ) );

            // table list value change event listener
            tablesList.addValueChangeListener( new ValueChangeListener()
            {
                private static final long serialVersionUID = 6761242121554232760L;

                @Override
                public void valueChange( ValueChangeEvent event )
                {
                    Object dataSourceId = ( (MenuItemElement) ( (ListSelect) event.getProperty() ).getParent()
                                                                                                  .getParent()
                                                                                                  .getParent()
                                                                                                  .getParent() ).getData();
                    if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
                    {
                        for ( MenuItem menuItem : getMenuItems() )
                        {
                            // menu item with data source id
                            if ( dataSourceId.toString()
                                             .equals( menuItem.getData() ) )
                            {
                                Set<?> values = (Set<?>) event.getProperty()
                                                              .getValue();
                                for ( MenuItemElement menuItemElement : menuItem.getMenuItemElements() )
                                {
                                    if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() ) )
                                    {
                                        String[] tableName = menuItemElement.getData()
                                                                            .toString()
                                                                            .split( StringUtil.STRING_DOT_SPLITTER );
                                        if ( tableName.length == 2 )
                                        {
                                            menuItemElement.setVisible( values.contains( tableName[1] ) );
                                        }
                                    }
                                }
                                break;
                            }
                        }
                    }
                }
            } );
        }

        // enable data source
        for ( String dataSourceId : dataSources.keySet() )
        {
            if ( !dataSourceEnabled.get( dataSourceId ) )
            {
                String description = dataSources.get( dataSourceId )
                                                .getDescription();
                dataSourcesList.unselect( description );
            }
        }

        // data sources item visibility
        setDataSourcesItemVisible( dataSourcesItemVisible );
    }

    /**
     * Add query item element.
     * 
     * @param jsonQuery
     *            json query
     * @return table item element
     */
    public void addQueryItemElement( JsonQuery jsonQuery )
    {
        MenuItemElement queryItemElement = null;
        if ( SharedUtil.isNotNull( jsonQuery ) )
        {
            // collect table column names
            ListSelect tableColumnsList = new ListSelect();
            tableColumnsList.setRows( getSourcesPageLength() );
            tableColumnsList.setMultiSelect( true );
            tableColumnsList.setNullSelectionAllowed( false );
            tableColumnsList.setSizeFull();

            List<String> fields = new ArrayList<>();
            JSONArray jsonFields = (JSONArray) jsonQuery.get( JsonQueryElement.FIELDS );
            for ( int index = 0; index < jsonFields.length(); index++ )
            {
                JSONObject field = (JSONObject) jsonFields.get( index );
                fields.add( field.getInt( JsonQuery.TABLE_FIELD ), field.getString( BaseTable.DESCRIPTION_NAME ) );
            }
            for ( String field : fields )
            {
                tableColumnsList.addItem( field );
                tableColumnsList.select( field );
            }

            // DATATABLE_ID format: data source id, query, table name, table columns
            queryItemElement = new MenuItemElement( jsonQuery.get( JsonQueryElement.NAME )
                                                             .toString(),
                                                    tableColumnsList );
            queryItemElement.setData( MessageFormat.format( DATA_QUERY_ID,
                                                            jsonQuery.get( JsonQueryElement.DATA_SOURCE_ID ),
                                                            jsonQuery.get( JsonQueryElement.GROUP ),
                                                            jsonQuery.get( JsonQueryElement.NAME ),
                                                            Arrays.toString( fields.toArray() ) ) );
        }

        // add menu item element to menu item
        if ( SharedUtil.isNotNull( queryItemElement ) )
        {
            String jsonQueryGroup = jsonQuery.get( JsonQueryElement.GROUP )
                                             .toString();

            // get existing menu item
            MenuItem queryMenuItem = null;
            for ( MenuItem menuItem : getMenuItems() )
            {
                if ( menuItem.getCaption()
                             .equals( jsonQueryGroup ) )
                {
                    queryMenuItem = menuItem;
                    break;
                }
            }

            if ( SharedUtil.isNull( queryMenuItem ) )
            {
                // create new menu item
                queryMenuItem = new MenuItem( jsonQueryGroup );
                queryMenuItem.setMenuItemElementsDraggable( true );
                queryMenuItem.addMenuItemElement( queryItemElement );
                queryMenuItem.setData( jsonQueryGroup );
                addMenuItem( queryMenuItem );
            }
            else
            {
                // save menu item
                MenuItemElement menuItemElement = null;
                for ( MenuItemElement itemElement : queryMenuItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( itemElement.getData() ) )
                    {
                        String[] data = itemElement.getData()
                                                   .toString()
                                                   .split( StringUtil.STRING_DOT_SPLITTER );
                        if ( 3 <= data.length && data[0].equals( jsonQuery.get( JsonQueryElement.DATA_SOURCE_ID ) )
                            && data[2].equals( jsonQuery.get( JsonQueryElement.NAME ) ) )
                        {
                            menuItemElement = itemElement;
                            break;
                        }
                    }
                }

                // save menu item element
                queryMenuItem.setData( jsonQueryGroup );
                queryMenuItem.setMenuItemElementsDraggable( true );
                if ( SharedUtil.isNull( menuItemElement ) )
                {
                    // add new menu item element
                    queryMenuItem.addMenuItemElement( queryItemElement );
                }
                else
                {
                    // replace existing menu item element
                    int index = queryMenuItem.getComponentIndex( menuItemElement );
                    queryMenuItem.removeComponent( menuItemElement );
                    queryMenuItem.addMenuItemElement( queryItemElement, index );
                }
            }
        }
    }

    public void removeQueryItemElement( String jsonQueryGroup, String jsonQueryName )
    {
        // get query menu item
        for ( MenuItem menuItem : getMenuItems() )
        {
            if ( menuItem.getCaption()
                         .equals( jsonQueryGroup ) )
            {
                // get and remove query menu item element
                for ( MenuItemElement itemElement : menuItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( itemElement.getCaption() ) && itemElement.getCaption()
                                                                                                   .equals( jsonQueryName ) )
                    {
                        menuItem.removeComponent( itemElement );
                        break;
                    }
                }

                // if query menu item is empty remove it from data source menu
                if ( menuItem.isEmpty() )
                {
                    removeMenuItem( getMenuItemIndex( menuItem ) );
                }
                break;
            }
        }
    }

}
