/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.json;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import org.dussan.vaadin.dquery.enums.JsonQueryElement;
import org.dussan.vaadin.dquery.utils.SharedUtil;

public class JsonBuilder
    implements Serializable
{

    private static final long serialVersionUID = 23010531792953775L;

    private transient Map<String, Map<String, JsonQuery>> jsonQueries = null;

    /**
     * Creates a new instance.
     */
    public JsonBuilder()
    {
        jsonQueries = new HashMap<>();
    }

    /**
     * Get all json queries.
     * 
     * @return all json queries
     */
    public Map<String, Map<String, JsonQuery>> getJsonQueries()
    {
        return jsonQueries;
    }

    /**
     * Get json queries for specified query group.
     * 
     * @param queryGroup
     *            query group
     * @return json queries for specified query group
     */
    public Map<String, JsonQuery> getQueries( String queryGroup )
    {
        return jsonQueries.get( queryGroup );
    }

    /**
     * Add json query.
     * 
     * @param jsonQuery
     *            json query.
     */
    public void addQuery( JsonQuery jsonQuery )
    {
        String queryGroupId = jsonQuery.get( JsonQueryElement.GROUP )
                                       .toString();
        String queryName = jsonQuery.get( JsonQueryElement.NAME )
                                    .toString();
        Map<String, JsonQuery> queries = getQueries( queryGroupId );
        if ( SharedUtil.isNull( queries ) )
        {
            queries = new HashMap<>();
        }
        queries.put( queryName, jsonQuery );
        jsonQueries.put( queryGroupId, queries );
    }

    /**
     * Remove json query from specified query group.
     * 
     * @param jsonQueryGroup
     *            query group
     * @param jsonQueryName
     *            query name
     */
    public void removeQuery( String jsonQueryGroup, String jsonQueryName )
    {
        Map<String, JsonQuery> queries = getQueries( jsonQueryGroup );
        if ( SharedUtil.isNotNull( queries ) )
        {
            queries.remove( jsonQueryName );
            jsonQueries.put( jsonQueryGroup, queries );
        }
    }

}
