/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.dussan.vaadin.dquery.client.events.QueryEvents;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.NodeList;
import com.google.gwt.dom.client.Style.Position;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.event.dom.client.HasMouseDownHandlers;
import com.google.gwt.event.dom.client.HasMouseMoveHandlers;
import com.google.gwt.event.dom.client.HasMouseUpHandlers;
import com.google.gwt.event.dom.client.HasMouseWheelHandlers;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.dom.client.MouseWheelEvent;
import com.google.gwt.event.dom.client.MouseWheelHandler;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.DOM;
import com.vaadin.client.ui.VCustomLayout;

public class VDQuery
    extends VCustomLayout
    implements MouseDownHandler, HasMouseDownHandlers, MouseMoveHandler, HasMouseMoveHandlers, MouseUpHandler,
    HasValueChangeHandlers<Object[]>, HasMouseUpHandlers, MouseWheelHandler, HasMouseWheelHandlers
{

    private static final String CLASS_NAME = "className";

    private static final String STRING_COLON_SPLITTER = ":";

    private static final String STRING_DASH_SPLITTER = "-";

    private static final String EMPTY_VALUE = "";

    private static final String SPACE_VALUE = " ";

    private static final String MENU_ITEM = "v-dmenu-item";

    private static final String MENU_ITEM_PANEL = "v-panel" + MENU_ITEM;

    private static final String TABLE = "v-table";

    private static final String TABLE_BODY = TABLE + "-table";

    private static final String TABLE_DRAG = "v-drag-element";

    private static final String TABLE_DROP = "table-drop";

    private static final String TABLE_DROP_BUTTON = TABLE_DROP + "-button";

    private static final String TABLE_DROP_BUTTON_LAST = TABLE_DROP_BUTTON + "-last";

    private static final String TABLE_DROP_LOCATION = TABLE_DROP + "-location";

    private static final String TABLE_BODY_WRAPPER = TABLE + "-body-wrapper";

    private static final String TABLE_CELL_WRAPPER = TABLE + "-cell-wrapper";

    private static final String TABLE_DESCRIPTION = "table-description";

    private static final String TABLE_MOVED = "v-absolutelayout-wrapper-";

    private static final String TABLE_ROW_FIELD = TABLE + "-row-field";

    private static final String TABLE_ROW_HIDDEN = TABLE_ROW_FIELD + "-hidden";

    private static final String TABLE_ROW_ODD = TABLE_ROW_FIELD + "-odd";

    private static final String TABLE_SOURCE = "table-source";

    private static final String TABLE_TR = "tr";

    private static final String TABLES_AREA = "tables-area";

    private static final String FIELDS_AREA = "fields-area";

    private static final String FIELDS_AREA_HEADER = FIELDS_AREA + "-header";

    private static final String FIELDS_AREA_CRITERIA = FIELDS_AREA + "-criteria";

    private static final String SVG = "v-has-svg";

    private static final String SVG_PARENT = "v-absolutelayout-margin";

    private static final String SVG_HEIGHT = "$height";

    private static final String SVG_WIDTH = "$width";

    private static final String SVG_TAG = "svg";

    private static final String SVG_JOIN_INDEX = "join-index";

    private static final String SVG_JOIN_INDEX_VALUE = "$join-index";

    private static final String SVG_JOIN_FIELDS_INDEX = "join-fields-index";

    private static final String SVG_JOIN_FIELDS_INDEX_VALUE = "$join-fields-index";

    private static final String SVG_JOIN_POINT_X1 = "$x1";

    private static final String SVG_JOIN_POINT_X2 = "$x2";

    private static final String SVG_JOIN_POINT_X3 = "$x3";

    private static final String SVG_JOIN_POINT_X4 = "$x4";

    private static final String SVG_JOIN_POINT_Y1 = "$y1";

    private static final String SVG_JOIN_POINT_Y2 = "$y2";

    private static final String SVG_JOIN_POINT_Y3 = "$y3";

    private static final String SVG_JOIN_POINT_Y4 = "$y4";

    private static final String SVG_JOIN_STROKE_WIDTH = "$strokeWidth";

    private static final String SVG_JOIN_TYPE_INNER = "0";

    private static final String SVG_JOIN_TYPE_LEFT = "1";

    private static final String SVG_JOIN_TYPE_RIGHT = "2";

    private static final String SVG_DEF = "<svg xmlns='http://www.w3.org/2000/svg' width='$width' height='$height'>"
        + "<defs><marker id='left' viewBox='0 0 10 10' refX='12' refY='5' markerUnits='strokeWidth' "
        + "markerWidth='10' markerHeight='5' orient='auto'><path d='M0,0 L10,5 L0,10 z' /></marker>"
        + "<marker id='right' viewBox='0 0 10 10' refX='-2' refY='5' markerUnits='strokeWidth' "
        + "markerWidth='10' markerHeight='5' orient='auto'><path d='M10,0 L0,5 L10,10 z' /></marker>"
        + "<marker id='circle' viewBox='0 0 10 10' refX='5' refY='5' markerUnits='strokeWidth' "
        + "markerWidth='10' markerHeight='10' orient='auto'><circle cx='5' cy='5' r='2' /></marker></defs>";

    private static final String SVG_LINE_PATH_JOIN_INNER = "<path marker-start='url(#circle)' marker-end="
        + "'url(#circle)' stroke-width='$strokeWidth' fill='none' d='M$x1,$y1 L$x2,$y2 L$x2,$y2 L$x3,$y3 L$x4,$y4' />";

    private static final String SVG_LINE_PATH_JOIN_LEFT = "<path marker-start='url(#circle)' marker-end='url(#left)'"
        + " stroke-width='$strokeWidth' fill='none' d='M$x1,$y1 L$x2,$y2 L$x2,$y2 L$x3,$y3 L$x4,$y4' />";

    private static final String SVG_LINE_PATH_JOIN_RIGHT = "<path marker-start='url(#right)' marker-end='url(#circle)' "
        + "stroke-width='$strokeWidth' fill='none' d='M$x1,$y1 L$x2,$y2 L$x2,$y2 L$x3,$y3 L$x4,$y4' />";

    private static final String SVG_LINE_PATH_EVENT = "<path stroke-width='$strokeWidth' fill='none' "
        + "stroke='rgba(0,0,0,0.0001)' class='joined-tables-line' d='M$x1,$y1 L$x2,$y2 L$x2,$y2 L$x3,$y3 L$x4,$y4' "
        + "join-index='$join-index' join-fields-index='$join-fields-index' />";

    private boolean mouseDownOnDroppedTable = false;

    private boolean mouseDownEventActive = false;

    private boolean criteriaTableScrollEventsActive = false;

    private boolean redrawJoinedTablesLines = false;

    private boolean redrawJoinedTablesLinesScheduler = false;

    private int redrawJoinedTablesLinesSchedulerCounter = 0;

    private int left = 0;

    private int top = 0;

    private List<Integer> invisibleHeaderRows = null;

    private List<String> joinedTables = null;

    private List<Element[]> joinedElements = null;

    private List<Integer[]> joinedElementsIndex = null;

    private Element droppedTableLocation = null;

    private Element movedTable = null;

    private Element svgElement = null;

    public VDQuery()
    {
        this.addMouseDownHandler( this );
        this.addMouseMoveHandler( this );
        this.addMouseUpHandler( this );
        this.addMouseWheelHandler( this );

        invisibleHeaderRows = new ArrayList<>();
        joinedTables = new ArrayList<>();
        joinedElements = new ArrayList<>();
        joinedElementsIndex = new ArrayList<>();
    }

    private static Element getParentElementByClassName( Element element, String parentClassName, String className )
    {
        String tempClassName = EMPTY_VALUE;
        Element searchingElement = element;
        while ( searchingElement.hasParentElement() && tempClassName.isEmpty() )
        {
            if ( searchingElement.hasClassName( parentClassName ) || searchingElement.hasClassName( className ) )
            {
                if ( searchingElement.hasClassName( className ) )
                {
                    return searchingElement;
                }
                tempClassName = searchingElement.getClassName();
            }
            searchingElement = searchingElement.getParentElement();
        }
        return null;
    }

    private static List<Element> getChildElementsByClassName( Element element, String className )
    {
        ArrayList<Element> findedElements = new ArrayList<Element>();
        findChildElementsByClassName( findedElements, element, className );
        return findedElements;
    }

    private static void findChildElementsByClassName( ArrayList<Element> findedChildElements, Element element,
                                                      String className )
    {
        if ( element != null )
        {
            String elementClasses = element.getPropertyString( CLASS_NAME );
            if ( elementClasses != null )
            {
                List<String> classes = new ArrayList<String>( Arrays.asList( elementClasses.split( SPACE_VALUE ) ) );
                if ( classes.contains( className ) )
                {
                    findedChildElements.add( element );
                }
            }

            for ( int i = 0; i < DOM.getChildCount( element ); i++ )
            {
                Element child = DOM.getChild( element, i );
                findChildElementsByClassName( findedChildElements, child, className );
            }
        }
    }

    private void processChangeJoinedTables( String joinedTablesId, int joinedFieldsIndex )
    {
        QueryEvents.fireChangeJoinedTables( this, joinedTablesId, joinedFieldsIndex );
    }

    private void processTableMoved( String movedTableId, int left, int top )
    {
        QueryEvents.fireTableMoved( this, movedTableId, left, top );
    }

    private void activateCriteriaTableScrollEvents()
    {
        // scroll events on criteria table
        if ( !criteriaTableScrollEventsActive )
        {
            List<Element> elements = getChildElementsByClassName( getElement(), FIELDS_AREA_HEADER );
            if ( 0 < elements.size() )
            {
                Element headerSourceElement = getChildElementsByClassName( elements.get( 0 ), TABLE_SOURCE ).get( 0 );
                headerSourceElement = getChildElementsByClassName( headerSourceElement, TABLE_BODY_WRAPPER ).get( 0 );

                elements = getChildElementsByClassName( getElement(), FIELDS_AREA_CRITERIA );
                if ( 0 < elements.size() )
                {
                    Element criteriaDescriptionElement =
                        getChildElementsByClassName( elements.get( 0 ), TABLE_DESCRIPTION ).get( 0 );
                    criteriaDescriptionElement =
                        getChildElementsByClassName( criteriaDescriptionElement, TABLE_BODY_WRAPPER ).get( 0 );
                    Element criteriaSourceElement =
                        getChildElementsByClassName( elements.get( 0 ), TABLE_SOURCE ).get( 0 );
                    criteriaSourceElement =
                        getChildElementsByClassName( criteriaSourceElement, TABLE_BODY_WRAPPER ).get( 0 );

                    // activate scroll event
                    criteriaTableScrollEventsActive = true;
                    activateCriteriaTablesScrollEvent( headerSourceElement, criteriaDescriptionElement,
                                                       criteriaSourceElement );
                }
            }
        }
    }

    private native void activateCriteriaTablesScrollEvent( Element headerSourceTable, Element criteriaDescriptionTable,
                                                           Element criteriaSourceTable )
    /*-{
        criteriaSourceTable.onscroll = function()
        {
            criteriaDescriptionTable.scrollTop =  criteriaSourceTable.scrollTop;
            headerSourceTable.scrollLeft =  criteriaSourceTable.scrollLeft ;
        }
    }-*/;

    private int getLeft()
    {
        return left;
    }

    private void setLeft( int left )
    {
        this.left = left;
    }

    private int getTop()
    {
        return top;
    }

    private void setTop( int top )
    {
        this.top = top;
    }

    private void drawJoinedLines()
    {
        int dx = 20;
        int width = 0;
        int height = 0;
        int strokeWidth = 2;
        int eventStrokeWidth = 14;
        String svgPath = new String();
        String svgEventPath = new String();
        int absoluteLeft = droppedTableLocation.getAbsoluteLeft() + droppedTableLocation.getScrollLeft();
        int absoluteTop = droppedTableLocation.getAbsoluteTop() + droppedTableLocation.getScrollTop();

        for ( int index = 0; index < joinedElements.size(); index++ )
        {
            // select svg line path
            String svgJoinedPath = new String();
            Element[] joinedElement = joinedElements.get( index );
            Integer joinIndex = joinedElementsIndex.get( index )[0];
            Integer joinFieldsIndex = joinedElementsIndex.get( index )[1];
            String joinType = joinedTables.get( joinIndex )
                                          .split( STRING_COLON_SPLITTER )[2];
            switch ( joinType )
            {
                case SVG_JOIN_TYPE_LEFT:
                    svgJoinedPath = SVG_LINE_PATH_JOIN_LEFT;
                    break;
                case SVG_JOIN_TYPE_RIGHT:
                    svgJoinedPath = SVG_LINE_PATH_JOIN_RIGHT;
                    break;
                default:
                case SVG_JOIN_TYPE_INNER:
                    svgJoinedPath = SVG_LINE_PATH_JOIN_INNER;
                    break;
            }

            if ( !svgJoinedPath.isEmpty() )
            {
                // calculate path and points
                int fromLeft = joinedElement[0].getAbsoluteLeft() - absoluteLeft;
                int fromRight = joinedElement[0].getAbsoluteRight() - absoluteLeft;
                int fromTop = joinedElement[0].getAbsoluteTop() - absoluteTop;
                int fromBottom = joinedElement[0].getAbsoluteBottom() - absoluteTop;

                int toLeft = joinedElement[1].getAbsoluteLeft() - absoluteLeft;
                int toRight = joinedElement[1].getAbsoluteRight() - absoluteLeft;
                int toTop = joinedElement[1].getAbsoluteTop() - absoluteTop;
                int toBottom = joinedElement[1].getAbsoluteBottom() - absoluteTop;

                int from = fromRight;
                int to = toLeft;
                int dxSign = 1;
                if ( ( toLeft - fromRight ) < 3 * dx )
                {
                    dxSign = -1;
                    from = fromLeft;
                    to = toRight;
                    if ( toRight < fromRight && ( fromLeft - toRight ) < 3 * dx )
                    {
                        dxSign = 1;
                        from = fromRight;
                        to = toLeft;
                    }
                }

                // calculate/set path dimensions/points
                int x1 = from;
                int y1 = ( fromTop + fromBottom ) / 2;
                int x2 = x1 + dxSign * dx;
                int y2 = y1;
                int x3 = to - dxSign * dx;
                int y3 = ( toTop + toBottom ) / 2;
                int x4 = to;
                int y4 = y3;
                int w = Math.max( x1, Math.max( x2, Math.max( x3, x4 ) ) ) + 2 * strokeWidth;
                int h = Math.max( y1, Math.max( y2, Math.max( y3, y4 ) ) ) + 2 * strokeWidth;
                width = Math.max( width, Math.max( w, Math.max( fromRight, toRight ) ) );
                height = Math.max( height, Math.max( h, Math.max( fromBottom, toBottom ) ) );

                // fill svg line path with points
                svgPath += svgJoinedPath.replace( SVG_JOIN_STROKE_WIDTH, EMPTY_VALUE + strokeWidth )
                                        .replace( SVG_JOIN_POINT_X1, EMPTY_VALUE + x1 )
                                        .replace( SVG_JOIN_POINT_Y1, EMPTY_VALUE + y1 )
                                        .replace( SVG_JOIN_POINT_X2, EMPTY_VALUE + x2 )
                                        .replace( SVG_JOIN_POINT_Y2, EMPTY_VALUE + y2 )
                                        .replace( SVG_JOIN_POINT_X3, EMPTY_VALUE + x3 )
                                        .replace( SVG_JOIN_POINT_Y3, EMPTY_VALUE + y3 )
                                        .replace( SVG_JOIN_POINT_X4, EMPTY_VALUE + x4 )
                                        .replace( SVG_JOIN_POINT_Y4, EMPTY_VALUE + y4 );
                svgEventPath += SVG_LINE_PATH_EVENT.replace( SVG_JOIN_STROKE_WIDTH, EMPTY_VALUE + eventStrokeWidth )
                                                   .replace( SVG_JOIN_INDEX_VALUE, joinIndex.toString() )
                                                   .replace( SVG_JOIN_FIELDS_INDEX_VALUE, joinFieldsIndex.toString() )
                                                   .replace( SVG_JOIN_POINT_X1, EMPTY_VALUE + x1 )
                                                   .replace( SVG_JOIN_POINT_Y1, EMPTY_VALUE + y1 )
                                                   .replace( SVG_JOIN_POINT_X2, EMPTY_VALUE + x2 )
                                                   .replace( SVG_JOIN_POINT_Y2, EMPTY_VALUE + y2 )
                                                   .replace( SVG_JOIN_POINT_X3, EMPTY_VALUE + x3 )
                                                   .replace( SVG_JOIN_POINT_Y3, EMPTY_VALUE + y3 )
                                                   .replace( SVG_JOIN_POINT_X4, EMPTY_VALUE + x4 )
                                                   .replace( SVG_JOIN_POINT_Y4, EMPTY_VALUE + y4 );
            }

            // create svg object
            if ( !svgPath.isEmpty() )
            {
                String svgDef = SVG_DEF.replace( SVG_WIDTH, EMPTY_VALUE + width )
                                       .replace( SVG_HEIGHT, EMPTY_VALUE + height );
                svgElement.setInnerHTML( svgDef + svgPath + svgEventPath );
            }
        }
    }

    private void activateRedrawJoinedTablesLinesScheduler()
    {
        Scheduler.get()
                 .scheduleFixedDelay( new Scheduler.RepeatingCommand()
                 {
                     @Override
                     public boolean execute()
                     {
                         if ( !joinedTables.isEmpty() && joinedElements.isEmpty()
                             && redrawJoinedTablesLinesSchedulerCounter++ < 100 )
                         {
                             redrawJoinedTablesLines = true;
                             redrawJoinedTablesLinesScheduler = true;
                             redrawJoinedTablesLines();
                             return true;
                         }

                         // lines are drawn, so stop scheduler
                         redrawJoinedTablesLinesScheduler = false;

                         // infinite loop protector
                         redrawJoinedTablesLinesSchedulerCounter = 0;

                         return false;
                     }
                 }, 50 );
    }

    public void prepare()
    {
        // activate/enable mouse down event
        List<Element> elements = getChildElementsByClassName( getElement(), TABLES_AREA );
        if ( 0 < elements.size() )
        {
            mouseDownEventActive = true;
        }

        // add svg layer for drawing lines between joined tables/fields
        elements = getChildElementsByClassName( getElement(), TABLE_DROP_LOCATION );
        if ( 0 < elements.size() && svgElement == null )
        {
            droppedTableLocation = elements.get( 0 );
            svgElement = DOM.createDiv();
            svgElement.addClassName( SVG );
            svgElement.getStyle()
                      .setPosition( Position.ABSOLUTE );
            svgElement.getStyle()
                      .setHeight( 100, Unit.PCT );
            svgElement.getStyle()
                      .setWidth( 100, Unit.PCT );
            Element svgParentElement = getChildElementsByClassName( droppedTableLocation, SVG_PARENT ).get( 0 );
            DOM.appendChild( svgParentElement, svgElement );
        }
    }

    @Override
    public HandlerRegistration addValueChangeHandler( ValueChangeHandler<Object[]> handler )
    {
        return addHandler( handler, ValueChangeEvent.getType() );
    }

    @Override
    public HandlerRegistration addMouseDownHandler( MouseDownHandler handler )
    {
        return addDomHandler( handler, MouseDownEvent.getType() );
    }

    @Override
    public HandlerRegistration addMouseMoveHandler( MouseMoveHandler handler )
    {
        return addDomHandler( handler, MouseMoveEvent.getType() );
    }

    @Override
    public HandlerRegistration addMouseUpHandler( MouseUpHandler handler )
    {
        return addDomHandler( handler, MouseUpEvent.getType() );
    }

    @Override
    public HandlerRegistration addMouseWheelHandler( MouseWheelHandler handler )
    {
        return addDomHandler( handler, MouseWheelEvent.getType() );
    }

    @Override
    public void onMouseDown( MouseDownEvent event )
    {
        if ( !mouseDownOnDroppedTable )
        {
            Element element = Element.as( event.getNativeEvent()
                                               .getEventTarget() );
            if ( element.hasAttribute( SVG_JOIN_INDEX ) )
            {
                int index = Integer.parseInt( element.getAttribute( SVG_JOIN_INDEX ) );
                int fieldsIndex = Integer.parseInt( element.getAttribute( SVG_JOIN_FIELDS_INDEX ) );
                processChangeJoinedTables( joinedTables.get( index ), fieldsIndex );
            }
            else if ( !element.hasTagName( SVG_TAG ) && !element.hasClassName( TABLE_DROP_BUTTON )
                && !element.hasClassName( TABLE_DROP_BUTTON_LAST ) )
            {
                // get parent for dropped table
                element = getParentElementByClassName( element, TABLE_DROP_LOCATION, TABLE_DROP );
                if ( element != null )
                {
                    // mouse is clicked inside dropped table
                    mouseDownOnDroppedTable = true;
                    movedTable = element.getParentElement();
                    movedTable.getParentElement()
                              .appendChild( movedTable );
                    setLeft( event.getClientX() - droppedTableLocation.getAbsoluteLeft() - movedTable.getOffsetLeft() );
                    setTop( event.getClientY() - droppedTableLocation.getAbsoluteTop() - movedTable.getOffsetTop() );
                }
            }
        }
        activateCriteriaTableScrollEvents();
    }

    @Override
    public void onMouseMove( MouseMoveEvent event )
    {
        if ( mouseDownOnDroppedTable )
        {
            if ( droppedTableLocation != null && movedTable == null )
            {
                List<Element> elements = getChildElementsByClassName( Document.get()
                                                                              .getBody(),
                                                                      TABLE_DRAG );
                if ( 0 < elements.size() )
                {
                    movedTable = elements.get( 0 );
                }
            }

            if ( droppedTableLocation != null && movedTable != null )
            {
                event.preventDefault();
                int left = Math.min( droppedTableLocation.getClientWidth(), event.getClientX()
                    - droppedTableLocation.getAbsoluteLeft() - getLeft() - droppedTableLocation.getScrollLeft() );
                int top = Math.min( droppedTableLocation.getClientHeight(), event.getClientY()
                    - droppedTableLocation.getAbsoluteTop() - getTop() - droppedTableLocation.getScrollTop() );
                movedTable.getStyle()
                          .setOpacity( 0.55 );
                movedTable.getStyle()
                          .setLeft( left, Unit.PX );
                movedTable.getStyle()
                          .setTop( top, Unit.PX );

                if ( !joinedElements.isEmpty() )
                {
                    drawJoinedLines();
                }
            }
        }
    }

    @Override
    public void onMouseUp( MouseUpEvent event )
    {
        // clear moved table
        if ( movedTable != null )
        {
            // moved table id
            String movedTableId = null;
            for ( String movedTableClass : movedTable.getClassName()
                                                     .split( SPACE_VALUE ) )
            {
                if ( movedTableClass.startsWith( TABLE_MOVED ) && !movedTableClass.endsWith( TABLE_DROP ) )
                {
                    String[] movedTableIds = movedTableClass.split( TABLE_MOVED );
                    if ( movedTableIds.length == 2 && !movedTableIds[1].trim()
                                                                       .isEmpty() )
                    {
                        movedTableId = movedTableIds[1].trim();
                    }
                }
            }

            // process table moved
            if ( movedTableId != null && !movedTableId.isEmpty() )
            {
                int left = Integer.parseInt( movedTable.getStyle()
                                                       .getLeft()
                                                       .replace( Unit.PX.getType(), "" ) );
                int top = Integer.parseInt( movedTable.getStyle()
                                                      .getTop()
                                                      .replace( Unit.PX.getType(), "" ) );
                processTableMoved( movedTableId, left, top );
            }

            // clear moved table data
            movedTable.getStyle()
                      .clearOpacity();
            movedTable = null;
        }

        // reset mouse down click
        mouseDownOnDroppedTable = false;
        criteriaTableScrollEventsActive = false;
    }

    @Override
    public void onMouseWheel( MouseWheelEvent event )
    {
        // scroll events on criteria table
        activateCriteriaTableScrollEvents();
    }

    public List<Integer> getInvisibleHeaderRows()
    {
        return invisibleHeaderRows;
    }

    public void setInvisibleHeaderRows( List<Integer> invisibleHeaderRows )
    {
        this.invisibleHeaderRows = invisibleHeaderRows;
    }

    public void refreshHeaderRows()
    {
        if ( invisibleHeaderRows != null )
        {
            List<Element> mainElement = getChildElementsByClassName( getElement(), FIELDS_AREA_HEADER );
            if ( !mainElement.isEmpty() )
            {
                for ( Element table : getChildElementsByClassName( mainElement.get( 0 ), TABLE_BODY_WRAPPER ) )
                {
                    // fix table height
                    table.getStyle()
                         .clearHeight();
                    table.getFirstChildElement()
                         .getStyle()
                         .clearHeight();

                    // set class to table row
                    int rowCounter = 0;
                    Element bodyTable = getChildElementsByClassName( table, TABLE_BODY ).get( 0 );
                    NodeList<Element> elements = bodyTable.getElementsByTagName( TABLE_TR );
                    for ( int index = 0; index < elements.getLength(); index++ )
                    {
                        Element element = elements.getItem( index );

                        // remove class
                        element.removeClassName( TABLE_ROW_HIDDEN );
                        element.removeClassName( TABLE_ROW_FIELD );
                        element.removeClassName( TABLE_ROW_ODD );

                        // add class
                        if ( invisibleHeaderRows.contains( index ) )
                        {
                            element.addClassName( TABLE_ROW_HIDDEN );
                        }
                        else
                        {
                            switch ( rowCounter++ % 2 )
                            {
                                case 0:
                                    element.addClassName( TABLE_ROW_FIELD );
                                    break;
                                case 1:
                                    element.addClassName( TABLE_ROW_ODD );
                                    break;
                                default:
                                    break;
                            }
                        }
                    }
                }
            }
        }
    }

    public List<String> getJoinedTables()
    {
        // format: fromTable, toTable, tablesJoin, fromToFields
        return joinedTables;
    }

    public void setJoinedTables( List<String> joinedTables )
    {
        List<String> tables = joinedTables;
        if ( tables == null )
        {
            tables = new ArrayList<String>();
        }
        redrawJoinedTablesLines = !this.joinedTables.equals( tables );
        this.joinedTables = tables;
    }

    public void redrawJoinedTablesLines()
    {
        if ( redrawJoinedTablesLines )
        {
            // empty svg object
            svgElement.setInnerHTML( EMPTY_VALUE );

            joinedElements = new ArrayList<>();
            joinedElementsIndex = new ArrayList<>();
            for ( int index = 0; index < joinedTables.size(); index++ )
            {
                // format: fromTable, toTable, tablesJoin, fromToFields
                String[] joinedData = joinedTables.get( index )
                                                  .split( STRING_COLON_SPLITTER );
                String fromTable = joinedData[0];
                String toTable = joinedData[1];
                String[] fromToFields = Arrays.copyOfRange( joinedData, 3, joinedData.length );
                List<Element> fromTableElements = getChildElementsByClassName( getElement(), fromTable );
                List<Element> toTableElements = getChildElementsByClassName( getElement(), toTable );

                if ( fromToFields.length != 0 && !fromTableElements.isEmpty() && !toTableElements.isEmpty() )
                {
                    for ( int indx = 0; indx < fromToFields.length; indx++ )
                    {
                        List<Element> fromElements =
                            getChildElementsByClassName( fromTableElements.get( 0 ), TABLE_CELL_WRAPPER );
                        Element fromElement = null;
                        for ( Element element : fromElements )
                        {
                            if ( fromToFields[indx].split( STRING_DASH_SPLITTER )[0].equals( element.getInnerText() ) )
                            {
                                fromElement = element.getParentElement();
                                break;
                            }
                        }

                        List<Element> toElements =
                            getChildElementsByClassName( toTableElements.get( 0 ), TABLE_CELL_WRAPPER );
                        Element toElement = null;
                        for ( Element element : toElements )
                        {
                            if ( fromToFields[indx].split( STRING_DASH_SPLITTER )[1].equals( element.getInnerText() ) )
                            {
                                toElement = element.getParentElement();
                                break;
                            }
                        }

                        if ( fromElement != null && toElement != null )
                        {
                            joinedElements.add( new Element[] { fromElement, toElement } );
                            joinedElementsIndex.add( new Integer[] { index, indx } );
                        }
                    }
                }
            }

            // draw lines between joined tables
            if ( !joinedElements.isEmpty() )
            {
                drawJoinedLines();
            }
            else if ( !joinedTables.isEmpty() && joinedElements.isEmpty() && !redrawJoinedTablesLinesScheduler )
            {
                // when restore query structure, joined tables lines not drawn
                // so run scheduler to draw lines when is possible
                activateRedrawJoinedTablesLinesScheduler();
            }
        }

        // reset redraw request
        redrawJoinedTablesLines = false;
    }

}
