/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui.windows;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.base.ui.BaseWindow;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;

import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.Align;
import com.vaadin.ui.themes.ValoTheme;

public class TableDataWindow
    extends BaseWindow
{

    private static final long serialVersionUID = 2774123349113924712L;

    private static final String MODAL_CAPTION = "Table data: {0}";

    private static final int PAGE_LENGTH = 10;

    private static final int ROW_VIEW = 100;

    public static final String ROW_SELECT = "select * from {0}.{1};";

    /**
     * Creates a new instance.
     */
    private TableDataWindow()
    {
        getCancelButton().setVisible( false );
        getOkButton().setEnabled( true );
    }

    /**
     * Creates a new instance.
     * 
     * @param dataSources
     *            data sources
     * @param tableData
     *            table data info
     */
    public TableDataWindow( JdbcUtil jdbcUtil, String tableName, String query )
    {
        this();

        Table table = new Table();
        table.setSizeFull();
        table.addStyleName( ValoTheme.TABLE_NO_HORIZONTAL_LINES );
        table.addStyleName( ValoTheme.TABLE_SMALL );
        addContextComponent( table );
        setCaption( MessageFormat.format( MODAL_CAPTION, tableName ) );

        try
        {
            Statement statement = jdbcUtil.getConnection()
                                          .createStatement( ResultSet.CONCUR_READ_ONLY, ResultSet.FETCH_FORWARD );
            ResultSet rowsData = statement.executeQuery( query );
            if ( SharedUtil.isNotNull( rowsData ) )
            {
                // get columns name
                ResultSetMetaData columns = rowsData.getMetaData();
                if ( SharedUtil.isNotNull( columns ) )
                {
                    for ( int column = 1; column <= columns.getColumnCount(); column++ )
                    {
                        table.addContainerProperty( column, Object.class, null, columns.getColumnLabel( column ), null,
                                                    Align.LEFT );
                    }
                }

                // get rows data
                int rowsCount = 0;
                while ( rowsData.next() && rowsCount < ROW_VIEW )
                {
                    Object[] rowData = new Object[table.getColumnHeaders().length];
                    for ( int index = 0; index < rowData.length; index++ )
                    {
                        rowData[index] = rowsData.getObject( index + 1 );
                    }
                    table.addItem( rowData, rowsCount++ );
                }
                table.setPageLength( Math.min( PAGE_LENGTH, rowsCount ) );
            }
            jdbcUtil.close();
        }
        catch ( SQLException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getMessage() );
        }

        // "OK" button click
        getOkButton().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 3491165668669637327L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                // close dialog window
                close();
            }
        } );
    }

}
