/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.sql.builder;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.dussan.vaadin.dquery.base.sql.builder.SqlColumn;
import org.dussan.vaadin.dquery.base.sql.builder.SqlJoin;
import org.dussan.vaadin.dquery.base.sql.builder.SqlTable;
import org.dussan.vaadin.dquery.base.sql.builder.SqlWhere;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.enums.SqlFunctionType;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class SqlBuilder
    implements Serializable
{

    private static final long serialVersionUID = -1983792327374980160L;

    private static final String FROM = "\nFROM\n\t";

    private static final String GROUP_BY = "\nGROUP BY\n\t";

    private static final String HAVING = "\nHAVING\n\t";

    private static final String NEW_LINE = "\n";

    private static final String ORDER_BY = "\nORDER BY\n\t";

    private static final String ORDER_BY_COLUMN = "{0} {1}";

    private static final String SELECT = "SELECT\n\t";

    private static final String STRING = "{0}{1}{2}{3}{4}{5}{6};";

    private static final String UNIQUE_ALIAS = "___{0}__{1}___";

    private static final String WHERE = "\nWHERE\n\t";

    private boolean tableGrouping = false;

    private String queryDataSourceId = null;

    private String sqlSelect = null;

    private String sqlFrom = null;

    private String sqlJoins = null;

    private String sqlWhere = null;

    private String sqlGroupBy = null;

    private String sqlHaving = null;

    private String sqlOrderBy = null;

    private SqlWhere where = null;

    private transient List<SqlColumn> columns = null;

    private transient List<SqlTable> tables = null;

    private transient List<SqlJoin> joins = null;

    /**
     * Creates a new instance.
     */
    public SqlBuilder()
    {
        columns = new ArrayList<>();
        tables = new ArrayList<>();
        joins = new ArrayList<>();
        where = new SqlWhere();
    }

    /**
     * Get query data source id.
     * 
     * @return query data source id
     */
    public String getQueryDataSourceId()
    {
        return queryDataSourceId;
    }

    /**
     * set query data source id.
     * 
     * @param queryDataSourceId
     *            query data source id
     */
    public void setQueryDataSourceId( String queryDataSourceId )
    {
        this.queryDataSourceId = queryDataSourceId;
    }

    /**
     * Check if table has grouping.
     * 
     * @return true if table has grouping, otherwise false
     */
    public boolean isTableGrouping()
    {
        return tableGrouping;
    }

    /**
     * Set table grouping.
     * 
     * @param tableGrouping
     *            table grouping
     */
    public void setTableGrouping( boolean tableGrouping )
    {
        this.tableGrouping = tableGrouping;
    }

    /**
     * Get sql select part.
     * 
     * @return sql select part
     */
    public String getSqlSelect()
    {
        return sqlSelect;
    }

    /**
     * Get sql from part.
     * 
     * @return sql from part
     */
    public String getSqlFrom()
    {
        return sqlFrom;
    }

    /**
     * Get sql joins part.
     * 
     * @return sql joins part
     */
    public String getSqlJoins()
    {
        return sqlJoins;
    }

    /**
     * Get sql where part.
     * 
     * @return sql where part
     */
    public String getSqlWhere()
    {
        return sqlWhere;
    }

    /**
     * Get sql group by part.
     * 
     * @return sql group by part
     */
    public String getSqlGroupBy()
    {
        return sqlGroupBy;
    }

    /**
     * Get sql having part.
     * 
     * @return sql having part
     */
    public String getSqlHaving()
    {
        return sqlHaving;
    }

    /**
     * Get sql order by part.
     * 
     * @return sql order by part
     */
    public String getSqlOrderBy()
    {
        return sqlOrderBy;
    }

    /**
     * Add column where condition.
     * 
     * @param columnId
     *            column id
     * @param rowId
     *            row id
     * @param columnValue
     *            column value
     * @param columnCondition
     *            column condition
     */
    public void addColumnWhereCondition( String columnId, String rowId, String columnValue, String columnCondition )
    {
        where.addWhereCondition( columnId, rowId, columnValue, columnCondition );
    }

    /**
     * Get sql columns.
     * 
     * @return sql columns
     */
    public List<SqlColumn> getColumns()
    {
        return columns;
    }

    /**
     * Add sql column.
     * 
     * @param column
     *            sql column
     */
    public void addColumn( SqlColumn column )
    {
        columns.add( column );
    }

    /**
     * Get sql tables.
     * 
     * @return sql tables
     */
    public List<SqlTable> getTables()
    {
        return tables;
    }

    /**
     * Add sql table.
     * 
     * @param table
     *            sql table
     */
    public void addTable( SqlTable table )
    {
        tables.add( table );
    }

    /**
     * Get sql joins.
     * 
     * @return sql joins
     */
    public List<SqlJoin> getJoins()
    {
        return joins;
    }

    /**
     * Add sql join.
     * 
     * @param join
     *            sql join
     */
    public void addJoin( SqlJoin join )
    {
        joins.add( join );
    }

    /**
     * Get sql generated string.
     */
    @Override
    public String toString()
    {

        // preparation: tables and unique aliases needed for create table joins
        List<String> uniqueAliases = new ArrayList<>();
        List<SqlTable> tables = new ArrayList<>();
        for ( SqlTable table : this.tables )
        {
            tables.add( table );
            uniqueAliases.add( table.getTableAlias() );
        }

        // preparation: check the order of joins
        // make sure that table alias (from table "fromTable") is set before
        // define joins for column which point to that alias (from table alias "toTable")
        // else sql server may complain that join column from table "toTable" is unknown
        if ( !joins.isEmpty() )
        {
            int mainIndex = 0;
            while ( mainIndex < joins.size() )
            {
                // collect all tables and aliases from joins
                List<String> joinFrom = new ArrayList<>();
                List<String> joinTo = new ArrayList<>();
                for ( SqlJoin join : joins )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( join ) )
                    {
                        joinFrom.add( join.getFromTable()
                                          .getTableAlias() );
                        joinTo.add( join.getToTable()
                                        .getTableAlias() );
                    }
                }

                // reorder joins
                for ( int index = 0; index < joinTo.size(); index++ )
                {
                    if ( joinFrom.contains( joinTo.get( index ) ) && index < joinFrom.indexOf( joinTo.get( index ) ) )
                    {
                        Collections.swap( joins, joinFrom.indexOf( joinTo.get( index ) ), index );
                        mainIndex = 0;
                        break;
                    }
                    else
                    {
                        mainIndex += 1;
                    }
                }
            }
        }

        // SELECT part
        StringBuilder sqlSelect = new StringBuilder();
        for ( SqlColumn column : columns )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( column.toString() ) )
            {
                sqlSelect.append( StringUtil.COMMA_WITH_SPACE );
                sqlSelect.append( column.toString() );
            }
        }
        if ( SharedUtil.isNotNullAndNotEmpty( sqlSelect ) )
        {
            sqlSelect.deleteCharAt( 0 );
            this.sqlSelect = StringUtil.trim( sqlSelect.toString() );
            sqlSelect.insert( 0, SELECT );
        }

        // JOINS part
        StringBuilder sqlJoins = new StringBuilder();
        for ( SqlJoin join : joins )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( join ) )
            {
                // remove table name and its alias
                for ( int index = 0; index < tables.size() && 1 < tables.size(); index++ )
                {
                    if ( tables.get( index )
                               .getTableName()
                               .equals( join.getFromTable()
                                            .getTableName() )
                        && tables.get( index )
                                 .getTableAlias()
                                 .equals( join.getFromTable()
                                              .getTableAlias() ) )
                    {
                        tables.remove( index );
                        uniqueAliases.remove( join.getFromTable()
                                                  .getTableAlias() );
                        break;
                    }
                }

                // add join table name and its alias
                if ( !uniqueAliases.contains( join.getToTable()
                                                  .getTableAlias() ) )
                {
                    tables.add( join.getToTable() );
                    uniqueAliases.add( join.getToTable()
                                           .getTableAlias() );
                }

                // alias name already exists, so change it
                if ( uniqueAliases.contains( join.getFromTable()
                                                 .getTableAlias() ) )
                {
                    join.getFromTable()
                        .setTableAlias( MessageFormat.format( UNIQUE_ALIAS, join.getFromTable()
                                                                                .getTableAlias(),
                                                              uniqueAliases.size() ) );
                }

                // create sql join
                if ( SharedUtil.isNotNullAndNotEmpty( sqlJoins ) )
                {
                    sqlJoins.append( NEW_LINE );
                }
                sqlJoins.append( join.toString() );

                // add new alias to unique aliases
                if ( !uniqueAliases.contains( join.getFromTable()
                                                  .getTableAlias() ) )
                {
                    uniqueAliases.add( join.getFromTable()
                                           .getTableAlias() );
                }
            }
        }
        if ( SharedUtil.isNotNullAndNotEmpty( sqlJoins ) )
        {
            this.sqlJoins = StringUtil.trim( sqlJoins.toString() );
            sqlJoins.insert( 0, NEW_LINE );
        }

        // FROM part
        StringBuilder sqlFrom = new StringBuilder();
        for ( SqlTable table : tables )
        {
            sqlFrom.append( StringUtil.COMMA_WITH_SPACE );
            sqlFrom.append( table.toString() );
        }
        if ( SharedUtil.isNotNullAndNotEmpty( sqlFrom ) )
        {
            sqlFrom.deleteCharAt( 0 );
            this.sqlFrom = StringUtil.trim( sqlFrom.toString() );
            sqlFrom.insert( 0, FROM );
        }

        // WHERE part
        StringBuilder sqlWhere = new StringBuilder();
        if ( !tableGrouping && SharedUtil.isNotNullAndNotEmpty( where.toString() ) )
        {
            sqlWhere.append( where.toString() );
            this.sqlWhere = StringUtil.trim( sqlWhere.toString() );
            sqlWhere.insert( 0, WHERE );
        }

        // GROUP BY part
        StringBuilder sqlGroupBy = new StringBuilder();
        for ( SqlColumn column : columns )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( column.toString() ) && !column.getColumn()
                                                                                .equals( BaseTable.FIELDS_ALL )
                && SharedUtil.isNotNullAndNotEmpty( column.getTotalFunction() ) && column.getTotalFunctionType()
                                                                                         .equals( SqlFunctionType.NULL ) )
            {
                sqlGroupBy.append( StringUtil.COMMA_WITH_SPACE );
                sqlGroupBy.append( column.getColumnAlias() );
            }
        }
        if ( SharedUtil.isNotNullAndNotEmpty( sqlGroupBy ) )
        {
            sqlGroupBy.deleteCharAt( 0 );
            this.sqlGroupBy = StringUtil.trim( sqlGroupBy.toString() );
            sqlGroupBy.insert( 0, GROUP_BY );
        }

        // HAVING part
        StringBuilder sqlHaving = new StringBuilder();
        if ( tableGrouping && SharedUtil.isNotNullAndNotEmpty( where.toString() ) )
        {
            sqlHaving.append( where.toString() );
            this.sqlHaving = StringUtil.trim( sqlHaving.toString() );
            sqlHaving.insert( 0, HAVING );
        }

        // ORDER BY part
        StringBuilder sqlOrderBy = new StringBuilder();
        for ( SqlColumn column : columns )
        {
            if ( SharedUtil.isNotNullAndNotEmpty( column.toString() ) && !column.getColumn()
                                                                                .equals( BaseTable.FIELDS_ALL )
                && SharedUtil.isNotNullAndNotEmpty( column.getOrderBy() ) )
            {
                sqlOrderBy.append( StringUtil.COMMA_WITH_SPACE );
                sqlOrderBy.append( MessageFormat.format( ORDER_BY_COLUMN, column.getColumnAlias(),
                                                         column.getOrderBy() ) );
            }
        }
        if ( SharedUtil.isNotNullAndNotEmpty( sqlOrderBy ) )
        {
            sqlOrderBy.deleteCharAt( 0 );
            this.sqlOrderBy = StringUtil.trim( sqlOrderBy.toString() );
            sqlOrderBy.insert( 0, ORDER_BY );
        }

        // generated sql string
        if ( SharedUtil.isNotNullAndNotEmpty( sqlSelect ) )
        {
            return MessageFormat.format( STRING, sqlSelect.toString(), sqlFrom.toString(), sqlJoins.toString(),
                                         sqlWhere.toString(), sqlGroupBy.toString(), sqlHaving.toString(),
                                         sqlOrderBy.toString() );
        }

        return StringUtil.EMPTY_VALUE;
    }

}
