/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui.windows;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.base.ui.BaseWindow;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.vaadin.aceeditor.AceEditor;
import org.vaadin.aceeditor.AceMode;
import org.vaadin.aceeditor.AceTheme;

import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.Align;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.themes.ValoTheme;

public class TestQueryWindow
    extends BaseWindow
{

    private static final long serialVersionUID = -6464952901512672756L;

    private static final String QUERY = "Query";

    private static final String QUERY_CAPTION = QUERY + " tester";

    private static final String QUERY_ERROR = QUERY + " error";

    private static final String QUERY_RESULT = QUERY + " result";

    private static final int EDITOR_HEIGHT = 150;

    private static final int PAGE_LENGTH = 10;

    private static final int ROW_VIEW = 100;

    /**
     * Creates a new instance.
     */
    private TestQueryWindow()
    {
        getCancelButton().setVisible( false );
        getOkButton().setEnabled( true );
    }

    /**
     * Creates a new instance.
     * 
     * @param jdbcUtil
     *            data sources
     * @param tableData
     *            table data info
     */
    public TestQueryWindow( JdbcUtil jdbcUtil, String query )
    {
        this();
        setCaption( QUERY_CAPTION );
        setWidth( 80, Unit.PERCENTAGE );

        AceEditor editor = new AceEditor();
        editor.setCaption( QUERY );
        editor.setSizeFull();
        editor.setHeight( EDITOR_HEIGHT, Unit.PIXELS );
        editor.setHighlightActiveLine( false );
        editor.setMode( AceMode.pgsql );
        editor.setShowPrintMargin( false );
        editor.setTheme( AceTheme.github );
        editor.setValue( query );
        editor.setCursorPosition( 0 );
        editor.setReadOnly( true );
        editor.setWordWrap( true );

        Table table = new Table();
        table.setCaption( QUERY_RESULT );
        table.setSizeFull();
        table.addStyleName( ValoTheme.TABLE_NO_HORIZONTAL_LINES );
        table.addStyleName( ValoTheme.TABLE_SMALL );

        VerticalLayout layout = new VerticalLayout( editor, table );
        layout.setSpacing( true );
        addContextComponent( layout );

        try
        {
            Statement statement = jdbcUtil.getConnection()
                                          .createStatement( ResultSet.CONCUR_READ_ONLY, ResultSet.FETCH_FORWARD );
            ResultSet rowsData = statement.executeQuery( query );
            if ( SharedUtil.isNotNull( rowsData ) )
            {
                // get columns name
                ResultSetMetaData columns = rowsData.getMetaData();
                if ( SharedUtil.isNotNull( columns ) )
                {
                    for ( int column = 1; column <= columns.getColumnCount(); column++ )
                    {
                        table.addContainerProperty( column, Object.class, null, columns.getColumnLabel( column ), null,
                                                    Align.LEFT );
                    }
                }

                // get rows data
                int rowsCount = 0;
                while ( rowsData.next() && rowsCount < ROW_VIEW )
                {
                    Object[] rowData = new Object[table.getColumnHeaders().length];
                    for ( int index = 0; index < rowData.length; index++ )
                    {
                        rowData[index] = rowsData.getObject( index + 1 );
                    }
                    table.addItem( rowData, rowsCount++ );
                }
                table.setPageLength( Math.min( PAGE_LENGTH, rowsCount ) );
            }
            jdbcUtil.close();
        }
        catch ( SQLException e )
        {
            // query test error
            AceEditor error = new AceEditor();
            error.setCaption( QUERY_ERROR );
            error.setSizeFull();
            error.setHeight( EDITOR_HEIGHT, Unit.PIXELS );
            error.setHighlightActiveLine( false );
            error.setMode( AceMode.text );
            error.setShowPrintMargin( false );
            error.setShowGutter( false );
            error.setTheme( AceTheme.github );
            error.setValue( e.getMessage() );
            error.setCursorPosition( 0 );
            error.setReadOnly( true );
            error.setWordWrap( true );
            layout.addComponent( error );
            layout.removeComponent( table );

            // write query test error
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getMessage(), e );
        }
        finally
        {
            jdbcUtil.close();
        }

        // "OK" button click
        getOkButton().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 8811410147954402762L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                // close dialog window
                close();
            }
        } );
    }

}
