/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.json;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.base.QueryElement;
import org.dussan.vaadin.dquery.base.sql.builder.SqlColumn;
import org.dussan.vaadin.dquery.enums.QueryElements;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.sql.builder.SqlBuilder;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.sql.functions.SortingFunctions;
import org.dussan.vaadin.dquery.sql.functions.TotalFunctions;
import org.dussan.vaadin.dquery.utils.ErrorUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class JsonQuery
    implements Serializable
{

    private static final long serialVersionUID = 23010531792953775L;

    public static final String QUERY_AS_COLUMN = "({0})";

    private String dataSourceId = null;

    private String queryGroup = null;

    private String queryName = null;

    private CommonFunctions commonFunctions = null;

    private SortingFunctions sortingFunctions = null;

    private TotalFunctions totalFunctions = null;

    private JSONObject queryData = null;

    /**
     * Creates a new instance.
     */
    private JsonQuery()
    {
        queryData = new JSONObject();
    }

    /**
     * Creates a new instance with query group and query name.
     */
    public JsonQuery( String queryGroup, String queryName )
    {
        this();
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
        {
            this.queryGroup = queryGroup;
            this.queryName = queryName;
        }
    }

    /**
     * Creates a new instance.
     * 
     * @param query
     *            query as string value
     */
    public JsonQuery( String query )
    {
        this();
        try
        {
            QueryElement queryElement = new QueryElement( query );
            if ( SharedUtil.isNotNullAndNotEmpty( queryElement )
                && SharedUtil.isNotNullAndNotEmpty( queryElement.get( QueryElements.GROUP ) )
                && SharedUtil.isNotNullAndNotEmpty( queryElement.get( QueryElements.NAME ) ) )
            {
                this.queryGroup = queryElement.get( QueryElements.GROUP );
                this.queryName = queryElement.get( QueryElements.NAME );

                // tables
                if ( SharedUtil.isNotNull( queryElement.gets( QueryElements.TABLES ) ) )
                {
                    JSONArray tables = queryElement.gets( QueryElements.TABLES );
                    for ( int index = 0; index < tables.length(); index++ )
                    {
                        QueryElement.Table table = new QueryElement.Table( tables.get( index ) );

                        // base table data
                        String sourceId = table.get( QueryElements.Table.DATA_SOURCE_ID );
                        String database = table.get( QueryElements.Table.DATABASE );
                        String name = table.get( QueryElements.Table.NAME );
                        String alias = table.get( QueryElements.Table.ALIAS );
                        String type = table.get( QueryElements.Table.TYPE );

                        // table position
                        QueryElement.Table.Position tablePosition =
                            new QueryElement.Table.Position( table.get( QueryElements.Table.POSITION ) );
                        int left = tablePosition.get( QueryElements.Table.Position.LEFT );
                        int top = tablePosition.get( QueryElements.Table.Position.TOP );

                        // table data
                        addTable( left, top, sourceId, database, name, alias, type );
                    }
                }

                // table fields
                if ( SharedUtil.isNotNull( queryElement.gets( QueryElements.FIELDS ) ) )
                {
                    Map<Integer, Map<String, String>> clumns = new HashMap<>();
                    JSONArray fields = queryElement.gets( QueryElements.FIELDS );
                    for ( int index = 0; index < fields.length(); index++ )
                    {
                        String columnId = null;
                        Map<String, String> column = new HashMap<>();
                        QueryElement.Field field = new QueryElement.Field( fields.get( index ) );
                        for ( QueryElements.Field element : field.getElements() )
                        {
                            switch ( element )
                            {
                                case COLUMN:
                                    columnId = field.get( element );
                                    break;
                                case UNDEFINED:
                                    // nothing to do
                                    break;
                                default:
                                    column.put( element.toString(), field.get( element ) );
                                    break;
                            }
                        }
                        clumns.put( Integer.parseInt( columnId ), column );
                        addTableField( columnId, column.keySet()
                                                       .toArray(),
                                       column.values()
                                             .toArray() );
                    }
                }

                // table joins
                if ( SharedUtil.isNotNull( queryElement.gets( QueryElements.JOINS ) ) )
                {
                    JSONArray joins = queryElement.gets( QueryElements.JOINS );
                    for ( int index = 0; index < joins.length(); index++ )
                    {
                        QueryElement.Join join = new QueryElement.Join( joins.get( index ) );
                        TablesJoin tablesJoin = TablesJoin.getJoinFromString( join.get( QueryElements.Join.TYPE ) );
                        String fromTable = join.get( QueryElements.Join.FROM );
                        String[] fromFields = StringUtil.asStringArray( join.gets( QueryElements.Join.FROM_FIELDS ) );
                        String toTable = join.get( QueryElements.Join.TO );
                        String[] toFields = StringUtil.asStringArray( join.gets( QueryElements.Join.TO_FIELDS ) );

                        // save table joins
                        addTableJoins( fromTable, fromFields, toTable, toFields, tablesJoin );
                    }
                }
            }
            else
            {
                this.queryGroup = null;
                this.queryName = null;
                this.queryData = null;
                new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_BECAUSE_GROUP_AND_NAME_ARE_NOT_DEFINED ).show();
            }
        }
        catch ( JSONException e )
        {
            this.queryGroup = null;
            this.queryName = null;
            this.queryData = null;
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getLocalizedMessage(), e );
            new ErrorUtil( MessageFormat.format( ErrorUtil.JSON_QUERY_IMPORT, e.getLocalizedMessage() ) ).show();
        }

    }

    /**
     * Set common functions.
     * 
     * @param commonFunctions
     *            common functions
     */
    public void setCommonFunctions( CommonFunctions commonFunctions )
    {
        this.commonFunctions = commonFunctions;
    }

    /**
     * Set sorting functions.
     * 
     * @param sortingFunctions
     *            sorting functions
     */
    public void setSortingFunctions( SortingFunctions sortingFunctions )
    {
        this.sortingFunctions = sortingFunctions;
    }

    /**
     * Set total functions.
     * 
     * @param totalFunctions
     *            total functions
     */
    public void setTotalFunctions( TotalFunctions totalFunctions )
    {
        this.totalFunctions = totalFunctions;
    }

    /**
     * Create sql string from json query object.
     * 
     * @return sql string
     */
    public String getSqlString()
    {
        QueryElement queryElement = new QueryElement( queryData );
        SqlBuilder sqlBuilder = new SqlBuilder();
        sqlBuilder.setTableGrouping( false );

        // table columns
        JSONArray fields = queryElement.gets( QueryElements.FIELDS );
        if ( fields != QueryElement.EMPTY_JSON_ARRAY )
        {
            for ( int index = 0; index < fields.length(); index++ )
            {
                boolean saveColumn = false;
                SqlColumn sqlColumn = new SqlColumn();
                QueryElement.Field field = new QueryElement.Field( fields.get( index ) );

                for ( QueryElements.Field element : field.getElements() )
                {
                    switch ( element )
                    {
                        case COLUMN:
                            // nothing to do
                            break;
                        case COMMON:
                            if ( SharedUtil.isNotNull( commonFunctions ) )
                            {
                                String function = field.get( element );
                                sqlColumn.setCommonFunction( function, commonFunctions.getType( function ) );
                            }
                            break;
                        case FIELD:
                            sqlColumn.setColumn( field.get( element ) );
                            break;
                        case NAME:
                            sqlColumn.as( field.get( element ) );
                            break;
                        case SHOW:
                            saveColumn = Boolean.parseBoolean( field.get( element ) );
                            break;
                        case SORTING:
                            if ( SharedUtil.isNotNull( sortingFunctions ) )
                            {
                                sqlColumn.setOrderBy( field.get( element ) );
                            }
                            break;
                        case TABLE:
                            sqlColumn.from( field.get( element ) );
                            break;
                        case TOTAL:
                            sqlBuilder.setTableGrouping( true );
                            if ( SharedUtil.isNotNull( totalFunctions ) )
                            {
                                String function = field.get( element );
                                sqlColumn.setTotalFunction( function, totalFunctions.getType( function ) );
                            }
                            break;
                        case UNDEFINED:
                            // nothing to do
                            break;
                        default:
                            if ( SharedUtil.isInteger( field.get( element ) ) )
                            {
                                String whereCondition = field.get( element );
                                if ( SharedUtil.isNotNullAndNotEmpty( whereCondition ) )
                                {
                                    sqlBuilder.addColumnWhereCondition( field.get( QueryElements.Field.COLUMN ),
                                                                        element.toString(), sqlColumn.toShortString(),
                                                                        whereCondition );
                                }
                            }
                            break;
                    }
                }

                // save column
                if ( saveColumn )
                {
                    sqlBuilder.addColumn( sqlColumn );
                }
            }
        }

        return sqlBuilder.toString();
    }

    /**
     * Get query element object.
     * 
     * @param element
     *            query element
     * @return query element object
     */
    public Object get( QueryElements element )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( element ) )
        {
            switch ( element )
            {
                case FIELDS:
                case JOINS:
                case TABLES:
                    return new QueryElement( queryData ).gets( element );
                default:
                    return new QueryElement( queryData ).get( element );
            }
        }
        return StringUtil.EMPTY_VALUE;
    }

    /**
     * Get query's table element object.
     * 
     * @param element
     *            query's table element
     * @return query's table element object
     */
    public Object get( QueryElements.Table element )
    {
        return new QueryElement.Table( queryData ).get( element );
    }

    /**
     * Get query data.
     * 
     * @return query data
     */
    public JSONObject getQueryData()
    {
        return queryData;
    }

    /**
     * Add table.
     * 
     * @param left
     *            table left position
     * @param top
     *            table top position
     * @param dataSourceId
     *            data source id
     * @param database
     *            table database
     * @param tableName
     *            table name
     * @param tableAlias
     *            table alias
     * @param tableType
     *            tableType
     * @return json query object
     */
    public JsonQuery addTable( int left, int top, String dataSourceId, String database, String tableName,
                               String tableAlias, String tableType )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && SharedUtil.isNotNullAndNotEmpty( database )
            && SharedUtil.isNotNullAndNotEmpty( tableName ) && SharedUtil.isNotNullAndNotEmpty( tableAlias ) )
        {
            // table
            JSONObject table = new JSONObject();
            table.put( QueryElements.Table.DATA_SOURCE_ID.toString(), dataSourceId );
            table.put( QueryElements.Table.DATABASE.toString(), database );
            if ( tableName.contains( StringUtil.DOT_VALUE ) )
            {
                table.put( QueryElements.Table.NAME.toString(), tableName.split( StringUtil.STRING_DOT_SPLITTER )[1] );
            }
            else
            {
                table.put( QueryElements.Table.NAME.toString(), tableName );
            }
            table.put( QueryElements.Table.ALIAS.toString(), tableAlias );
            table.put( QueryElements.Table.TYPE.toString(), tableType );
            this.dataSourceId = dataSourceId;

            // table position
            JSONObject tablePosition = new JSONObject();
            tablePosition.put( QueryElements.Table.Position.LEFT.toString(), left );
            tablePosition.put( QueryElements.Table.Position.TOP.toString(), top );
            table.put( QueryElements.Table.POSITION.toString(), tablePosition );

            // tables
            JSONArray tables = new QueryElement( queryData ).gets( QueryElements.TABLES );
            if ( SharedUtil.isNull( tables ) )
            {
                tables = new JSONArray();
            }
            tables.put( table );

            // save query
            queryData.put( QueryElements.GROUP.toString(), queryGroup );
            queryData.put( QueryElements.NAME.toString(), queryName );
            queryData.put( QueryElements.DATA_SOURCE_ID.toString(), this.dataSourceId );
            queryData.put( QueryElements.TABLES.toString(), tables );
        }
        return this;
    }

    /**
     * Add table joins.
     * 
     * @param fromTable
     *            from table
     * @param fromFields
     *            from fields
     * @param toTable
     *            to table
     * @param toFields
     *            to fields
     * @param tablesJoin
     *            tables join type
     * @return json query object
     */
    public JsonQuery addTableJoins( String fromTable, String[] fromFields, String toTable, String[] toFields,
                                    TablesJoin tablesJoin )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( fromTable ) && SharedUtil.isNotNullAndNotEmpty( toTable )
            && SharedUtil.isNotNull( fromFields ) && SharedUtil.isNotNull( toFields ) && fromFields.length != 0
            && toFields.length != 0 )
        {
            // table join
            JSONObject join = new JSONObject();
            join.put( QueryElements.Join.FROM.toString(), fromTable );
            join.put( QueryElements.Join.FROM_FIELDS.toString(), fromFields );
            join.put( QueryElements.Join.TO.toString(), toTable );
            join.put( QueryElements.Join.TO_FIELDS.toString(), toFields );
            join.put( QueryElements.Join.TYPE.toString(), tablesJoin.toString() );

            // joins
            JSONArray joins = new QueryElement( queryData ).gets( QueryElements.JOINS );
            if ( SharedUtil.isNull( joins ) )
            {
                joins = new JSONArray();
            }
            joins.put( join );

            // save query
            queryData.put( QueryElements.GROUP.toString(), queryGroup );
            queryData.put( QueryElements.NAME.toString(), queryName );
            queryData.put( QueryElements.DATA_SOURCE_ID.toString(), dataSourceId );
            queryData.put( QueryElements.JOINS.toString(), joins );
        }
        return this;
    }

    /**
     * Add table fields.
     * 
     * @param columnId
     *            column id
     * @param rowIds
     *            row ids
     * @param rowValues
     *            row values
     * @return json query object
     */
    public JsonQuery addTableField( String columnId, Object[] rowIds, Object[] rowValues )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( columnId ) && SharedUtil.isNotNull( rowIds )
            && SharedUtil.isNotNull( rowValues ) && rowIds.length != 0 && rowValues.length != 0 )
        {
            // table fields
            JSONObject field = new JSONObject();
            for ( int index = 0; index < Math.min( rowIds.length, rowValues.length ); index++ )
            {
                if ( SharedUtil.isNotNullAndNotEmpty( rowIds[index] )
                    && SharedUtil.isNotNullAndNotEmpty( rowValues[index] ) )
                {
                    field.put( rowIds[index].toString(), rowValues[index] );
                }
            }
            field.put( QueryElements.Field.COLUMN.toString(), columnId );

            // fields
            JSONArray fields = new QueryElement( queryData ).gets( QueryElements.FIELDS );
            if ( SharedUtil.isNull( fields ) )
            {
                fields = new JSONArray();
            }
            fields.put( field );

            // save query
            queryData.put( QueryElements.GROUP.toString(), queryGroup );
            queryData.put( QueryElements.NAME.toString(), queryName );
            queryData.put( QueryElements.DATA_SOURCE_ID.toString(), dataSourceId );
            queryData.put( QueryElements.FIELDS.toString(), fields );
        }
        return this;
    }

    /**
     * Add query.
     * 
     * @param query
     *            query
     * @return json query object
     */
    public JsonQuery addQuery( String query )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( query ) )
        {
            // save query
            queryData.put( QueryElements.GROUP.toString(), queryGroup );
            queryData.put( QueryElements.NAME.toString(), queryName );
            queryData.put( QueryElements.DATA_SOURCE_ID.toString(), dataSourceId );
            queryData.put( QueryElements.QUERY.toString(), query );
        }
        return this;
    }

    /**
     * Return query data as string.
     */
    @Override
    public String toString()
    {
        return queryData.toString();
    }

}
