/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.utils;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.enums.SqlDriver;

public class JdbcUtil
    implements Serializable
{

    private static final long serialVersionUID = -7126855114873216472L;

    private static final String CATALOG = "database";

    private static final String SQL_STATE = "Sql state: {0}";

    private String description = null;

    private transient Connection connection = null;

    private SqlDriver driver = null;

    private String customDriver = null;

    private String catalog = null;

    private String[] tables = null;

    private String url = null;

    private String user = null;

    private String password = null;

    /**
     * Creates a new instance.
     */
    public JdbcUtil()
    {
        setDriver( SqlDriver.OFFLINE_MODE );
        setCatalog( CATALOG );
    }

    /**
     * Get description.
     * 
     * @return description
     */
    public String getDescription()
    {
        return description;
    }

    /**
     * Set description.
     * 
     * @param description
     *            description
     * @return JdbcUtil instance
     */
    public JdbcUtil setDescription( String description )
    {
        this.description = description;
        return this;
    }

    /**
     * Get sql driver.
     * 
     * @return sql driver
     */
    public SqlDriver getDriver()
    {
        return driver;
    }

    /**
     * Set sql driver.
     * 
     * @param driver
     *            sql driver
     * @return JdbcUtil instance
     */
    public JdbcUtil setDriver( SqlDriver driver )
    {
        this.driver = driver;
        return this;
    }

    /**
     * Get custom sql driver.
     * 
     * @return custom sql driver
     */
    public String getCustomDriver()
    {
        return customDriver;
    }

    /**
     * Set custom sql driver.
     * 
     * @param customDriver
     *            custom sql driver
     * @return JdbcUtil instance
     */
    public JdbcUtil setCustomDriver( String customDriver )
    {
        this.customDriver = customDriver;
        return this;
    }

    /**
     * Get catalog (only for offline mode).
     * 
     * @return catalog (only for offline mode)
     */
    public String getCatalog()
    {
        return catalog;
    }

    /**
     * Set catalog (only for offline mode).
     * 
     * @param catalog
     *            catalog (only for offline mode)
     * @return JdbcUtil instance
     */
    public JdbcUtil setCatalog( String catalog )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( catalog ) )
        {
            this.catalog = catalog;
        }
        return this;
    }

    /**
     * Get tables in database (only for offline mode).
     * 
     * @return tables in database (only for offline mode)
     */
    public String[] getTables()
    {
        if ( SharedUtil.isNotNull( tables ) && 0 < tables.length )
        {
            return tables;
        }
        return new String[] {};
    }

    /**
     * Set tables for database (only for offline mode).
     * 
     * @param tables
     *            tables for database (only for offline mode)
     * @return JdbcUtil instance
     */
    public JdbcUtil setTables( String... tables )
    {
        this.tables = tables;
        return this;
    }

    /**
     * Get database url.
     * 
     * @return database url
     */
    public String getUrl()
    {
        return url;
    }

    /**
     * Set database url.
     * 
     * @param url
     *            database url
     * @return JdbcUtil instance
     */
    public JdbcUtil setUrl( String url )
    {
        this.url = url;
        return this;
    }

    /**
     * Get database user.
     * 
     * @return database user
     */
    public String getUser()
    {
        return user;
    }

    /**
     * Set database user.
     * 
     * @param user
     *            database user
     * @return JdbcUtil instance
     */
    public JdbcUtil setUser( String user )
    {
        this.user = user;
        return this;
    }

    /**
     * Get database user password.
     * 
     * @return database user password
     */
    public String getPassword()
    {
        return password;
    }

    /**
     * Set database user password.
     * 
     * @param password
     *            database user password
     * @return JdbcUtil instance
     */
    public JdbcUtil setPassword( String password )
    {
        this.password = password;
        return this;
    }

    /**
     * Get connection instance.
     * 
     * @return connection instance
     */
    public Connection getConnection()
    {
        try
        {
            if ( connection == null || connection.isClosed() )
            {
                if ( SharedUtil.isNotNullAndNotEmpty( getCustomDriver() ) )
                {
                    Class.forName( getCustomDriver() )
                         .newInstance();
                }
                else
                {
                    Class.forName( getDriver().toString() )
                         .newInstance();
                }
                connection = DriverManager.getConnection( getUrl(), getUser(), getPassword() );
            }
        }
        catch ( InstantiationException | IllegalAccessException | ClassNotFoundException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getMessage(), e );
            new ErrorUtil( MessageFormat.format( ErrorUtil.CANNOT_CONNECT_TO_DATABASE,
                                                 e.getLocalizedMessage() ) ).show();
        }
        catch ( SQLException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getMessage(), e );
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, MessageFormat.format( SQL_STATE, e.getSQLState() ), e );
            new ErrorUtil( MessageFormat.format( ErrorUtil.CANNOT_CONNECT_TO_DATABASE,
                                                 e.getLocalizedMessage() ) ).show();
        }

        return connection;
    }

    /**
     * Close connection instance.
     */
    public void close()
    {
        try
        {
            if ( connection != null && !connection.isClosed() )
            {
                connection.close();
            }
        }
        catch ( SQLException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getMessage(), e );
        }
        connection = null;
    }

}
