/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.ui;

import java.text.MessageFormat;

import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.ui.DroppedTableContainer;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.event.dd.DropHandler;
import com.vaadin.server.FontAwesome;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.Panel;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.TableDragMode;
import com.vaadin.ui.VerticalLayout;
import com.vaadin.ui.themes.ValoTheme;

public class DroppedTable
    extends Panel
{

    private static final long serialVersionUID = 7910897057453111578L;

    private static final String TABLE_POSITION = "{0}:{1}";

    public static final String BUTTON_DATA = "View table data";

    public static final String BUTTON_QUERY = "View/change query definition";

    public static final String BUTTON_REMOVE = "Remove table";

    private boolean showTableData = true;

    private boolean showQueryDefinition = true;

    private String caption = null;

    private Table table = null;

    private Button dataButton = null;

    private Button queryButton = null;

    private Button removeButton = null;

    /**
     * Creates a new instance.
     */
    public DroppedTable()
    {
        table = new Table();
        table.addContainerProperty( StringUtil.EMPTY_VALUE, String.class, null );
        table.setWidth( 100, Unit.PERCENTAGE );
        table.setDragMode( TableDragMode.MULTIROW );
        table.setMultiSelect( true );
        table.setSelectable( true );
        table.addStyleName( ValoTheme.TABLE_NO_HEADER );
        table.addStyleName( ValoTheme.TABLE_COMPACT );
        table.addStyleName( ValoTheme.TABLE_SMALL );
    }

    /**
     * Check if can show table data.
     * 
     * @return true if can show table data, otherwise false
     */
    public boolean isShowTableDataEnabled()
    {
        return showTableData;
    }

    /**
     * Show table data.
     * 
     * @param showTableData
     *            if true table data is visible, otherwise not
     */
    public void setShowTableDataEnabled( boolean showTableData )
    {
        this.showTableData = showTableData;
        if ( SharedUtil.isNotNull( dataButton ) )
        {
            dataButton.setVisible( showTableData );
        }
    }

    /**
     * Check if can show query definition.
     * 
     * @return true if can show query definition, otherwise false
     */
    public boolean isShowQueryDefinitionEnabled()
    {
        return showQueryDefinition;
    }

    /**
     * Show query definition.
     * 
     * @param showQueryDefinition
     *            if true query definition is visible, otherwise not
     */
    public void setShowQueryDefinitionEnabled( boolean showQueryDefinition )
    {
        this.showQueryDefinition = showQueryDefinition;
        if ( SharedUtil.isNotNull( queryButton ) )
        {
            queryButton.setVisible( showQueryDefinition );
        }
    }

    /**
     * Prepare dropped table.
     * 
     * @param dataSource
     *            data source
     * @param dataSourceId
     *            id of the data source
     * @param databaseName
     *            database name
     * @param tableName
     *            table name
     * @param tableAlias
     *            table alias
     * @param tableType
     *            table type: table or query
     */
    public void prepare( JdbcUtil dataSource, String dataSourceId, String databaseName, String tableName,
                         String tableAlias, String tableColumns, String tableType )
    {
        String itemId;
        String[] columns = StringUtil.asStringArray( tableColumns );
        for ( String column : columns )
        {
            // add item for all fields (char '*')
            if ( 1 < columns.length && table.getItemIds()
                                            .size() == 0 )
            {
                // TABLE_ROW_ID format: database source id, database name, table name, column name, table alias
                itemId = MessageFormat.format( DroppedTableContainer.TABLE_ROW_ID, dataSourceId, databaseName,
                                               tableName, BaseTable.FIELDS_ALL, tableAlias );
                table.addItem( new Object[] { BaseTable.FIELDS_ALL }, itemId );
            }

            // TABLE_ROW_ID format: database source id, database name, table name, column name, table alias
            itemId = MessageFormat.format( DroppedTableContainer.TABLE_ROW_ID, dataSourceId, databaseName, tableName,
                                           column, tableAlias );
            table.addItem( new Object[] { column }, itemId );
        }
        table.setPageLength( 0 );

        // drop table into dropped tables context layer
        if ( 0 < table.getItemIds()
                      .size() )
        {
            // TABLE_DATA_ID format: database source id, database name, table name, table alias, table type
            String data = MessageFormat.format( DroppedTableContainer.TABLE_DATA_ID, dataSourceId, databaseName,
                                                tableName, tableAlias, tableType );

            caption = tableAlias;
            table.setData( data );

            dataButton = new Button();
            dataButton.setCaption( BUTTON_DATA );
            dataButton.setData( data );
            dataButton.setDescription( BUTTON_DATA );
            dataButton.setIcon( FontAwesome.TABLE );
            dataButton.setVisible( showTableData );
            dataButton.addStyleName( DroppedTableContainer.TABLE_DROP_BUTTON );
            dataButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
            dataButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );

            queryButton = new Button();
            queryButton.setCaption( BUTTON_QUERY );
            queryButton.setData( data );
            queryButton.setDescription( BUTTON_QUERY );
            queryButton.setIcon( FontAwesome.PENCIL_SQUARE_O );
            queryButton.setVisible( showQueryDefinition );
            queryButton.addStyleName( DroppedTableContainer.TABLE_DROP_BUTTON );
            queryButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
            queryButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );

            removeButton = new Button();
            removeButton.setCaption( BUTTON_REMOVE );
            removeButton.setData( tableAlias );
            removeButton.setDescription( BUTTON_REMOVE );
            removeButton.setIcon( FontAwesome.REMOVE );
            removeButton.addStyleName( DroppedTableContainer.TABLE_DROP_BUTTON_LAST );
            removeButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
            removeButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );

            Label captionLabel = new Label( caption );
            captionLabel.addStyleName( DroppedTableContainer.TABLE_DROP_LABEL );

            HorizontalLayout captionLayout =
                new HorizontalLayout( captionLabel, queryButton, dataButton, removeButton );
            captionLayout.addStyleName( DroppedTableContainer.TABLE_DROP_CAPTION );
            captionLayout.addStyleName( ValoTheme.PANEL_BORDERLESS );

            VerticalLayout layout = new VerticalLayout( captionLayout, table );
            layout.setSizeUndefined();

            setContent( layout );
            setSizeUndefined();
            addStyleName( DroppedTableContainer.TABLE_DROP );
            addStyleName( caption );
        }
    }

    /**
     * Get caption.
     */
    @Override
    public String getCaption()
    {
        return caption;
    }

    /**
     * Add button click listener.
     * 
     * @param clickListener
     *            button click listener
     */
    public void addClickListener( ClickListener clickListener )
    {
        queryButton.addClickListener( clickListener );
        dataButton.addClickListener( clickListener );
        removeButton.addClickListener( clickListener );
    }

    /**
     * Add item click listener.
     * 
     * @param itemClickListener
     *            item click listener
     */

    public void addItemClickListener( ItemClickListener itemClickListener )
    {
        table.addItemClickListener( itemClickListener );
    }

    public void addDropHandler( DropHandler dropHandler )
    {
        table.setDropHandler( dropHandler );
    }

    /**
     * Get data source id.
     * 
     * @return data source id
     */
    public String getDataSourceId()
    {
        Object data = table.getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_DOT_SPLITTER );
            if ( 1 <= tableData.length )
            {
                return tableData[0];
            }
        }
        return null;
    }

    /**
     * Get database name.
     * 
     * @return database name
     */
    public String getDatabaseName()
    {
        Object data = table.getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_DOT_SPLITTER );
            if ( 2 <= tableData.length )
            {
                return tableData[1];
            }
        }
        return null;
    }

    /**
     * Set dropped table position.
     * 
     * @param left
     *            left position of the dropped table
     * @param top
     *            top position of the dropped table
     */
    public void setTablePosition( Integer left, Integer top )
    {
        setData( MessageFormat.format( TABLE_POSITION, left.toString(), top.toString() ) );
    }

    /**
     * Get dropped table left position.
     * 
     * @return dropped table left position
     */
    public Integer getLeft()
    {
        Object data = getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_COLON_SPLITTER );
            if ( 2 <= tableData.length )
            {
                return Integer.parseInt( tableData[0].replace( StringUtil.COMMA_VALUE, StringUtil.EMPTY_VALUE )
                                                     .replace( StringUtil.DOT_VALUE, StringUtil.EMPTY_VALUE ) );
            }
        }
        return null;
    }

    /**
     * Get dropped table top position.
     * 
     * @return dropped table top position
     */
    public Integer getTop()
    {
        Object data = getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_COLON_SPLITTER );
            if ( 2 <= tableData.length )
            {
                return Integer.parseInt( tableData[1].replace( StringUtil.COMMA_VALUE, StringUtil.EMPTY_VALUE )
                                                     .replace( StringUtil.DOT_VALUE, StringUtil.EMPTY_VALUE ) );
            }
        }
        return null;
    }

    /**
     * Get table name.
     * 
     * @return table name
     */
    public String getTableName()
    {
        Object data = table.getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_DOT_SPLITTER );
            if ( 3 <= tableData.length )
            {
                return MessageFormat.format( DroppedTableContainer.TABLE_DATA_NAME, tableData[1], tableData[2] );
            }
        }
        return null;
    }

    /**
     * Get table alias.
     * 
     * @return table alias
     */
    public String getTableAlias()
    {
        Object data = table.getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_DOT_SPLITTER );
            if ( 4 <= tableData.length )
            {
                return tableData[3];
            }
        }
        return null;
    }

    /**
     * Get table type.
     * 
     * @return table type
     */
    public TableType getTableType()
    {
        Object data = table.getData();
        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
        {
            String[] tableData = data.toString()
                                     .split( StringUtil.STRING_DOT_SPLITTER );
            if ( 5 <= tableData.length )
            {
                return TableType.getType( tableData[4] );
            }
        }
        return null;
    }

}
