/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery;

import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dmenu.events.MenuItemElementClickEvent;
import org.dussan.vaadin.dmenu.events.MenuItemElementClickHandler;
import org.dussan.vaadin.dmenu.menuitem.MenuItem;
import org.dussan.vaadin.dmenu.menuitem.MenuItemElement;
import org.dussan.vaadin.dquery.base.QueryElement;
import org.dussan.vaadin.dquery.base.sql.builder.SqlColumn;
import org.dussan.vaadin.dquery.base.sql.builder.SqlJoin;
import org.dussan.vaadin.dquery.base.sql.builder.SqlTable;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.base.ui.DroppedTable;
import org.dussan.vaadin.dquery.base.ui.JoinedTables;
import org.dussan.vaadin.dquery.client.rpc.QueryServerRpc;
import org.dussan.vaadin.dquery.component.DQueryComponent;
import org.dussan.vaadin.dquery.enums.QueryElements;
import org.dussan.vaadin.dquery.enums.SqlDriver;
import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.events.GroupQueryChangeEvent;
import org.dussan.vaadin.dquery.events.GroupQueryChangeHandler;
import org.dussan.vaadin.dquery.events.QueryChangeEvent;
import org.dussan.vaadin.dquery.events.QueryChangeHandler;
import org.dussan.vaadin.dquery.events.TableContainerDropEvent;
import org.dussan.vaadin.dquery.events.TableContainerDropHandler;
import org.dussan.vaadin.dquery.json.JsonQuery;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.ui.DroppedTableContainer;
import org.dussan.vaadin.dquery.ui.SqlMenu;
import org.dussan.vaadin.dquery.ui.windows.JoinedTablesWindow;
import org.dussan.vaadin.dquery.ui.windows.MessageWindow;
import org.dussan.vaadin.dquery.ui.windows.SqlFunctionsWindow;
import org.dussan.vaadin.dquery.ui.windows.SqlParametersWindow;
import org.dussan.vaadin.dquery.ui.windows.TableDataWindow;
import org.dussan.vaadin.dquery.ui.windows.TestQueryWindow;
import org.dussan.vaadin.dquery.utils.ErrorUtil;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONException;

import com.google.gwt.event.shared.HandlerManager;
import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.event.DataBoundTransferable;
import com.vaadin.event.FieldEvents.BlurEvent;
import com.vaadin.event.FieldEvents.BlurListener;
import com.vaadin.event.FieldEvents.FocusEvent;
import com.vaadin.event.FieldEvents.FocusListener;
import com.vaadin.event.FieldEvents.TextChangeEvent;
import com.vaadin.event.FieldEvents.TextChangeListener;
import com.vaadin.event.ItemClickEvent;
import com.vaadin.event.ItemClickEvent.ItemClickListener;
import com.vaadin.event.dd.DragAndDropEvent;
import com.vaadin.event.dd.DropHandler;
import com.vaadin.event.dd.DropTarget;
import com.vaadin.event.dd.acceptcriteria.AcceptAll;
import com.vaadin.event.dd.acceptcriteria.AcceptCriterion;
import com.vaadin.shared.MouseEventDetails;
import com.vaadin.ui.AbstractSelect.AbstractSelectTargetDetails;
import com.vaadin.ui.Alignment;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.Component;
import com.vaadin.ui.DragAndDropWrapper;
import com.vaadin.ui.DragAndDropWrapper.WrapperTargetDetails;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Label;
import com.vaadin.ui.ProgressBar;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.TableTransferable;
import com.vaadin.ui.TextField;
import com.vaadin.ui.UI;
import com.vaadin.ui.VerticalLayout;

public class DQuery
    extends DQueryComponent

{

    private static final long serialVersionUID = 8421686217816942853L;

    private static final int INIT_COMPLETED = 100;

    private static final int INIT_DATA_SOURCE_MENU = 0;

    private static final int INIT_DROPPED_TABLE_CONTAINER = 15;

    private static final int INIT_SQL_MENU = 30;

    private static final int INIT_SQL_TABS_HIDE_DESCRIPTION_AND_FIELD_ROWS = 50;

    private static final int INIT_SQL_TABS_SHOW_DESCRIPTION_AND_FIELD_ROWS = 75;

    private static final int INIT_STARTED = 0;

    private static final int INIT_STOPPED = 101;

    private boolean fireEvents = true;

    private boolean showDroppedTableData = true;

    private boolean showQueryDefinition = true;

    private int addedTableLeftPosition = 0;

    private int addedTableTopPosition = 0;

    private long currentTime = 0;

    private HandlerManager handlerManager = null;

    private ProgressBar progressStatus = null;

    private VerticalLayout progressStatusLayout = null;

    /**
     * Creates a new instance.
     */
    public DQuery()
    {
        // events handler manager
        handlerManager = new HandlerManager( this );

        // register RPC
        registerRpc( new QueryServerRpc()
        {
            private static final long serialVersionUID = -7489427457610395747L;

            @Override
            public void changeJoinedTables( String joinedTablesId, Integer joinedFieldsIndex )
            {
                UI.getCurrent()
                  .addWindow( new JoinedTablesWindow( joinedTablesId, joinedFieldsIndex )
                  {
                      private static final long serialVersionUID = -7082985030501860887L;

                      @Override
                      protected void setJoinedTablesJoinType( String fromTable, String toTable, TablesJoin tablesJoin,
                                                              int joinedFieldsIndex )
                      {
                          if ( tablesJoin.equals( TablesJoin.REMOVE ) )
                          {
                              getDroppedTableContainer().removeJoinedTablesJoin( fromTable, toTable,
                                                                                 joinedFieldsIndex );
                          }
                          else
                          {
                              getDroppedTableContainer().setJoinedTablesJoinType( fromTable, toTable, tablesJoin );
                          }
                          getThis().getState( true )
                                   .setJoinedTables( getDroppedTableContainer().getJoinedTablesAsStringList() );

                          // enable clear table container
                          getSqlMenu().setClearButtonEnabled( true );

                          // enable/disable save query
                          getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                          // fire events
                          fireEvents();
                      }
                  } );
            }

            @Override
            public void initProgress( Integer initProgress )
            {
                int progress = INIT_STOPPED;
                switch ( initProgress )
                {
                    case INIT_DROPPED_TABLE_CONTAINER:
                        getDroppedTableContainer().setVisible( true );
                        progress = INIT_SQL_MENU;
                        break;
                    case INIT_DATA_SOURCE_MENU:
                        getDataSourcesMenu().setVisible( true );
                        progress = INIT_DROPPED_TABLE_CONTAINER;
                        break;
                    case INIT_SQL_MENU:
                        getSqlMenu().setVisible( true );
                        progress = INIT_SQL_TABS_HIDE_DESCRIPTION_AND_FIELD_ROWS;
                        break;
                    case INIT_SQL_TABS_HIDE_DESCRIPTION_AND_FIELD_ROWS:
                        getSqlTabs().setVisible( true );
                        progress = INIT_SQL_TABS_SHOW_DESCRIPTION_AND_FIELD_ROWS;
                        break;
                    case INIT_SQL_TABS_SHOW_DESCRIPTION_AND_FIELD_ROWS:
                        getSqlTabs().showDescriptionAndFieldsRows( true );
                        progress = INIT_COMPLETED;
                        break;
                    case INIT_COMPLETED:
                    default:
                        break;
                }

                if ( progress != INIT_STOPPED )
                {
                    setProgressStatus( progress );
                    getThis().getState( true )
                             .setInitProgress( progress );
                }
                else
                {
                    deleteProgressStatusLayout();
                }
            }

            @Override
            public void tableMoved( String movedTableId, Integer left, Integer top )
            {
                if ( SharedUtil.isNotNullAndNotEmpty( movedTableId ) )
                {
                    for ( Iterator<Component> iterator = getDroppedTableContainer().getContainer()
                                                                                   .iterator(); iterator.hasNext(); )
                    {
                        Component component = iterator.next();
                        if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
                        {
                            DroppedTable droppedTable = (DroppedTable) component;
                            if ( droppedTable.getCaption()
                                             .equals( movedTableId ) )
                            {
                                droppedTable.setTablePosition( left, top );

                                // enable/disable save query
                                getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // fire events
                                fireEvents();
                                break;
                            }
                        }
                    }
                }
            }
        } );

        // set added table start position
        setAddedTablePosition( 50, 50 );

        // prepare dQuery
        prepare();
    }

    /**
     * Get dQuery instance.
     * 
     * @return dQuery instance
     */
    private DQuery getThis()
    {
        return this;
    }

    /**
     * Fire events.
     */
    private void fireEvents()
    {
        if ( fireEvents && 1000 < ( System.currentTimeMillis() - currentTime ) )
        {
            // refresh sql editor
            refreshSqlEditorContext();

            // fire events
            handlerManager.fireEvent( new QueryChangeEvent( getJsonBuilder().getQueries(), getCurrentQuery( true ) ) );
            handlerManager.fireEvent( new GroupQueryChangeEvent( getSqlMenu().getQueryGroup(),
                                                                 getSqlMenu().getQueryName(),
                                                                 getJsonBuilder().getGroupQueries( getSqlMenu().getQueryGroup() ) ) );

            // save current time of fired events
            currentTime = System.currentTimeMillis();
        }
    }

    /**
     * Set header row visible.
     * 
     * @param rowId
     *            row id
     * @param visible
     *            if true header row is visible, otherwise is hidden
     */
    private void setHeaderRowVisible( String rowId, boolean visible )
    {
        // button activity
        switch ( QueryElements.Field.get( rowId ) )
        {
            case COMMON:
                getSqlMenu().setButtonActive( SqlMenu.BUTTON_COMMON, visible );
                break;
            case SORTING:
                getSqlMenu().setButtonActive( SqlMenu.BUTTON_SORTING, visible );
                break;
            case TOTAL:
                getSqlMenu().setButtonActive( SqlMenu.BUTTON_TOTAL, visible );
                break;
            default:
                break;
        }

        // header row visibility
        getSqlTabs().setHeaderRowVisible( rowId, visible );
        getState( true ).setInvisibleHeaderRows( getSqlTabs().getInvisibleHeaderRows() );
        markAsDirty();
    }

    /**
     * Refresh sql editor context.
     */
    private void refreshSqlEditorContext()
    {
        // sql builder
        resetSqlBuilder();

        // table columns
        for ( Object column : getSqlTabs().getHeaderFieldsTable()
                                          .getVisibleColumns() )
        {
            String columnId = column.toString();
            if ( !BaseTable.COLUMN_LAST_FINAL.equals( column ) && !getSqlTabs().isFieldColumnEmpty( columnId ) )
            {
                SqlColumn sqlColumn = new SqlColumn();
                sqlColumn.setColumn( getSqlTabs().getHeaderFieldsTable()
                                                 .getCellValue( columnId, BaseTable.ID_FIELD ) )
                         .as( getSqlTabs().getHeaderFieldsTable()
                                          .getCellValue( columnId, BaseTable.ID_NAME ) )
                         .from( getSqlTabs().getHeaderFieldsTable()
                                            .getCellValue( columnId, BaseTable.ID_TABLE ) );

                // common function
                if ( getSqlTabs().isHeaderRowVisible( BaseTable.ID_COMMON ) )
                {
                    String function = getSqlTabs().getHeaderFieldsTable()
                                                  .getCellValue( columnId, BaseTable.ID_COMMON );
                    if ( SharedUtil.isNotNullAndNotEmpty( function ) )
                    {
                        sqlColumn.setCommonFunction( function, getSqlTabs().getCommonFunctions()
                                                                           .getType( function ) );
                    }
                }

                // total function
                if ( getSqlTabs().isHeaderRowVisible( BaseTable.ID_TOTAL ) )
                {
                    String function = getSqlTabs().getHeaderFieldsTable()
                                                  .getCellValue( columnId, BaseTable.ID_TOTAL );
                    if ( SharedUtil.isNotNullAndNotEmpty( function ) )
                    {
                        sqlColumn.setTotalFunction( function, getSqlTabs().getTotalFunctions()
                                                                          .getType( function ) );
                    }
                }

                // sorting
                if ( getSqlTabs().isHeaderRowVisible( BaseTable.ID_SORTING ) )
                {
                    String sorting = getSqlTabs().getHeaderFieldsTable()
                                                 .getCellValue( columnId, BaseTable.ID_SORTING );
                    if ( SharedUtil.isNotNullAndNotEmpty( sorting ) )
                    {
                        sqlColumn.setOrderBy( sorting );
                    }
                }

                // where conditions
                getSqlBuilder().setTableGrouping( getSqlTabs().isHeaderRowVisible( BaseTable.ID_TOTAL ) );
                if ( SharedUtil.isNotNullAndNotEmpty( sqlColumn.toShortString() ) && !sqlColumn.toShortString()
                                                                                               .contains( BaseTable.FIELDS_ALL ) )
                {
                    for ( Object rowId : getSqlTabs().getCriteriaFieldsTable()
                                                     .getItemIds() )
                    {
                        String whereCondition = getSqlTabs().getCriteriaFieldsTable()
                                                            .getCellValue( columnId, rowId.toString() );
                        if ( SharedUtil.isNotNullAndNotEmpty( whereCondition ) )
                        {
                            getSqlBuilder().addColumnWhereCondition( columnId, rowId.toString(),
                                                                     sqlColumn.toShortString(), whereCondition );
                        }
                    }
                }

                // save column
                boolean saveColumn = Boolean.parseBoolean( getSqlTabs().getHeaderFieldsTable()
                                                                       .getCellValue( columnId, BaseTable.ID_SHOW ) );
                if ( saveColumn )
                {
                    getSqlBuilder().addColumn( sqlColumn );
                }
            }
        }

        // table and its alias
        for ( Iterator<Component> iterator = getDroppedTableContainer().getContainer()
                                                                       .iterator(); iterator.hasNext(); )
        {
            Component component = iterator.next();
            if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
            {
                DroppedTable table = (DroppedTable) component;
                String tableName = table.getTableName();

                // check if table is query element form json query object
                if ( tableName.contains( StringUtil.DOT_VALUE )
                    && tableName.split( StringUtil.STRING_DOT_SPLITTER ).length == 2 )
                {
                    String queryGroup = tableName.split( StringUtil.STRING_DOT_SPLITTER )[0];
                    String queryName = tableName.split( StringUtil.STRING_DOT_SPLITTER )[1];
                    if ( getJsonBuilder().getQueries()
                                         .containsKey( queryGroup ) )
                    {
                        JsonQuery jsonQuery = getJsonBuilder().getQuery( queryGroup, queryName );
                        if ( SharedUtil.isNotNull( jsonQuery ) )
                        {
                            getSqlBuilder().setDataSourceId( jsonQuery.get( QueryElements.DATA_SOURCE_ID )
                                                                      .toString() );
                            tableName = jsonQuery.get( QueryElements.QUERY_AS_COLUMN )
                                                 .toString();
                            if ( tableName.equals( MessageFormat.format( JsonQuery.QUERY_AS_COLUMN,
                                                                         StringUtil.EMPTY_VALUE ) ) )
                            {
                                tableName = getJsonBuilder().getSqlQueryAsColumn( queryGroup, queryName );
                            }
                        }
                    }
                }

                SqlTable sqlTable = new SqlTable();
                sqlTable.setTableName( tableName )
                        .as( table.getTableAlias() );
                getSqlBuilder().addTable( sqlTable );
            }
        }

        // table joins
        for ( JoinedTables joinedTables : getDroppedTableContainer().getJoinedTables() )
        {
            SqlTable fromTable = new SqlTable().as( joinedTables.getFromTable() );
            SqlTable toTable = new SqlTable().as( joinedTables.getToTable() );
            for ( SqlTable table : getSqlBuilder().getTables() )
            {
                if ( table.getTableAlias()
                          .equals( fromTable.getTableAlias() ) )
                {
                    fromTable = new SqlTable( table.getTableName(), table.getTableAlias() );
                }
                if ( table.getTableAlias()
                          .equals( toTable.getTableAlias() ) )
                {
                    toTable = new SqlTable( table.getTableName(), table.getTableAlias() );
                }
            }

            SqlJoin sqlJoin = new SqlJoin( fromTable, toTable, joinedTables.getTablesJoin() );
            for ( int index = 0; index < Math.min( joinedTables.getFromFields().length,
                                                   joinedTables.getToFields().length ); index++ )
            {
                sqlJoin.setFromToColumn( joinedTables.getFromFields()[index], joinedTables.getToFields()[index] );
            }
            getSqlBuilder().addJoin( sqlJoin );
        }

        // sql editor context
        getSqlTabs().getSqlQueryEditor()
                    .setReadOnly( false );
        getSqlTabs().getSqlQueryEditor()
                    .setValue( getSqlBuilder().toString() );
        getSqlTabs().getSqlQueryEditor()
                    .setCursorRowCol( 0, 0 );
        getSqlTabs().getSqlQueryEditor()
                    .setReadOnly( true );

        // test current query
        getSqlMenu().setQueryButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() && !getSqlTabs().getSqlQueryEditor()
                                                                                                   .isEmpty() );
    }

    /**
     * Drop table.
     * 
     * @param dataSourceId
     *            id of the data source for the dropped table
     * @param databaseName
     *            database name for the dropped table
     * @param tableName
     *            table name for the dropped table
     * @param tableColumns
     *            table columns
     * @param tableType
     *            table type: table or query
     * @param left
     *            dropped table left position
     * @param top
     *            dropped table top position
     */
    private void dropTable( String dataSourceId, String databaseName, String tableName, String tableColumns,
                            String tableType, Integer left, Integer top )
    {
        // check if added query is itself definition
        if ( MessageFormat.format( BaseTable.TABLE_NAME, databaseName, tableName )
                          .equals( MessageFormat.format( BaseTable.TABLE_NAME, getSqlMenu().getQueryGroup(),
                                                         getSqlMenu().getQueryName() ) ) )
        {
            new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_TO_ITSELF ).show();
            return;
        }

        JdbcUtil dataSource = getDataSource( dataSourceId );
        if ( SharedUtil.isNotNull( dataSource ) )
        {
            DroppedTable droppedTable = new DroppedTable();
            droppedTable.prepare( dataSource, dataSourceId, databaseName, tableName,
                                  MessageFormat.format( DroppedTableContainer.TABLE_DROP_NAME, tableName,
                                                        getDroppedTableContainer().getSameNameIndex( tableName ) ),
                                  tableColumns, tableType );
            droppedTable.setShowTableDataEnabled( isShowDroppedTableDataEnabled( dataSourceId )
                && !SqlDriver.OFFLINE_MODE.equals( dataSource.getDriver() ) );
            droppedTable.setShowQueryDefinitionEnabled( showQueryDefinition
                && tableType.equals( TableType.QUERY.toString() ) && getJsonBuilder().getQueries()
                                                                                     .containsKey( databaseName ) );

            // add table field to tabs header field table
            droppedTable.addItemClickListener( new ItemClickListener()
            {
                private static final long serialVersionUID = -2054759352120240782L;

                @Override
                public void itemClick( ItemClickEvent event )
                {
                    if ( event.isDoubleClick() && SharedUtil.isNotNullAndNotEmpty( event.getItemId() ) )
                    {
                        fireEvents = false;

                        // insert new field column
                        // format: database source id, database name, table name, column name, table alias
                        String itemId = event.getItemId()
                                             .toString();
                        String[] data = itemId.split( StringUtil.STRING_DOT_SPLITTER );
                        if ( 5 <= data.length )
                        {
                            String newColumnId = getSqlTabs().addFieldColumn( data[4], data[3], data[3], true );
                            String emptyColumnId = getSqlTabs().getFirstEmptyFieldColumn();

                            if ( SharedUtil.isNotNullAndNotEmpty( newColumnId ) )
                            {
                                // new field column is added to the fields table
                                if ( SharedUtil.isNotNullAndNotEmpty( emptyColumnId ) )
                                {
                                    getSqlTabs().reverseFieldColumns( newColumnId, emptyColumnId );
                                    getSqlTabs().addMissingFieldColumns( BaseTable.COLUMNS_TO_ADD );
                                }

                                // refresh sql editor
                                refreshSqlEditorContext();

                                // unselect row in dropped table
                                ( (Table) event.getComponent() ).unselect( itemId );

                                // enable clear table container
                                getSqlMenu().setClearButtonEnabled( true );

                                // enable/disable save query
                                getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // enable/disable function on group of fields
                                getSqlMenu().setGroupButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // refresh sql editor
                                fireEvents = true;

                                // fire events
                                fireEvents();
                            }
                            else
                            {
                                new ErrorUtil( ErrorUtil.CANNOT_ADD_NEW_COLUMN ).show();
                            }
                        }
                        fireEvents = true;
                    }
                }
            } );

            // remove table from table container
            droppedTable.addClickListener( new ClickListener()
            {
                private static final long serialVersionUID = -1948012368378356027L;

                @Override
                public void buttonClick( ClickEvent event )
                {
                    if ( SharedUtil.isNotNull( event.getButton() ) && SharedUtil.isNotNullAndNotEmpty( event.getButton()
                                                                                                            .getData() ) )
                    {
                        Button button = event.getButton();
                        String tableData = button.getData()
                                                 .toString();
                        switch ( button.getCaption() )
                        {
                            case DroppedTable.BUTTON_DATA:
                                if ( tableData.contains( StringUtil.DOT_VALUE )
                                    && 4 <= tableData.split( StringUtil.STRING_DOT_SPLITTER ).length )
                                {
                                    // format: database source id, database name, table name, table alias
                                    String dataSourceId = tableData.split( StringUtil.STRING_DOT_SPLITTER )[0];
                                    String database = tableData.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                    String table = tableData.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                    String tableAlias = tableData.split( StringUtil.STRING_DOT_SPLITTER )[3];

                                    // data query: SELECT * FROM database.table
                                    String query = MessageFormat.format( TableDataWindow.ROW_SELECT, database, table );

                                    // check if table is query element form json query object
                                    // queryGroup is database, queryName is table, data query is from saved query
                                    if ( getJsonBuilder().getQueries()
                                                         .containsKey( database ) )
                                    {
                                        JsonQuery jsonQuery = getJsonBuilder().getQuery( database, table );
                                        if ( SharedUtil.isNotNull( jsonQuery ) )
                                        {
                                            query = jsonQuery.get( QueryElements.QUERY )
                                                             .toString();
                                            if ( query.isEmpty() )
                                            {
                                                query = getJsonBuilder().getSqlQuery( database, table );
                                            }
                                        }
                                    }

                                    // show table data window
                                    if ( getDataSources().containsKey( dataSourceId )
                                        && SharedUtil.isNotNull( getDataSources().get( dataSourceId ) ) )
                                    {
                                        UI.getCurrent()
                                          .addWindow( new TableDataWindow( getDataSources().get( dataSourceId ),
                                                                           tableAlias, query ) );
                                    }
                                    else
                                    {
                                        new ErrorUtil( ErrorUtil.CANNOT_RUN_QUERY ).show();
                                    }
                                }
                                break;
                            case DroppedTable.BUTTON_REMOVE:
                                getSqlTabs().removeFieldColumns( tableData );
                                getDroppedTableContainer().removeJoinedTables( tableData );
                                getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // remove table from table container
                                getDroppedTableContainer().getContainer()
                                                          .removeComponent( button.getParent()
                                                                                  .getParent()
                                                                                  .getParent() );

                                // enable clear table container
                                getSqlMenu().setClearButtonEnabled( getDroppedTableContainer().getContainer()
                                                                                              .getComponentCount() != 0 );

                                // enable/disable save query
                                getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // enable/disable function on group of fields
                                getSqlMenu().setGroupButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                                // fire events
                                fireEvents();

                                // remove table joins form removed table
                                getState( true ).setJoinedTables( getDroppedTableContainer().getJoinedTablesAsStringList() );
                                break;
                            case DroppedTable.BUTTON_QUERY:
                                if ( tableData.contains( StringUtil.DOT_VALUE )
                                    && 4 <= tableData.split( StringUtil.STRING_DOT_SPLITTER ).length )
                                {
                                    // format: database source id, database name, table name, table alias
                                    final String queryGroup = tableData.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                    final String queryName = tableData.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                    final JsonQuery jsonQuery = getJsonBuilder().getQuery( queryGroup, queryName );
                                    if ( SharedUtil.isNotNull( jsonQuery ) )
                                    {
                                        UI.getCurrent()
                                          .addWindow( new MessageWindow( MessageWindow.WARNING,
                                                                         MessageWindow.QUERY_VIEW )
                                          {
                                              private static final long serialVersionUID = -4103581879644197513L;

                                              @Override
                                              public void onOkClick()
                                              {
                                                  getSqlMenu().setOldQueryGroup( queryGroup );
                                                  getSqlMenu().setOldQueryName( queryName );
                                                  addJsonQuery( jsonQuery.toString(), true );
                                              }
                                          } );
                                    }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                }
            } );

            // create tables join
            droppedTable.addDropHandler( new DropHandler()
            {

                private static final long serialVersionUID = -2389935918885846115L;

                @Override
                public AcceptCriterion getAcceptCriterion()
                {
                    return AcceptAll.get();
                }

                @Override
                public void drop( DragAndDropEvent event )
                {
                    if ( event.getTransferable() instanceof DataBoundTransferable )
                    {
                        DataBoundTransferable dragData = (DataBoundTransferable) event.getTransferable();
                        AbstractSelectTargetDetails dropData = (AbstractSelectTargetDetails) event.getTargetDetails();
                        if ( SharedUtil.isNotNull( dragData ) && SharedUtil.isNotNull( dropData ) )
                        {
                            Object dragId = dragData.getItemId();
                            Object dropId = dropData.getItemIdOver();
                            if ( SharedUtil.isNotNullAndNotEmpty( dragId )
                                && SharedUtil.isNotNullAndNotEmpty( dropId ) )
                            {
                                // format: database source id, database name, table name, column name, table alias
                                String[] dragIdSplitted = dragId.toString()
                                                                .split( StringUtil.STRING_DOT_SPLITTER );
                                String[] dropIdSplitted = dropId.toString()
                                                                .split( StringUtil.STRING_DOT_SPLITTER );
                                if ( 5 <= dragIdSplitted.length && 5 <= dropIdSplitted.length )
                                {
                                    getDroppedTableContainer().setJoinedTables( dragIdSplitted[4], dragIdSplitted[3],
                                                                                dropIdSplitted[4], dropIdSplitted[3],
                                                                                TablesJoin.INNER );

                                    getSqlMenu().setClearButtonEnabled( true );
                                    getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );
                                    getState( true ).setJoinedTables( getDroppedTableContainer().getJoinedTablesAsStringList() );
                                    ( (Table) dragData.getSourceComponent() ).unselect( dragId );
                                }
                            }
                        }

                        // fire events
                        fireEvents();
                    }
                }
            } );

            // add dropped table to table container
            getDroppedTableContainer().addTable( droppedTable, left, top );

            // enable clear table container
            getSqlMenu().setClearButtonEnabled( true );

            // fire events
            fireEvents();
        }
        else
        {
            new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_BECAUSE_SQL_DRIVER_IS_NOT_DEFINED ).show();
        }
    }

    /**
     * Get current query in json format.
     * 
     * @param skipError
     *            skip error
     * @return query in json format
     */
    private JsonQuery getCurrentQuery( boolean skipError )
    {
        JsonQuery jsonQuery = new JsonQuery( getSqlMenu().getQueryGroup(), getSqlMenu().getQueryName() );

        // tables
        for ( Iterator<Component> iterator = getDroppedTableContainer().getContainer()
                                                                       .iterator(); iterator.hasNext(); )
        {
            Component component = iterator.next();
            if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
            {
                DroppedTable table = (DroppedTable) component;

                // check if saved query include itself definition
                if ( table.getTableName()
                          .equals( MessageFormat.format( BaseTable.TABLE_NAME, getSqlMenu().getQueryGroup(),
                                                         getSqlMenu().getQueryName() ) ) )
                {
                    if ( !skipError )
                    {
                        new ErrorUtil( ErrorUtil.CANNOT_SAVE_QUERY ).show();
                    }
                    return null;
                }

                // add table
                jsonQuery.addTable( table.getLeft(), table.getTop(), table.getDataSourceId(), table.getDatabaseName(),
                                    table.getTableName(), table.getTableAlias(), table.getTableType()
                                                                                      .toString() );
            }
        }

        // joins
        for ( JoinedTables joinedTables : getDroppedTableContainer().getJoinedTables() )
        {
            jsonQuery.addTableJoins( joinedTables.getFromTable(), joinedTables.getFromFields(),
                                     joinedTables.getToTable(), joinedTables.getToFields(),
                                     joinedTables.getTablesJoin() );
        }

        // fields
        Integer column = 0;
        for ( Object visibleColumn : getSqlTabs().getHeaderFieldsTable()
                                                 .getVisibleColumns() )
        {
            String columnId = visibleColumn.toString();
            if ( !BaseTable.COLUMN_LAST_FINAL.equals( columnId ) && !getSqlTabs().isFieldColumnEmpty( columnId ) )
            {
                List<Object> rowIds = new ArrayList<>();
                List<Object> rowValues = new ArrayList<>();

                // header fields
                for ( Object row : getSqlTabs().getHeaderFieldsTable()
                                               .getItemIds() )
                {
                    String rowId = row.toString();
                    if ( getSqlTabs().isHeaderRowVisible( rowId ) )
                    {
                        rowIds.add( QueryElements.Field.get( rowId ) );
                        rowValues.add( getSqlTabs().getHeaderFieldsTable()
                                                   .getCellValue( columnId, rowId ) );
                    }
                }

                // criteria fields
                for ( Object row : getSqlTabs().getCriteriaFieldsTable()
                                               .getItemIds() )
                {
                    rowIds.add( row.toString() );
                    rowValues.add( getSqlTabs().getCriteriaFieldsTable()
                                               .getCellValue( columnId, row.toString() ) );
                }
                jsonQuery.addTableField( column.toString(), rowIds.toArray(), rowValues.toArray() );
                column += 1;
            }
        }

        // query
        jsonQuery.addQuery( getSqlBuilder().toString() );
        return jsonQuery;
    }

    /**
     * Add json query.
     * 
     * @param queryString
     *            query string
     * @param showQuery
     *            if true query's structure is showing in table container and field columns, otherwise not
     */
    private void addJsonQuery( String queryString, boolean showQuery )
    {
        fireEvents = false;
        try
        {
            JsonQuery query = new JsonQuery( queryString );
            if ( SharedUtil.isNotNullAndNotEmpty( query ) )
            {
                Object queryGroup = query.get( QueryElements.GROUP );
                Object queryName = query.get( QueryElements.NAME );
                if ( SharedUtil.isNullOrEmpty( queryGroup ) || SharedUtil.isNullOrEmpty( queryName ) )
                {
                    new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_BECAUSE_GROUP_AND_NAME_ARE_NOT_DEFINED ).show();
                }
                else if ( ( getSqlMenu().getOldQueryGroup()
                                        .equals( queryGroup.toString() )
                    && getSqlMenu().getOldQueryName()
                                   .equals( queryName.toString() ) )
                    || !isQueryAlreadyCreated( queryGroup.toString(), queryName.toString() ) )
                {
                    // query name
                    getSqlMenu().setQueryGroup( queryGroup.toString() );
                    getSqlMenu().setQueryName( queryName.toString() );

                    // drop table
                    if ( showQuery )
                    {
                        clearTableContainerAndFieldColumns();
                        getSqlMenu().getQueryNameField()
                                    .setEnabled( true );
                        getSqlMenu().setQueryGroup( queryGroup.toString() );
                        getSqlMenu().setQueryName( queryName.toString() );
                        getSqlMenu().getQueryNameField()
                                    .setValue( getSqlMenu().getQueryGroupWithQueryName() );

                        // tables
                        if ( SharedUtil.isNotNull( query.get( QueryElements.TABLES ) ) )
                        {
                            JSONArray tables = (JSONArray) query.get( QueryElements.TABLES );
                            for ( int index = 0; index < tables.length(); index++ )
                            {
                                QueryElement.Table table = new QueryElement.Table( tables.get( index ) );

                                // base table data
                                String dataSourceId = table.get( QueryElements.Table.DATA_SOURCE_ID );
                                String databaseName = table.get( QueryElements.Table.DATABASE );
                                String tableName = table.get( QueryElements.Table.NAME );
                                String tableType = table.get( QueryElements.Table.TYPE );

                                // table position
                                QueryElement.Table.Position tablePosition =
                                    new QueryElement.Table.Position( table.get( QueryElements.Table.POSITION ) );
                                int left = tablePosition.get( QueryElements.Table.Position.LEFT );
                                int top = tablePosition.get( QueryElements.Table.Position.TOP );

                                String id = dataSourceId;
                                if ( TableType.QUERY.equals( TableType.getType( tableType ) ) )
                                {
                                    id = databaseName;
                                }

                                for ( MenuItem menuItem : getDataSourcesMenu().getMenuItems() )
                                {
                                    if ( id.equals( menuItem.getData() ) )
                                    {
                                        for ( MenuItemElement itemElement : menuItem.getMenuItemElements() )
                                        {
                                            Object data = itemElement.getData();
                                            if ( SharedUtil.isNotNullAndNotEmpty( data ) && data.toString()
                                                                                                .contains( StringUtil.DOT_VALUE ) )
                                            {
                                                // format: data source id, database name, table name, table columns
                                                String[] datas = data.toString()
                                                                     .split( StringUtil.STRING_DOT_SPLITTER );
                                                if ( 3 <= datas.length && dataSourceId.equals( datas[0] )
                                                    && databaseName.equals( datas[1] ) && tableName.equals( datas[2] ) )
                                                {
                                                    dropTable( dataSourceId, databaseName, tableName, datas[3],
                                                               tableType, left, top );
                                                    break;
                                                }
                                            }
                                        }
                                        break;
                                    }
                                }
                            }
                        }

                        // header and criteria fields
                        if ( SharedUtil.isNotNull( query.get( QueryElements.FIELDS ) ) )
                        {
                            Map<Integer, Map<QueryElements.Field, String>> columns = new HashMap<>();
                            JSONArray fields = (JSONArray) query.get( QueryElements.FIELDS );
                            for ( int index = 0; index < fields.length(); index++ )
                            {
                                String columnId = null;
                                Map<QueryElements.Field, String> columnValues = new HashMap<>();
                                QueryElement.Field field = new QueryElement.Field( fields.get( index ) );
                                for ( QueryElements.Field element : field.getElements() )
                                {
                                    switch ( element )
                                    {
                                        case COLUMN:
                                            columnId = field.get( element );
                                            break;
                                        case UNDEFINED:
                                            // nothing to do
                                            break;
                                        default:
                                            columnValues.put( element, field.get( element ) );
                                            break;
                                    }
                                }

                                if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                                {
                                    columns.put( Integer.parseInt( columnId ), columnValues );
                                }
                            }

                            // values for header and criteria table
                            for ( int index = 0; index < columns.size(); index++ )
                            {
                                Map<QueryElements.Field, String> columnValues = columns.get( index );

                                // add column with base values
                                String columnId =
                                    getSqlTabs().addFieldColumn( columnValues.get( QueryElements.Field.TABLE ),
                                                                 columnValues.get( QueryElements.Field.FIELD ),
                                                                 columnValues.get( QueryElements.Field.NAME ), true );

                                // move column to the right place
                                getSqlTabs().addFieldColumnBefore( columnId, getSqlTabs().getHeaderFieldsTable()
                                                                                         .getVisibleColumns()[index].toString() );

                                // set values to field header and to field criteria
                                for ( Entry<QueryElements.Field, String> entry : columnValues.entrySet() )
                                {
                                    QueryElements.Field field = entry.getKey();
                                    String value = entry.getValue();
                                    if ( !SharedUtil.isInteger( field.toString() ) )
                                    {
                                        // field header
                                        getSqlTabs().getHeaderFieldsTable()
                                                    .setCellValue( columnId, field.toString(), value );
                                    }
                                    else
                                    {
                                        // field criteria
                                        getSqlTabs().getCriteriaFieldsTable()
                                                    .setCellValue( columnId, field.toString(), value );
                                    }
                                }
                            }
                        }

                        // table joins
                        if ( SharedUtil.isNotNull( query.get( QueryElements.JOINS ) ) )
                        {
                            JSONArray joins = (JSONArray) query.get( QueryElements.JOINS );
                            for ( int index = 0; index < joins.length(); index++ )
                            {
                                QueryElement.Join join = new QueryElement.Join( joins.get( index ) );
                                TablesJoin tablesJoin =
                                    TablesJoin.getJoinFromString( join.get( QueryElements.Join.TYPE ) );
                                String fromTable = join.get( QueryElements.Join.FROM );
                                String[] fromFields =
                                    StringUtil.asStringArray( join.gets( QueryElements.Join.FROM_FIELDS ) );
                                String toTable = join.get( QueryElements.Join.TO );
                                String[] toFields =
                                    StringUtil.asStringArray( join.gets( QueryElements.Join.TO_FIELDS ) );

                                // add to table container
                                for ( int i = 0; i < Math.min( fromFields.length, toFields.length ); i++ )
                                {
                                    getDroppedTableContainer().setJoinedTables( fromTable, fromFields[i], toTable,
                                                                                toFields[i], tablesJoin );
                                }
                            }
                            getState( true ).setJoinedTables( getDroppedTableContainer().getJoinedTablesAsStringList() );
                        }
                    }

                    // add json query to query builders
                    getJsonBuilder().addQuery( query );

                    // fire events
                    fireEvents = true;
                    if ( showQuery )
                    {
                        fireEvents();
                    }
                }
                else
                {
                    new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_BECAUSE_IT_IS_ALREADY_CREATED ).show();
                }
            }
            else
            {
                new ErrorUtil( ErrorUtil.CANNOT_ADD_EMPTY_QUERY ).show();
            }
        }
        catch ( JSONException e )
        {
            Logger.getLogger( DQuery.class.getName() )
                  .log( Level.INFO, e.getLocalizedMessage(), e );
            new ErrorUtil( MessageFormat.format( ErrorUtil.JSON_QUERY_IMPORT, e.getLocalizedMessage() ) ).show();
        }
        fireEvents = true;
    }

    /**
     * Remove json query.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     */
    private void removeJsonQuery( final String queryGroup, final String queryName )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
        {
            UI.getCurrent()
              .addWindow( new MessageWindow( MessageWindow.WARNING,
                                             MessageFormat.format( MessageWindow.QUERY_DELETE, queryGroup, queryName ) )
              {
                  private static final long serialVersionUID = -5167834406049144385L;

                  @Override
                  public void onOkClick()
                  {
                      getJsonBuilder().removeQuery( queryGroup, queryName );
                      getDataSourcesMenu().removeQueryItemElement( queryGroup, queryName );
                      hardReset();

                      // fire events
                      fireEvents();
                  }
              } );
        }
    }

    /**
     * Save query in json format.
     */
    private void saveJsonQuery()
    {
        if ( SharedUtil.isNotNullAndNotEmpty( getSqlMenu().getQueryGroup() )
            && SharedUtil.isNotNullAndNotEmpty( getSqlMenu().getQueryName() ) )
        {
            if ( getSqlMenu().getOldQueryGroupWithQueryName()
                             .equals( getSqlMenu().getQueryGroupWithQueryName() )
                || !isQueryAlreadyCreated( getSqlMenu().getQueryGroup(), getSqlMenu().getQueryName() ) )
            {
                JsonQuery jsonQuery = getCurrentQuery( false );
                if ( SharedUtil.isNotNullAndNotEmpty( jsonQuery.toString() ) )
                {
                    // save json query
                    getJsonBuilder().addQuery( jsonQuery );
                    getDataSourcesMenu().addQueryItemElement( jsonQuery );
                    getSqlMenu().setOldQueryGroup( getSqlMenu().getQueryGroup() );
                    getSqlMenu().setOldQueryName( getSqlMenu().getQueryName() );

                    // fire events
                    fireEvents();
                }
            }
            else
            {
                new ErrorUtil( ErrorUtil.CANNOT_SAVE_QUERY_BECAUSE_IT_IS_ALREADY_CREATED ).show();
            }
        }
    }

    /**
     * Clear table container and field columns.
     */
    private void clearTableContainerAndFieldColumns()
    {
        for ( Iterator<Component> iterator = getDroppedTableContainer().getContainer()
                                                                       .iterator(); iterator.hasNext(); )
        {
            Component component = iterator.next();
            if ( SharedUtil.isNotNull( component ) && component instanceof DroppedTable )
            {
                String tableName = ( (DroppedTable) component ).getTableAlias();
                if ( SharedUtil.isNotNullAndNotEmpty( tableName ) )
                {
                    // remove all field columns
                    getSqlTabs().removeFieldColumns( tableName );

                    // remove all joins
                    getDroppedTableContainer().removeJoinedTables( tableName );
                }
            }
        }

        // remove all components from table container
        getDroppedTableContainer().getContainer()
                                  .removeAllComponents();

        // send refreshed data for joined tables to the client
        getState( true ).setJoinedTables( getDroppedTableContainer().getJoinedTablesAsStringList() );

        // clear sql editor
        getSqlTabs().getSqlQueryEditor()
                    .setReadOnly( false );
        getSqlTabs().getSqlQueryEditor()
                    .clear();
        getSqlTabs().getSqlQueryEditor()
                    .setReadOnly( true );

        // clear query name
        getSqlMenu().resetQueryName();

        // disable buttons
        getSqlMenu().setClearButtonEnabled( false );
        getSqlMenu().setGroupButtonEnabled( false );
        getSqlMenu().setQueryButtonEnabled( false );
        getSqlMenu().setRemoveButtonEnabled( false );
        getSqlMenu().setSaveButtonEnabled( false );
    }

    /**
     * Run query.
     */
    private void runQuery()
    {
        if ( !getSqlTabs().isAllFieldColumnsEmpty() && SharedUtil.isNotNullAndNotEmpty( getSqlBuilder().toString() ) )
        {
            // get jdbc util
            JdbcUtil jdbcUtil;
            if ( getSqlBuilder().getTables()
                                .isEmpty() )
            {
                jdbcUtil = getDataSource( getDataSources().keySet()
                                                          .iterator()
                                                          .next() );
            }
            else if ( SharedUtil.isNotNullAndNotEmpty( getSqlBuilder().getDataSourceId() ) )
            {
                jdbcUtil = getDataSource( getSqlBuilder().getDataSourceId() );
            }
            else
            {
                String database = getSqlBuilder().getTables()
                                                 .get( 0 )
                                                 .getTableName()
                                                 .split( StringUtil.STRING_DOT_SPLITTER )[0];
                jdbcUtil = getDataSourceForDatabase( database );
            }

            // run query
            if ( SharedUtil.isNotNull( jdbcUtil ) )
            {
                if ( SqlDriver.OFFLINE_MODE.equals( jdbcUtil.getDriver() ) )
                {
                    new ErrorUtil( ErrorUtil.CANNOT_CONNECT_TO_DATABASE_IN_OFFLINE_MODE ).show();
                }
                else
                {
                    UI.getCurrent()
                      .addWindow( new TestQueryWindow( jdbcUtil, getSqlBuilder().toString() ) );
                }
            }
            else
            {
                new ErrorUtil( ErrorUtil.CANNOT_RUN_QUERY ).show();
            }
        }
    }

    /**
     * Prepare data source menu.
     */
    private void prepareDataSourcesMenu()
    {
        getDataSourcesMenu().prepare( getDataSources() );
        getDataSourcesMenu().addStyleName( DATA_SOURCE_MENU );
        getDataSourcesMenu().addHandler( new MenuItemElementClickHandler()
        {
            @Override
            public void onMenuItemElementClick( MenuItemElementClickEvent event )
            {
                // drop table
                if ( event.isDoubleClick() )
                {
                    Object data = event.getMenuItemElement()
                                       .getData();
                    if ( SharedUtil.isNotNullAndNotEmpty( data ) && 5 <= data.toString()
                                                                             .split( StringUtil.STRING_DOT_SPLITTER ).length )
                    {
                        // format: database source id, database name, table name, table columns, table type
                        String[] dataTableName = data.toString()
                                                     .split( StringUtil.STRING_DOT_SPLITTER );
                        String dataSourceId = dataTableName[0];
                        String databaseName = dataTableName[1];
                        String tableName = dataTableName[2];
                        String tableColumns = dataTableName[3];
                        String tableType = dataTableName[4];
                        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId )
                            && SharedUtil.isNotNullAndNotEmpty( databaseName )
                            && SharedUtil.isNotNullAndNotEmpty( tableName )
                            && SharedUtil.isNotNullAndNotEmpty( tableColumns )
                            && SharedUtil.isNotNullAndNotEmpty( tableType ) )
                        {
                            dropTable( dataSourceId, databaseName, tableName, tableColumns, tableType,
                                       addedTableLeftPosition, addedTableTopPosition );
                        }
                    }
                }
            }
        } );
    }

    /**
     * Prepare dropped table container.
     */
    private void prepareDroppedTableContainer()
    {
        getDroppedTableContainer().prepare();
        getDroppedTableContainer().addStyleName( CONTENT );
        getDroppedTableContainer().setDropHandler( new DropHandler()
        {
            private static final long serialVersionUID = 4406196213826842664L;

            @Override
            public AcceptCriterion getAcceptCriterion()
            {
                return AcceptAll.get();
            }

            @Override
            public void drop( DragAndDropEvent event )
            {
                if ( SharedUtil.isNotNull( event.getTransferable() ) && SharedUtil.isNotNull( event.getTargetDetails() )
                    && SharedUtil.isNotNull( ( (WrapperTargetDetails) event.getTargetDetails() ).getMouseEvent() ) )
                {
                    // left top coordinates of drop item
                    WrapperTargetDetails targetDetails = (WrapperTargetDetails) event.getTargetDetails();
                    MouseEventDetails eventDetails = targetDetails.getMouseEvent();
                    int left = eventDetails.getClientX() - targetDetails.getAbsoluteLeft();
                    int top = eventDetails.getClientY() - targetDetails.getAbsoluteTop();

                    // drop table
                    Component component = event.getTransferable()
                                               .getSourceComponent();
                    if ( SharedUtil.isNotNull( component.getParent() )
                        && component.getParent() instanceof MenuItemElement )
                    {
                        Object dataObject = ( (MenuItemElement) component.getParent() ).getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( dataObject ) )
                        {
                            // format: database source id, database name, table name, field name, table alias, table
                            // columns
                            String dataTableName = dataObject.toString();
                            if ( 5 <= dataTableName.split( StringUtil.STRING_DOT_SPLITTER ).length )
                            {
                                String dataSourceId = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[0];
                                String databaseName = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[1];
                                String tableName = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[2];
                                String tableColumns = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[3];
                                String tableType = dataTableName.split( StringUtil.STRING_DOT_SPLITTER )[4];

                                if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId )
                                    && SharedUtil.isNotNullAndNotEmpty( databaseName )
                                    && SharedUtil.isNotNullAndNotEmpty( tableName )
                                    && SharedUtil.isNotNullAndNotEmpty( tableColumns ) )
                                {
                                    dropTable( dataSourceId, databaseName, tableName, tableColumns, tableType, left,
                                               top );
                                }
                            }
                        }
                    }
                    // if cannot drop table fire table container drop event
                    else if ( SharedUtil.isNotNull( handlerManager )
                        && SharedUtil.isNotNull( event.getTransferable() ) )
                    {
                        handlerManager.fireEvent( new TableContainerDropEvent( left, top, event.getTransferable() ) );
                    }
                }
            }
        } );
    }

    /**
     * Prepare sql menu.
     */
    private void prepareSqlMenu()
    {
        getSqlMenu().prepare();
        getSqlMenu().addStyleName( SQL_MENU );
        getSqlMenu().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = -6740871942258241472L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof Button )
                {
                    boolean saveButtonEnabled = true;
                    Button button = (Button) event.getComponent();
                    Object data = button.getData();
                    if ( SharedUtil.isNotNullAndNotEmpty( data ) )
                    {
                        // button action
                        switch ( data.toString() )
                        {
                            case SqlMenu.BUTTON_CLEAR:
                                UI.getCurrent()
                                  .addWindow( new MessageWindow( MessageWindow.WARNING,
                                                                 MessageWindow.CLEAR_TABLE_CONTAINER_AND_FIELD_COLUMNS )
                                  {
                                      private static final long serialVersionUID = -8196232907668242224L;

                                      @Override
                                      public void onOkClick()
                                      {
                                          hardReset();
                                      }
                                  } );
                                break;
                            case SqlMenu.BUTTON_COMMON:
                                setHeaderRowVisible( BaseTable.ID_COMMON,
                                                     !getSqlTabs().isHeaderRowVisible( BaseTable.ID_COMMON ) );
                                break;
                            case SqlMenu.BUTTON_DELETE:
                                removeJsonQuery( getSqlMenu().getQueryGroup(), getSqlMenu().getQueryName() );
                                break;
                            case SqlMenu.BUTTON_GROUP:
                                UI.getCurrent()
                                  .addWindow( new SqlFunctionsWindow( getSqlTabs() ) );
                                fireEvents = false;
                                break;
                            case SqlMenu.BUTTON_QUERY:
                                runQuery();
                                fireEvents = false;
                                break;
                            case SqlMenu.BUTTON_SAVE:
                                saveJsonQuery();
                                saveButtonEnabled = false;
                                break;
                            case SqlMenu.BUTTON_SORTING:
                                setHeaderRowVisible( BaseTable.ID_SORTING,
                                                     !getSqlTabs().isHeaderRowVisible( BaseTable.ID_SORTING ) );
                                break;
                            case SqlMenu.BUTTON_TOTAL:
                                setHeaderRowVisible( BaseTable.ID_TOTAL,
                                                     !getSqlTabs().isHeaderRowVisible( BaseTable.ID_TOTAL ) );
                                break;
                            default:
                                break;
                        }

                        // enable/disable save query
                        getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty()
                            && saveButtonEnabled );

                        // enable/disable group button
                        if ( !getSqlTabs().isAllFieldColumnsEmpty()
                            && ( getSqlTabs().isHeaderRowVisible( BaseTable.ID_COMMON )
                                || getSqlTabs().isHeaderRowVisible( BaseTable.ID_SORTING )
                                || getSqlTabs().isHeaderRowVisible( BaseTable.ID_TOTAL ) ) )
                        {
                            getSqlMenu().setGroupButtonEnabled( true );
                        }
                        else
                        {
                            getSqlMenu().setGroupButtonEnabled( false );
                        }

                        // fire events
                        if ( fireEvents )
                        {
                            fireEvents();
                        }
                        fireEvents = true;
                    }
                }
            }
        } );

        getSqlMenu().addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = 4999340710848144792L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                if ( SharedUtil.isNotNull( event ) && SharedUtil.isNotNull( event.getProperty() ) )
                {
                    TextField textField = (TextField) event.getProperty();
                    if ( SharedUtil.isNotNull( textField ) )
                    {
                        if ( SharedUtil.isNotNullAndNotEmpty( textField.getValue() ) )
                        {
                            getSqlMenu().setQueryName( textField.getValue() );
                            textField.setValue( getSqlMenu().getQueryGroupWithQueryName() );
                        }
                        else
                        {
                            getSqlMenu().resetQueryName();
                        }
                    }

                    // enable remove query
                    getSqlMenu().setRemoveButtonEnabled( !textField.isEmpty() );

                    // enable/disable save query
                    getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                    // fire events
                    fireEvents();
                }
            }
        } );
    }

    /**
     * Prepare sql tabs.
     */
    private void prepareSqlTabs()
    {
        getSqlTabs().addStyleName( FIELDS );
        getSqlTabs().addBlurListener( new BlurListener()
        {
            private static final long serialVersionUID = 2245396496450080414L;

            @Override
            public void blur( BlurEvent event )
            {
                if ( SharedUtil.isNotNull( event.getComponent() ) )
                {
                    if ( event.getComponent() instanceof ComboBox )
                    {
                        String description = null;
                        ComboBox comboBox = (ComboBox) event.getComponent();
                        Object data = comboBox.getData();
                        Object value = comboBox.getValue();
                        if ( SharedUtil.isNotNullAndNotEmpty( data ) && SharedUtil.isNotNullAndNotEmpty( value ) )
                        {
                            switch ( QueryElements.Field.get( data ) )
                            {
                                case COMMON:
                                    description = getSqlTabs().getCommonFunctions()
                                                              .getDescription( value.toString() );
                                    break;
                                case SORTING:
                                    description = getSqlTabs().getSortingFunctions()
                                                              .getDescription( value.toString() );
                                    break;
                                case TOTAL:
                                    description = getSqlTabs().getTotalFunctions()
                                                              .getDescription( value.toString() );
                                    break;
                                default:
                                    break;
                            }
                        }
                        comboBox.setDescription( description );
                    }
                    else if ( SharedUtil.isNotNull( event.getComponent()
                                                         .getParent() ) //
                        && event.getComponent()
                                .getParent() instanceof HorizontalLayout )
                    {
                        HorizontalLayout layout = (HorizontalLayout) event.getComponent()
                                                                          .getParent();
                        if ( layout.getComponentCount() == 2 )
                        {
                            // trim value in text field
                            if ( layout.getComponent( 0 ) instanceof TextField )
                            {
                                TextField textField = (TextField) layout.getComponent( 0 );
                                textField.setValue( StringUtil.trim( textField.getValue() ) );
                                if ( 0 < textField.getValue()
                                                  .length() )
                                {
                                    getSqlTabs().addMissingFieldColumns( BaseTable.COLUMNS_TO_ADD );
                                }
                            }

                            // disable fields column in header and criteria tables
                            if ( layout.getComponent( 1 ) instanceof Button )
                            {
                                Button button = (Button) layout.getComponent( 1 );
                                Object columnId = button.getData();
                                if ( !button.isVisible() && SharedUtil.isNotNullAndNotEmpty( columnId ) )
                                {
                                    getSqlTabs().setFieldColumnEnabled( columnId.toString(), false );
                                }
                            }

                            // enable/disable function on group of fields
                            getSqlMenu().setGroupButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );
                        }
                    }

                    // fire events
                    fireEvents();
                }
            }
        } );

        getSqlTabs().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = -34202109096977621L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof Button
                    && SharedUtil.isNotNull( event.getComponent()
                                                  .getParent() )
                    && event.getComponent()
                            .getParent() instanceof HorizontalLayout )
                {
                    HorizontalLayout layout = (HorizontalLayout) event.getComponent()
                                                                      .getParent();
                    if ( layout.getComponent( 0 ) instanceof TextField )
                    {
                        Object columnId = ( (Button) event.getComponent() ).getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                        {
                            getSqlTabs().removeFieldColumn( columnId.toString() );
                        }
                    }
                    else if ( layout.getComponent( 0 ) instanceof ComboBox )
                    {
                        Object columnId = ( (Button) event.getComponent() ).getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                        {
                            ComboBox comboBox = (ComboBox) layout.getComponent( 0 );
                            Object value = comboBox.getValue();
                            if ( SharedUtil.isNotNullAndNotEmpty( value ) && getSqlTabs().getCommonFunctions()
                                                                                         .hasParameters( value.toString() ) )
                            {
                                UI.getCurrent()
                                  .addWindow( new SqlParametersWindow( getSqlTabs(), value.toString(),
                                                                       columnId.toString() )
                                  {
                                      private static final long serialVersionUID = -4502056994959979919L;

                                      @Override
                                      public void saveNewParameters( CommonFunctions commonFunctions, String function )
                                      {
                                          // nothing to do
                                      }
                                  } );
                            }
                        }
                    }
                    else if ( layout.getComponent( 0 ) instanceof Label )
                    {
                        Object data = ( (Button) event.getComponent() ).getData();
                        if ( SharedUtil.isNotNullAndNotEmpty( data ) )
                        {
                            switch ( data.toString() )
                            {
                                case BaseTable.ID_CRITERIA_LESS:
                                    getSqlTabs().setMoreCriteriaRowsVisibled( false );
                                    break;
                                case BaseTable.ID_CRITERIA_MORE:
                                    getSqlTabs().setMoreCriteriaRowsVisibled( true );
                                    break;
                                default:
                                    break;
                            }
                            layout.getComponent( 1 )
                                  .setEnabled( getSqlTabs().getCriteriaMinPageLength() < getSqlTabs().getCriteriaPageLength() );
                            layout.getComponent( 2 )
                                  .setEnabled( getSqlTabs().getCriteriaPageLength() < getSqlTabs().getCriteriaMaxPageLength() );
                        }
                    }

                    // enable/disable save query
                    getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                    // enable/disable function on group of fields
                    getSqlMenu().setGroupButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                    // fire events
                    fireEvents();
                }
            }
        } );

        getSqlTabs().addDropHandler( new DropHandler()
        {
            private static final long serialVersionUID = 876785811093238175L;

            @Override
            public AcceptCriterion getAcceptCriterion()
            {
                return AcceptAll.get();
            }

            @Override
            public void drop( DragAndDropEvent event )
            {
                if ( SharedUtil.isNotNull( event.getTransferable() )
                    && event.getTransferable() instanceof DataBoundTransferable
                    && SharedUtil.isNotNull( event.getTargetDetails() )
                    && SharedUtil.isNotNull( event.getTargetDetails()
                                                  .getTarget() ) )
                {
                    DataBoundTransferable draggableObject = (DataBoundTransferable) event.getTransferable();
                    DropTarget droppedObject = event.getTargetDetails()
                                                    .getTarget();
                    if ( draggableObject instanceof TableTransferable && droppedObject instanceof DragAndDropWrapper )
                    {
                        Table draggableTable = (Table) draggableObject.getSourceComponent();
                        Object draggableItemId = draggableObject.getItemId();
                        Object droppedData = ( (DragAndDropWrapper) droppedObject ).getData();
                        if ( SharedUtil.isNotNull( draggableTable )
                            && SharedUtil.isNotNullAndNotEmpty( draggableItemId )
                            && SharedUtil.isNotNullAndNotEmpty( droppedData ) )
                        {
                            String columnId = droppedData.toString()
                                                         .split( StringUtil.STRING_DOT_SPLITTER )[0];
                            String[] itemIds = StringUtil.asStringArray( draggableTable.getItemIds()
                                                                                       .toArray() );
                            fireEvents = false;
                            for ( String itemId : itemIds )
                            {
                                if ( SharedUtil.isNotNullAndNotEmpty( itemId )
                                    && ( draggableTable.isSelected( itemId ) || itemId.equals( draggableItemId ) ) )
                                {

                                    String[] itemData = itemId.split( StringUtil.STRING_DOT_SPLITTER );

                                    // insert new field column
                                    // format: database source id, database name, table name, field name, table
                                    // alias
                                    String newColumnId =
                                        getSqlTabs().addFieldColumn( itemData[4], itemData[3], itemData[3], true );

                                    if ( SharedUtil.isNotNullAndNotEmpty( newColumnId ) )
                                    {
                                        if ( !newColumnId.equals( columnId ) )
                                        {
                                            if ( !getSqlTabs().isFieldColumnEmpty( columnId ) )
                                            {
                                                getSqlTabs().addFieldColumnBefore( newColumnId, columnId );
                                            }
                                            else
                                            {
                                                getSqlTabs().reverseFieldColumns( newColumnId, columnId );
                                            }
                                        }
                                        getSqlTabs().addMissingFieldColumns( BaseTable.COLUMNS_TO_ADD );
                                        columnId = getSqlTabs().getNextFieldColumnId( newColumnId );

                                        // unselect row in draggable table
                                        draggableTable.unselect( itemId );
                                    }
                                    else
                                    {
                                        new ErrorUtil( ErrorUtil.CANNOT_ADD_NEW_COLUMN ).show();
                                        break;
                                    }
                                }
                            }
                            fireEvents = true;
                        }
                    }

                    // enable clear table container
                    getSqlMenu().setClearButtonEnabled( true );

                    // enable/disable save query
                    getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                    // enable/disable function on group of fields
                    getSqlMenu().setGroupButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                    // fire events
                    fireEvents();
                }
            }
        } );

        getSqlTabs().addFocusListener( new FocusListener()
        {
            private static final long serialVersionUID = -4220158327597345370L;

            @Override
            public void focus( FocusEvent event )
            {
                if ( SharedUtil.isNotNull( event.getComponent() ) && event.getComponent() instanceof ComboBox )
                {
                    ( (ComboBox) event.getComponent() ).setDescription( null );
                }
            }
        } );

        getSqlTabs().addTextChangeListener( new TextChangeListener()
        {
            private static final long serialVersionUID = 3493664093363280359L;

            @Override
            public void textChange( TextChangeEvent event )
            {
                if ( SharedUtil.isNotNull( event.getComponent() ) && SharedUtil.isNotNull( event.getComponent()
                                                                                                .getParent() )
                    && event.getComponent()
                            .getParent() instanceof HorizontalLayout )
                {

                    HorizontalLayout parent = (HorizontalLayout) event.getComponent()
                                                                      .getParent();
                    if ( parent.getComponentCount() == 2 && parent.getComponent( 1 ) instanceof Button )
                    {
                        Button button = (Button) parent.getComponent( 1 );
                        button.setVisible( SharedUtil.isNotNullAndNotEmpty( StringUtil.trim( event.getText() ) ) );
                        if ( button.isVisible() && SharedUtil.isNotNullAndNotEmpty( button.getData() ) )
                        {
                            String columnId = button.getData()
                                                    .toString();

                            getSqlTabs().setFieldColumnEnabled( columnId, true );

                            // combo boxes for common functions, total functions and sorting
                            String commonValue = getSqlTabs().getHeaderFieldsTable()
                                                             .getCellValue( columnId, BaseTable.ID_COMMON );
                            String sortValue = getSqlTabs().getHeaderFieldsTable()
                                                           .getCellValue( columnId, BaseTable.ID_SORTING );
                            String totalValue = getSqlTabs().getHeaderFieldsTable()
                                                            .getCellValue( columnId, BaseTable.ID_TOTAL );
                            if ( SharedUtil.isNull( commonValue ) && SharedUtil.isNull( sortValue )
                                && SharedUtil.isNull( totalValue ) )
                            {
                                // add values to common combo box
                                getSqlTabs().getHeaderFieldsTable()
                                            .setCellValues( columnId, BaseTable.ID_COMMON,
                                                            getSqlTabs().getCommonFunctions()
                                                                        .getFunctions() );

                                // add values to sorting combo box
                                getSqlTabs().getHeaderFieldsTable()
                                            .setCellValues( columnId, BaseTable.ID_SORTING,
                                                            getSqlTabs().getSortingFunctions()
                                                                        .getFunctions() );

                                // add values to total combo box
                                getSqlTabs().getHeaderFieldsTable()
                                            .setCellValues( columnId, BaseTable.ID_TOTAL,
                                                            getSqlTabs().getTotalFunctions()
                                                                        .getFunctions() );

                                // enable to show
                                getSqlTabs().getHeaderFieldsTable()
                                            .setCellValue( columnId, BaseTable.ID_SHOW, Boolean.TRUE.toString() );
                            }
                        }

                        // enable/disable save query
                        getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );
                    }

                    // fire events
                    fireEvents();
                }
            }
        } );

        getSqlTabs().addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = -3939080926028623701L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                if ( SharedUtil.isNotNull( event.getProperty() ) && event.getProperty() instanceof ComboBox )
                {
                    ComboBox comboBox = (ComboBox) event.getProperty();
                    Object data = comboBox.getData();
                    Object value = comboBox.getValue();
                    if ( SharedUtil.isNotNullAndNotEmpty( data ) && SharedUtil.isNotNullAndNotEmpty( value ) )
                    {
                        // enable button for setting function's parameter values
                        String description = data.toString();
                        if ( description.equals( BaseTable.ID_COMMON ) && SharedUtil.isNotNull( comboBox.getParent() )
                            && comboBox.getParent() instanceof HorizontalLayout )
                        {
                            HorizontalLayout layout = (HorizontalLayout) comboBox.getParent();
                            if ( layout.getComponentCount() == 2 && layout.getComponent( 1 ) instanceof Button )
                            {
                                Button button = (Button) layout.getComponent( 1 );
                                if ( SharedUtil.isNotNull( button.getData() ) )
                                {
                                    button.setEnabled( getSqlTabs().getCommonFunctions()
                                                                   .hasParameters( value.toString() ) );
                                }
                            }
                        }
                    }
                }

                // enable/disable save query
                getSqlMenu().setSaveButtonEnabled( !getSqlTabs().isAllFieldColumnsEmpty() );

                // fire events
                fireEvents();
            }
        } );

        // prepare sql tabs
        getSqlTabs().prepare();
    }

    /**
     * Create progress status layout.
     */
    private void createProgressStatusLayout()
    {
        progressStatus = new ProgressBar();
        progressStatusLayout = new VerticalLayout( progressStatus );
        progressStatusLayout.setComponentAlignment( progressStatus, Alignment.MIDDLE_CENTER );
        progressStatusLayout.setSizeFull();
        progressStatusLayout.addStyleName( "indicator" );
        addComponent( progressStatusLayout );

        // fire initialization process
        getThis().getState( true )
                 .setInitProgress( INIT_STARTED );
    }

    /**
     * Delete progress status layout.
     */
    private void deleteProgressStatusLayout()
    {
        if ( SharedUtil.isNotNull( progressStatusLayout ) )
        {
            removeComponent( progressStatusLayout );
            progressStatus = null;
            progressStatusLayout = null;
        }
    }

    /**
     * Set progress status.
     * 
     * @param progress
     *            progress value
     */
    private void setProgressStatus( int progress )
    {
        if ( SharedUtil.isNotNull( progressStatus ) )
        {
            progressStatus.setValue( progress / 100f );
        }
    }

    /**
     * Context preparation.
     */
    private void prepare()
    {
        // set data sources item visible
        setDataSourcesItemVisible( true );

        // prepare context components
        prepareDataSourcesMenu();
        prepareDroppedTableContainer();
        prepareSqlMenu();
        prepareSqlTabs();

        // context layout
        addComponent( getDataSourcesMenu() );
        addComponent( getDroppedTableContainer() );
        addComponent( getSqlMenu() );
        addComponent( getSqlTabs() );
        setExpandRatio( getDroppedTableContainer(), 1f );
        getDataSourcesMenu().setVisible( false );
        getDroppedTableContainer().setVisible( false );
        getSqlMenu().setVisible( false );
        getSqlTabs().setVisible( false );

        // visibility of header rows
        setHeaderRowVisible( BaseTable.ID_COMMON, false );
        setHeaderRowVisible( BaseTable.ID_SORTING, true );
        setHeaderRowVisible( BaseTable.ID_TOTAL, false );

        // create progress layout
        createProgressStatusLayout();
    }

    /**
     * Add event handler for query change event.
     * 
     * @param handler
     *            event handler for query change event
     */
    public void addHandler( QueryChangeHandler handler )
    {
        handlerManager.addHandler( QueryChangeEvent.getType(), handler );
    }

    /**
     * Remove event handler for query change event.
     * 
     * @param handler
     *            event handler for query change event
     */
    public void removeHandler( QueryChangeHandler handler )
    {
        if ( handlerManager.isEventHandled( QueryChangeEvent.getType() ) )
        {
            handlerManager.removeHandler( QueryChangeEvent.getType(), handler );
        }
    }

    /**
     * Add event handler for group query change event.
     * 
     * @param handler
     *            event handler for group query change event
     */
    public void addHandler( GroupQueryChangeHandler handler )
    {
        handlerManager.addHandler( GroupQueryChangeEvent.getType(), handler );
    }

    /**
     * Remove event handler for group query change event.
     * 
     * @param handler
     *            event handler for group query change event
     */
    public void removeHandler( GroupQueryChangeHandler handler )
    {
        if ( handlerManager.isEventHandled( GroupQueryChangeEvent.getType() ) )
        {
            handlerManager.removeHandler( GroupQueryChangeEvent.getType(), handler );
        }
    }

    /**
     * Add event handler for table container drop event.
     * 
     * @param handler
     *            event handler for table container drop event
     */
    public void addHandler( TableContainerDropHandler handler )
    {
        handlerManager.addHandler( TableContainerDropEvent.getType(), handler );
    }

    /**
     * Remove event handler for table container drop event.
     * 
     * @param handler
     *            event handler for table container drop event
     */
    public void removeHandler( TableContainerDropHandler handler )
    {
        if ( handlerManager.isEventHandled( TableContainerDropEvent.getType() ) )
        {
            handlerManager.removeHandler( TableContainerDropEvent.getType(), handler );
        }
    }

    /**
     * Hard reset query: remove all saved queries and put query in initial state.
     */
    public void hardReset()
    {
        clearTableContainerAndFieldColumns();
        getSqlMenu().resetOldQueryName();
    }

    /**
     * Set data sources item visible.
     * 
     * @param visible
     *            if true data sources item is visible, otherwise is hidden
     */
    public void setDataSourcesItemVisible( boolean visible )
    {
        if ( !getDataSourcesMenu().isInitialized() || getDataSourcesMenu().getMenuItems()
                                                                          .size() <= 1 )
        {
            // don't remove: protect against stack overflow error
            getDataSourcesMenu().setDataSourcesItemVisible( true );
        }
        else
        {
            getDataSourcesMenu().setDataSourcesItemVisible( visible );
        }
    }

    /**
     * Get added table left position. Table left position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @return added table left position
     */
    public int getAddedTableLeftPosition()
    {
        return addedTableLeftPosition;
    }

    /**
     * Set added table left position. Table left position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @param left
     *            added table left position
     */
    public void setAddedTableLeftPosition( int left )
    {
        addedTableLeftPosition = left;
    }

    /**
     * Get added table top position. Table top position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @return added table top position
     */
    public int getAddedTableTopPosition()
    {
        return addedTableTopPosition;
    }

    /**
     * Set added table top position. Table top position is a start position for the tables which is added to the table
     * container (middle part of dQuery UI) from data source menu with double click on menu item element.
     * 
     * @param top
     *            added table top position
     */
    public void setAddedTableTopPosition( int top )
    {
        addedTableTopPosition = top;
    }

    /**
     * Set added table left and top positions. Table left and top positions are a start position for the tables which is
     * added to the table container (middle part of dQuery UI) from data source menu with double click on menu item
     * element.
     * 
     * @param left
     *            added table left position
     * @param top
     *            added table top position
     */
    public void setAddedTablePosition( int left, int top )
    {
        setAddedTableLeftPosition( left );
        setAddedTableTopPosition( top );
    }

    /**
     * Get data source for specified database.
     * 
     * @param database
     *            database name
     * @return data source
     */
    public JdbcUtil getDataSourceForDatabase( String database )
    {
        JdbcUtil jdbcUtil = null;
        if ( !SharedUtil.isNullOrEmpty( database ) )
        {
            for ( Entry<String, JdbcUtil> entry : getDataSources().entrySet() )
            {
                String key = entry.getKey();
                JdbcUtil dataSource = entry.getValue();

                if ( SqlDriver.OFFLINE_MODE.equals( dataSource.getDriver() ) )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( dataSource.getCatalog() )
                        && database.equals( dataSource.getCatalog() ) )
                    {
                        jdbcUtil = getDataSources().get( key );
                        break;
                    }
                }
                else
                {
                    try
                    {
                        String catalog = dataSource.getConnection()
                                                   .getCatalog();
                        String schema = dataSource.getConnection()
                                                  .getSchema();
                        if ( SharedUtil.isNull( jdbcUtil )
                            && ( ( SharedUtil.isNotNullAndNotEmpty( catalog ) && catalog.equals( database ) )
                                || ( SharedUtil.isNotNullAndNotEmpty( schema ) && schema.equals( database ) ) ) )
                        {
                            jdbcUtil = getDataSources().get( key );
                        }
                    }
                    catch ( SQLException e )
                    {
                        Logger.getLogger( DQuery.class.getName() )
                              .log( Level.INFO, e.getLocalizedMessage(), e );
                        jdbcUtil = null;
                    }
                }
            }
        }

        // can not find data source for database
        if ( SharedUtil.isNull( jdbcUtil ) )
        {
            new ErrorUtil( MessageFormat.format( ErrorUtil.CANNOT_FIND_DATASOURCE_FOR_DATABASE, database ) ).show();
        }

        return jdbcUtil;
    }

    /**
     * Get data source.
     * 
     * @param dataSourceId
     *            id of the data source
     * @return data source, or null if data source does not exists
     */
    public JdbcUtil getDataSource( String dataSourceId )
    {
        if ( SharedUtil.isNullOrEmpty( dataSourceId ) )
        {
            throw new IllegalArgumentException( IDENTIFIER_IS_EMPTY );
        }
        return getDataSources().get( dataSourceId );
    }

    /**
     * Add data source.
     * 
     * @param dataSourceId
     *            id of added data source
     * @param JdbcUtil
     *            data source
     */
    public void addDataSource( String dataSourceId, JdbcUtil dataSource )
    {
        if ( SharedUtil.isNotNull( getDataSources().get( dataSourceId ) ) )
        {
            throw new IllegalArgumentException( MessageFormat.format( DATA_SOURCES_ALREADY_HAVE_THIS_IDENTIFIER,
                                                                      dataSourceId ) );
        }
        else if ( SharedUtil.isNotNull( dataSource ) )
        {
            getDataSources().put( dataSourceId, dataSource );

            // activate data source.
            if ( getDataSourcesMenu().isInitialized() && !getDataSourcesMenu().getDataSources()
                                                                              .containsKey( dataSourceId ) )
            {
                getDataSourcesMenu().addDataSourceItem( dataSourceId, dataSource );
            }
        }
    }

    /**
     * Remove data source.
     * 
     * @param dataSourceId
     *            id of added data source
     */
    public void removeDataSource( String dataSourceId )
    {
        if ( getDataSourcesMenu().isInitialized() && SharedUtil.isNotNullAndNotEmpty( dataSourceId )
            && SharedUtil.isNotNull( getDataSources().get( dataSourceId ) ) && getDataSourcesMenu().getDataSources()
                                                                                                   .containsKey( dataSourceId ) )
        {
            getDataSourcesMenu().removeDataSourceItem( dataSourceId );
            getShowDroppedTableDataEnabled().remove( dataSourceId );
            getDataSources().remove( dataSourceId );
        }
    }

    /**
     * Check if data source is enabled.
     * 
     * @param dataSourceId
     *            data source id
     * @return true data source is enabled, otherwise false
     */
    public boolean isDataSourceEnabled( String dataSourceId )
    {
        return getDataSourcesMenu().isDataSourceEnabled( dataSourceId );
    }

    /**
     * Set data source enabled. User can later change a state of the data source if data source menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param enabled
     *            if true data source is enabled, otherwise is disabled
     */
    public void setDataSourceEnabled( String dataSourceId, boolean enabled )
    {
        getDataSourcesMenu().setDataSourceEnabled( dataSourceId, enabled );
    }

    /**
     * Check if data source table is disabled.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @return true data source table is disabled, otherwise false
     */
    public boolean isDataSourceTableDisabled( String dataSourceId, String tableName )
    {
        return getDataSourcesMenu().isDataSourceTableDisabled( dataSourceId, tableName );
    }

    /**
     * Set data source table disabled. User can later change a state of the data source table if data source menu is
     * visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @param disabled
     *            if true data source table is disabled, otherwise is enabled
     */
    public void setDataSourceTableDisabled( String dataSourceId, String tableName, boolean disabled )
    {
        getDataSourcesMenu().setDataSourceTableDisabled( dataSourceId, tableName, disabled );
    }

    /**
     * Check if can show dropped table data.
     * 
     * @return true if can show dropped table data, otherwise false
     */
    public boolean isShowDroppedTableDataEnabled()
    {
        return showDroppedTableData;
    }

    /**
     * Show dropped table data.
     * 
     * @param showDroppedTableData
     *            if true show dropped table data, otherwise not
     */
    public void setShowDroppedTableDataEnabled( boolean showDroppedTableData )
    {
        this.showDroppedTableData = showDroppedTableData;
    }

    /**
     * Check if can show dropped table data per data source.
     * 
     * @param dataSourceId
     *            data source id
     * @return true if can show dropped table data, otherwise false
     */
    @Override
    public boolean isShowDroppedTableDataEnabled( String dataSourceId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId )
            && getShowDroppedTableDataEnabled().containsKey( dataSourceId ) )
        {
            return super.isShowDroppedTableDataEnabled( dataSourceId );
        }
        return isShowDroppedTableDataEnabled();
    }

    /**
     * Show dropped table data per data source.
     * 
     * @param dataSourceId
     *            data source id
     * @param showDroppedTableData
     *            if true show dropped table data, otherwise not
     */
    @Override
    public void setShowDroppedTableDataEnabled( String dataSourceId, boolean showDroppedTableData )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            super.setShowDroppedTableDataEnabled( dataSourceId, showDroppedTableData );
        }
    }

    /**
     * Check if can show query definition.
     * 
     * @return true if can show query definition, otherwise false
     */
    public boolean isShowQueryDefinitionEnabled()
    {
        return showQueryDefinition;
    }

    /**
     * Show query definition.
     * 
     * @param showQueryDefinition
     *            if true show query definition, otherwise not
     */
    public void setShowQueryDefinitionEnabled( boolean showQueryDefinition )
    {
        this.showQueryDefinition = showQueryDefinition;
    }

    /**
     * Check if save changes is enabled.
     * 
     * @return true if save changes is enabled, otherwise false
     */
    public boolean isSaveChangesEnabled()
    {
        return getSqlMenu().isSaveChangesEnabled();
    }

    /**
     * Set save changes enabled.
     * 
     * @param changeQueryEnabled
     *            enable save changes
     */
    public void setSaveChangesEnabled( boolean saveChangesEnabled )
    {
        getSqlMenu().setSaveChangesEnabled( saveChangesEnabled );
    }

    /**
     * Check if show query data is enabled.
     * 
     * @return true if show query data is enabled, otherwise false
     */
    public boolean isShowQueryDataEnabled()
    {
        return getSqlMenu().isShowQueryDataEnabled();
    }

    /**
     * Set show query data enabled.
     * 
     * @param showQueryDataEnabled
     *            if true query data show is enabled, otherwise not
     */
    public void setShowQueryDataEnabled( boolean showQueryDataEnabled )
    {
        getSqlMenu().setShowQueryDataEnabled( showQueryDataEnabled );
    }

    /**
     * Check if create a new group is enabled.
     * 
     * @return true if create a new group is enabled, otherwise false
     */
    public boolean isCreateNewQueryGroupEnabled()
    {
        return getSqlMenu().isCreateNewQueryGroupEnabled();
    }

    /**
     * Set create a new group enabled.
     * 
     * @param createNewQueryGroupEnabled
     *            enable create a new query group
     */
    public void setCreateNewQueryGroupEnabled( boolean createNewQueryGroupEnabled )
    {
        getSqlMenu().setCreateNewQueryGroupEnabled( createNewQueryGroupEnabled );
    }

    /**
     * Get common functions.
     * 
     * @return common functions
     */
    public String[] getCommonFunctions()
    {
        return getSqlTabs().getCommonFunctions()
                           .getFunctions();
    }

    /**
     * Add common functions.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void addCommonFunction( String function, String description )
    {
        getSqlTabs().getCommonFunctions()
                    .addFunction( function, description );
    }

    /**
     * Get common function description.
     * 
     * @param function
     *            function name
     * @return function description
     */
    public String getCommonFunctionDescription( String function )
    {
        return getSqlTabs().getCommonFunctions()
                           .getDescription( function );
    }

    /**
     * Set common function description.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void setCommonFunctionDescription( String function, String description )
    {
        getSqlTabs().getCommonFunctions()
                    .setDescription( function, description );
    }

    /**
     * Get common function's parameter values.
     * 
     * @param function
     *            function name
     * @return function parameter values
     */
    public Object[] getCommonFunctionParameterValues( String function )
    {
        return getSqlTabs().getCommonFunctions()
                           .getParameterValues( function );
    }

    /**
     * Set common function's parameter values.
     * 
     * @param function
     *            function name
     * @param parameterValues
     *            function parameter values
     */
    public void setCommonFunctionParameterValues( String function, Object[] parameterValues )
    {
        getSqlTabs().getCommonFunctions()
                    .setParameterValues( function, parameterValues );
    }

    /**
     * Get common function parameters.
     * 
     * @param function
     *            function name
     * @return function parameters
     */
    public String[] getCommonFunctionParameters( String function )
    {
        return getSqlTabs().getCommonFunctions()
                           .getParameters( function );
    }

    /**
     * Set common function parameters.
     * 
     * @param function
     *            function name
     * @param parameters
     *            function parameters
     */
    public void setCommonFunctionParameters( String function, String... parameters )
    {
        getSqlTabs().getCommonFunctions()
                    .setParameters( function, parameters );
    }

    /**
     * Get total functions.
     * 
     * @return total functions
     */
    public String[] getTotalFunctions()
    {
        return getSqlTabs().getTotalFunctions()
                           .getFunctions();
    }

    /**
     * Add total functions.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void addTotalFunction( String function, String description )
    {
        getSqlTabs().getTotalFunctions()
                    .addFunction( function, description );
    }

    /**
     * Get total function description.
     * 
     * @param function
     *            function name
     * @return function description
     */
    public String getTotalFunctionDescription( String function )
    {
        return getSqlTabs().getTotalFunctions()
                           .getDescription( function );
    }

    /**
     * Set total function description.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void setTotalFunctionDescription( String function, String description )
    {
        getSqlTabs().getTotalFunctions()
                    .setDescription( function, description );
    }

    /**
     * Get total function's parameter values.
     * 
     * @param function
     *            function name
     * @return function parameter values
     */
    public Object[] getTotalFunctionParameterValues( String function )
    {
        return getSqlTabs().getTotalFunctions()
                           .getParameterValues( function );
    }

    /**
     * Set total function's parameter values.
     * 
     * @param function
     *            function name
     * @param parameterValues
     *            function parameter values
     */
    public void setTotalFunctionParameterValues( String function, Object[] parameterValues )
    {
        getSqlTabs().getTotalFunctions()
                    .setParameterValues( function, parameterValues );
    }

    /**
     * Get total function parameters.
     * 
     * @param function
     *            function name
     * @return function parameters
     */
    public String[] getTotalFunctionParameters( String function )
    {
        return getSqlTabs().getTotalFunctions()
                           .getParameters( function );
    }

    /**
     * Set total function parameters.
     * 
     * @param function
     *            function name
     * @param parameters
     *            function parameters
     */
    public void setTotalFunctionParameters( String function, String... parameters )
    {
        getSqlTabs().getCommonFunctions()
                    .setParameters( function, parameters );
    }

    /**
     * Get sources page length.
     * 
     * @return sources page length
     */
    public int getSourcesPageLength()
    {
        return getDataSourcesMenu().getSourcesPageLength();
    }

    /**
     * Set sources page length.
     * 
     * @param sourcesPageLength
     *            sources page length
     */
    public void setSourcesPageLength( int sourcesPageLength )
    {
        getDataSourcesMenu().setSourcesPageLength( sourcesPageLength );
    }

    /**
     * Get width of description column in pixels.
     * 
     * @return width of description column in pixels.
     */
    public int getDescriptionColumnWidth()
    {
        return getSqlTabs().getDescriptionColumnWidth();
    }

    /**
     * Set width of description column in pixels.
     * 
     * @param descriptionColumnWidth
     *            width of description column in pixels
     */
    public void setDescriptionColumnWidth( int descriptionColumnWidth )
    {
        getSqlTabs().setDescriptionColumnWidth( descriptionColumnWidth );
    }

    /**
     * Get number of max allowed visible field columns.
     * 
     * @return number of max allowed visible field columns
     */
    public int getAllFieldColumns()
    {
        return getSqlTabs().getAllFieldColumns();
    }

    /**
     * Set number of max allowed visible field columns.
     * 
     * @param allFieldColumns
     *            number of max allowed visible field columns
     */
    public void setAllFieldColumns( int allFieldColumns )
    {
        getSqlTabs().setAllFieldColumns( allFieldColumns );
    }

    /**
     * Get number of visible field columns.
     * 
     * @return number of visible filed columns
     */
    public int getFieldColumns()
    {
        return getSqlTabs().getFieldColumns();
    }

    /**
     * Set number of visible field columns.
     * 
     * @param fieldColumns
     *            number of visible field columns
     */
    public void setFieldColumns( int fieldColumns )
    {
        getSqlTabs().setFieldColumns( fieldColumns );
    }

    /**
     * Get width of field column in pixels.
     * 
     * @return width of field column in pixels.
     */
    public int getFieldColumnWidth()
    {
        return getSqlTabs().getFieldColumnWidth();
    }

    /**
     * Set width of field column in pixels.
     * 
     * @param fieldColumnWidth
     *            width of field column in pixels
     */
    public void setFieldColumnWidth( int fieldColumnWidth )
    {
        getSqlTabs().setFieldColumnWidth( fieldColumnWidth );
    }

    /**
     * Get criteria page length.
     * 
     * @return criteria page length
     */
    public int getCriteriaPageLength()
    {
        return getSqlTabs().getCriteriaPageLength();
    }

    /**
     * Set criteria page length.
     * 
     * @param criteriaPageLength
     *            criteria page length
     */
    public void setCriteriaPageLength( int criteriaPageLength )
    {
        getSqlTabs().setCriteriaPageLength( criteriaPageLength );
    }

    /**
     * Get number of criteria rows.
     * 
     * @return number of criteria rows
     */
    public int getCriteriaRows()
    {
        return getSqlTabs().getCriteriaRows();
    }

    /**
     * Set number of criteria rows.
     * 
     * @param criteriaRows
     *            number of criteria rows
     */
    public void setCriteriaRows( int criteriaRows )
    {
        getSqlTabs().setCriteriaRows( criteriaRows );
    }

    /**
     * Get criteria maximum page length.
     * 
     * @return criteria maximum page length
     */
    public int getCriteriaMaxPageLength()
    {
        return getSqlTabs().getCriteriaMaxPageLength();
    }

    /**
     * Set criteria maximum page length.
     * 
     * @param criteriaMaxPageLength
     *            criteria maximum page length
     */
    public void setCriteriaMaxPageLength( int criteriaMaxPageLength )
    {
        getSqlTabs().setCriteriaMaxPageLength( criteriaMaxPageLength );
    }

    /**
     * Get criteria minimum page length.
     * 
     * @return criteria minimum page length
     */
    public int getCriteriaMinPageLength()
    {
        return getSqlTabs().getCriteriaMinPageLength();
    }

    /**
     * Set criteria minimum page length.
     * 
     * @param criteriaMinPageLength
     *            criteria minimum page length
     */
    public void setCriteriaMinPageLength( int criteriaMinPageLength )
    {
        getSqlTabs().setCriteriaMinPageLength( criteriaMinPageLength );
    }

    /**
     * Get all json queries.
     * 
     * @return all json queries
     */
    public Map<String, Map<String, JsonQuery>> getQueries()
    {
        return getJsonBuilder().getQueries();
    }

    /**
     * Get json queries for specified query group.
     * 
     * @param queryGroup
     *            query group
     * @return json queries for specified query group
     */
    public Map<String, JsonQuery> getGroupQueries( String queryGroup )
    {
        return getJsonBuilder().getQueries()
                               .get( queryGroup );
    }

    /**
     * Get query group.
     * 
     * @return query group
     */
    public String getQueryGroup()
    {
        return getSqlMenu().getQueryGroup();
    }

    /**
     * Set query group.
     * 
     * @param queryGroup
     *            query group
     */
    public void setQueryGroup( String queryGroup )
    {
        getSqlMenu().setQueryGroup( queryGroup );
    }

    /**
     * Get query name.
     * 
     * @return query name
     */
    public String getQueryName()
    {
        return getSqlMenu().getQueryName();
    }

    /**
     * Set query name.
     * 
     * @param queryName
     *            query name
     */
    public void setQueryName( String queryName )
    {
        getSqlMenu().setQueryName( queryName );
    }

    /**
     * Reset query name.
     */
    public void resetQueryName()
    {
        getSqlMenu().resetQueryName();
    }

    /**
     * Check if query is already created.
     * 
     * @param group
     *            query group
     * @param name
     *            query name
     * @return true if query is already created, otherwise false
     */
    public boolean isQueryAlreadyCreated( String group, String name )
    {
        return getDataSourcesMenu().isQueryItemElementAlreadyCreated( group, name );
    }

    /**
     * Add query.
     * 
     * @param query
     *            query
     * @param showQuery
     *            if true query is showing in table container and field columns, otherwise not
     */
    public void addQuery( String query, boolean showQuery )
    {
        addJsonQuery( query, showQuery );
    }

    /**
     * Remove query.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     */
    public void removeQuery( final String queryGroup, final String queryName )
    {
        removeJsonQuery( queryGroup, queryName );
    }

    /**
     * Save query.
     */
    public void saveQuery()
    {
        if ( isQueryAlreadyCreated( getSqlMenu().getQueryGroup(), getSqlMenu().getQueryName() ) )
        {
            new ErrorUtil( ErrorUtil.CANNOT_ADD_QUERY_BECAUSE_IT_IS_ALREADY_CREATED ).show();
            return;
        }
        saveJsonQuery();
    }

    /**
     * Add query to the dropped table container.
     * 
     * @param jsonQuery
     *            json query
     * @param left
     *            dropped query left position
     * @param top
     *            dropped query top position
     */
    public void addQuery( JsonQuery jsonQuery, Integer left, Integer top )
    {
        getJsonBuilder().addQuery( jsonQuery );
        getJsonBuilder().addSqlQuery( jsonQuery.get( QueryElements.GROUP )
                                               .toString(),
                                      jsonQuery.get( QueryElements.NAME )
                                               .toString(),
                                      jsonQuery.get( QueryElements.QUERY )
                                               .toString() );

        List<String> tableColumns = new ArrayList<>();
        JSONArray tableFields = (JSONArray) jsonQuery.get( QueryElements.FIELDS );
        for ( int index = 0; index < tableFields.length(); index++ )
        {
            tableColumns.add( new QueryElement.Field( tableFields.get( index ) ).get( QueryElements.Field.NAME ) );
        }

        dropTable( jsonQuery.get( QueryElements.DATA_SOURCE_ID )
                            .toString(),
                   jsonQuery.get( QueryElements.GROUP )
                            .toString(),
                   jsonQuery.get( QueryElements.NAME )
                            .toString(),
                   tableColumns.toString(), TableType.QUERY.toString(), left, top );
    }

    /**
     * Set table description.
     * 
     * @param dataSourceId
     *            data source id
     * @param table
     *            table name
     * @param description
     *            table description
     */
    public void setTableDescription( String dataSourceId, String table, String description )
    {
        getDataSourcesMenu().setTableDescription( dataSourceId, table, description );
    }

    /**
     * Set query description.
     * 
     * @param group
     *            group name
     * @param query
     *            query name
     * @param description
     *            query description
     */
    public void setQueryDescription( String group, String query, String description )
    {
        getDataSourcesMenu().setQueryDescription( group, query, description );
    }

}
