/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui.windows;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.base.ui.BaseWindow;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.ui.SqlTabs;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.event.FieldEvents.TextChangeEvent;
import com.vaadin.event.FieldEvents.TextChangeListener;
import com.vaadin.server.FontAwesome;
import com.vaadin.server.UserError;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.Component;
import com.vaadin.ui.RichTextArea;
import com.vaadin.ui.TextField;
import com.vaadin.ui.themes.ValoTheme;

public abstract class SqlParametersWindow
    extends BaseWindow
{

    private static final long serialVersionUID = 2274629566493842599L;

    private static final String DESCRIPTION = "Description";

    private static final String MODAL_CAPTION = "Change parameter values";

    private static final String MODAL_REQUIRED = "Required field and cannot be empty";

    /**
     * Creates a new instance.
     */
    private SqlParametersWindow()
    {
        setCaption( MODAL_CAPTION );
    }

    /**
     * Creates a new instance.
     * 
     * @param sqlTabs
     *            sqlTabs instance
     * @param function
     *            common function name
     * @param columnId
     *            column id
     */
    public SqlParametersWindow( final SqlTabs sqlTabs, final String function, final String columnId )
    {
        this();

        // clone common functions
        final CommonFunctions commonFunctions = sqlTabs.getCommonFunctions()
                                                       .cloneFunctions();
        // function's parameter values
        String[] parameters = commonFunctions.getParameters( function );
        Object[] parameterValues = commonFunctions.getParameterValues( function );
        for ( int index = 0; index < parameters.length; index++ )
        {
            TextField textField = new TextField( parameters[index] );
            textField.setData( index );
            textField.setResponsive( true );
            textField.setWidth( 100, Unit.PERCENTAGE );
            textField.addStyleName( ValoTheme.TEXTFIELD_TINY );

            // set parameter value
            if ( SharedUtil.isNotNull( parameterValues ) && index < parameterValues.length
                && SharedUtil.isNotNull( parameterValues[index] ) )
            {
                textField.setValue( StringUtil.trim( parameterValues[index].toString() ) );
            }

            // add parameter text field to content
            addContextComponent( textField );

            // check if required text is not empty
            textField.addTextChangeListener( new TextChangeListener()
            {
                private static final long serialVersionUID = 6080723427369435700L;

                @Override
                public void textChange( TextChangeEvent event )
                {
                    TextField textField = (TextField) event.getComponent();
                    Object data = textField.getData();
                    if ( SharedUtil.isNotNull( data ) )
                    {
                        if ( Integer.valueOf( data.toString() ) == 0 )
                        {
                            if ( SharedUtil.isNullOrEmpty( StringUtil.trim( event.getText() ) ) )
                            {
                                getOkButton().setEnabled( false );
                                textField.setComponentError( new UserError( MODAL_REQUIRED ) );
                            }
                            else
                            {
                                getOkButton().setEnabled( true );
                                textField.setComponentError( null );
                            }
                        }
                        else
                        {
                            getOkButton().setEnabled( true );
                        }
                    }
                }
            } );
        }

        // function description
        RichTextArea richTextArea = new RichTextArea( DESCRIPTION, commonFunctions.getDescription( function ) );
        richTextArea.setIcon( FontAwesome.COMMENTING_O );
        richTextArea.setReadOnly( true );
        richTextArea.setWidth( 100, Unit.PERCENTAGE );
        richTextArea.addStyleName( ValoTheme.TEXTFIELD_TINY );
        addContextComponent( richTextArea );

        // "OK" button click
        getOkButton().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 3491165668669637327L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                // get parameter values from dialog window
                List<String> values = new ArrayList<>();
                for ( Iterator<Component> iterator = getContext().iterator(); iterator.hasNext(); )
                {
                    Component component = iterator.next();
                    if ( component instanceof TextField )
                    {
                        String value = StringUtil.trim( ( (TextField) component ).getValue() );
                        if ( SharedUtil.isNotNullAndNotEmpty( value ) )
                        {
                            values.add( value );
                        }
                    }
                }

                // set new parameter values
                if ( !values.isEmpty() )
                {
                    commonFunctions.setParameterValues( function, values.toArray() );
                    if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
                    {
                        // save new parameters in header field table
                        sqlTabs.getHeaderFieldsTable()
                               .resetCell( columnId, BaseTable.ID_COMMON );
                        sqlTabs.getHeaderFieldsTable()
                               .setCellValues( columnId, BaseTable.ID_COMMON, commonFunctions.getFunctions() );
                        sqlTabs.getHeaderFieldsTable()
                               .setCellValue( columnId, BaseTable.ID_COMMON, commonFunctions.getFunction( function ) );
                        sqlTabs.getHeaderFieldsTable()
                               .setCellDescription( columnId, BaseTable.ID_COMMON,
                                                    commonFunctions.getDescription( function ) );
                    }
                    else
                    {
                        // pass new parameters to other object
                        saveNewParameters( commonFunctions, function );
                    }
                }

                // close dialog window
                close();
            }
        } );
    }

    public abstract void saveNewParameters( CommonFunctions commonFunctions, String function );

}
