/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui;

import java.text.MessageFormat;

import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.server.FontAwesome;
import com.vaadin.ui.Button;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.TextField;
import com.vaadin.ui.themes.ValoTheme;

public class SqlMenu
    extends HorizontalLayout
{

    private static final long serialVersionUID = 9017316084903681182L;

    private static final int TEXT_FIELD_LENGTH = 30;

    public static final String BUTTON_CLEAR = "Clear table container";

    public static final String BUTTON_COMMON = "Hide/show common row";

    public static final String BUTTON_DELETE = "Delete current saved query";

    public static final String BUTTON_GROUP = "Set the function on group of fields";

    public static final String BUTTON_SAVE = "Save current query";

    public static final String BUTTON_SORTING = "Hide/show sorting row";

    public static final String BUTTON_TOTAL = "Hide/show total row";

    public static final String BUTTON_QUERY = "Test current query";

    private static final String PROMPT_QUERY_NAME_WITH_GROUP = "Enter the query name ( query group/query_name )";

    private static final String PROMPT_QUERY_NAME_WITHOUT_GROUP = "Enter the query name ( query_name )";

    private static final String QUERY_GROUP = "Queries";

    private static final String QUERY_NAME = "{0}/{1}";

    private static final String SQL_MENU = "sql-menu";

    private boolean saveChangesEnabled = true;

    private boolean createNewQueryGroupEnabled = true;

    private boolean showQueryData = true;

    private String queryGroup = null;

    private String queryName = null;

    private Button clearButton = null;

    private Button commonButton = null;

    private Button groupButton = null;

    private Button queryButton = null;

    private Button removeButton = null;

    private Button sortingButton = null;

    private Button saveButton = null;

    private Button totalButton = null;

    private TextField queryNameField = null;

    /**
     * Creates a new empty instance.
     */
    public SqlMenu()
    {
        addStyleName( SQL_MENU );
        addStyleName( ValoTheme.LAYOUT_CARD );
    }

    /**
     * Add button click listener.
     * 
     * @param clickListener
     *            button click listener
     */
    public void addClickListener( Button.ClickListener clickListener )
    {
        clearButton.addClickListener( clickListener );
        commonButton.addClickListener( clickListener );
        groupButton.addClickListener( clickListener );
        queryButton.addClickListener( clickListener );
        removeButton.addClickListener( clickListener );
        saveButton.addClickListener( clickListener );
        sortingButton.addClickListener( clickListener );
        totalButton.addClickListener( clickListener );
    }

    /**
     * Add value change listened.
     * 
     * @param valueChangeListener
     *            value change listened
     */
    public void addValueChangeListener( ValueChangeListener valueChangeListener )
    {
        queryNameField.addValueChangeListener( valueChangeListener );
    }

    /**
     * Set button active
     * 
     * @param buttonId
     *            button id
     * @param active
     *            if true button is active, otherwise is not
     */
    public void setButtonActive( String buttonId, boolean active )
    {
        if ( active )
        {
            switch ( buttonId )
            {
                case BUTTON_COMMON:
                    commonButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS );
                    commonButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    break;
                case BUTTON_SORTING:
                    sortingButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS );
                    sortingButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    break;
                case BUTTON_TOTAL:
                    totalButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS );
                    totalButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    break;
                default:
                    break;
            }
        }
        else
        {
            switch ( buttonId )
            {
                case BUTTON_COMMON:
                    commonButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    commonButton.addStyleName( ValoTheme.BUTTON_BORDERLESS );
                    break;
                case BUTTON_SORTING:
                    sortingButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    sortingButton.addStyleName( ValoTheme.BUTTON_BORDERLESS );
                    break;
                case BUTTON_TOTAL:
                    totalButton.removeStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
                    totalButton.addStyleName( ValoTheme.BUTTON_BORDERLESS );
                    break;
                default:
                    break;
            }
        }
    }

    /**
     * Check if save changes is enabled.
     * 
     * @return true if save changes is enabled, otherwise false
     */
    public boolean isSaveChangesEnabled()
    {
        return saveChangesEnabled;
    }

    /**
     * Set save changes enabled.
     * 
     * @param changeQueryEnabled
     *            enable save changes
     */
    public void setSaveChangesEnabled( boolean saveChangesEnabled )
    {
        this.saveChangesEnabled = saveChangesEnabled;
        removeButton.setVisible( saveChangesEnabled );
        saveButton.setVisible( saveChangesEnabled );
        queryNameField.setVisible( saveChangesEnabled );
    }

    /**
     * Check if clear button is enabled.
     * 
     * @return true if clear button is enabled, otherwise false
     */
    public boolean isClearButtonEnabled()
    {
        return clearButton.isEnabled();
    }

    /**
     * Set the clear button enabled.
     * 
     * @param enabled
     *            enable the clear button
     */
    public void setClearButtonEnabled( boolean enabled )
    {
        clearButton.setEnabled( enabled );
    }

    /**
     * Check if group button is enabled.
     * 
     * @return true if group button is enabled, otherwise false
     */
    public boolean isGroupButtonEnabled()
    {
        return groupButton.isEnabled();
    }

    /**
     * Set the group button enabled.
     * 
     * @param enabled
     *            enable the group button
     */
    public void setGroupButtonEnabled( boolean enabled )
    {
        groupButton.setEnabled( enabled );
    }

    /**
     * Check if query button is enabled.
     * 
     * @return true if query button is enabled, otherwise false
     */
    public boolean isQueryButtonEnabled()
    {
        return queryButton.isEnabled();
    }

    /**
     * Set query button enabled.
     * 
     * @param enabled
     *            enable query button
     */
    public void setQueryButtonEnabled( boolean enabled )
    {
        queryButton.setEnabled( enabled );
    }

    /**
     * Check if remove button is enabled.
     * 
     * @return true if remove button is enabled, otherwise false
     */
    public boolean isRemoveButtonEnabled()
    {
        return removeButton.isEnabled();
    }

    /**
     * Set the remove button enabled.
     * 
     * @param enabled
     *            enable the remove button
     */
    public void setRemoveButtonEnabled( boolean enabled )
    {
        removeButton.setEnabled( enabled );
    }

    /**
     * Check if save button is enabled.
     * 
     * @return true if save button is enabled, otherwise false
     */
    public boolean isSaveButtonEnabled()
    {
        return saveButton.isEnabled();
    }

    /**
     * Set save button enabled.
     * 
     * @param enabled
     *            enable save button
     */
    public void setSaveButtonEnabled( boolean enabled )
    {
        saveButton.setEnabled( enabled && !StringUtil.trim( queryNameField.getValue() )
                                                     .isEmpty() );
    }

    /**
     * Check if show query data is enabled.
     * 
     * @return true if show query data is enabled, otherwise false
     */
    public boolean isShowQueryDataEnabled()
    {
        return showQueryData;
    }

    /**
     * Set show query data enabled.
     * 
     * @param showQueryData
     *            if true query data show is enabled, otherwise not
     */
    public void setShowQueryDataEnabled( boolean showQueryData )
    {
        this.showQueryData = showQueryData;
        if ( SharedUtil.isNotNull( queryButton ) )
        {
            queryButton.setVisible( showQueryData );
        }
    }

    /**
     * Check if create a new group is enabled.
     * 
     * @return true if create a new group is enabled, otherwise false
     */
    public boolean isCreateNewQueryGroupEnabled()
    {
        return createNewQueryGroupEnabled;
    }

    /**
     * Set create a new group enabled.
     * 
     * @param createNewQueryGroupEnabled
     *            enable create a new query group
     */
    public void setCreateNewQueryGroupEnabled( boolean createNewQueryGroupEnabled )
    {
        this.createNewQueryGroupEnabled = createNewQueryGroupEnabled;
        if ( SharedUtil.isNotNullAndNotEmpty( queryNameField ) )
        {
            queryNameField.setInputPrompt( PROMPT_QUERY_NAME_WITH_GROUP );
            if ( !createNewQueryGroupEnabled )
            {
                queryNameField.setInputPrompt( PROMPT_QUERY_NAME_WITHOUT_GROUP );
            }
        }
    }

    /**
     * Get query group.
     * 
     * @return query group
     */
    public String getQueryGroup()
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) )
        {
            return queryGroup;
        }
        return QUERY_GROUP;
    }

    /**
     * Set query group.
     * 
     * @param queryGroup
     *            query group
     */
    public void setQueryGroup( String queryGroup )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) )
        {
            this.queryGroup = StringUtil.trim( queryGroup );
        }
    }

    /**
     * Get query group with query name.
     * 
     * @return query group with query name
     */
    public String getQueryGroupWithQueryName()
    {
        return MessageFormat.format( QUERY_NAME, getQueryGroup(), queryName );
    }

    /**
     * Get query name.
     * 
     * @return query name
     */
    public String getQueryName()
    {
        return queryName;
    }

    /**
     * Set query name.
     * 
     * @param queryName
     *            query name
     */
    public void setQueryName( String queryName )
    {
        String name = StringUtil.trim( queryName );
        if ( SharedUtil.isNotNullAndNotEmpty( name ) )
        {
            this.queryName = name;
            if ( this.queryName.contains( StringUtil.FORWARD_SLASH_VALUE )
                && SharedUtil.isNotNullAndNotEmpty( this.queryName.split( StringUtil.STRING_FORWARD_SLASH_SPLITTER )[1] ) )
            {
                this.queryName = StringUtil.trim( this.queryName.split( StringUtil.STRING_FORWARD_SLASH_SPLITTER )[1] );
                this.queryName = StringUtil.replaceSpacesWithUnderscores( this.queryName );
            }

            if ( createNewQueryGroupEnabled && name.contains( StringUtil.FORWARD_SLASH_VALUE )
                && SharedUtil.isNotNullAndNotEmpty( StringUtil.trim( name.split( StringUtil.STRING_FORWARD_SLASH_SPLITTER )[0] ) ) )
            {
                setQueryGroup( StringUtil.trim( name.split( StringUtil.STRING_FORWARD_SLASH_SPLITTER )[0] ) );
            }
        }
    }

    /**
     * Reset query name.
     */
    public void resetQueryName()
    {
        queryGroup = StringUtil.EMPTY_VALUE;
        queryName = StringUtil.EMPTY_VALUE;
        queryNameField.setValue( queryName );
    }

    /**
     * Get query name field instance.
     * 
     * @return query name instance
     */
    public TextField getQueryNameField()
    {
        return queryNameField;
    }

    /**
     * Prepare sql menu instance.
     */
    public void prepare()
    {
        // hide/show common row
        commonButton = new Button();
        commonButton.setData( BUTTON_COMMON );
        commonButton.setDescription( BUTTON_COMMON );
        commonButton.setIcon( FontAwesome.CALCULATOR );
        commonButton.addStyleName( ValoTheme.BUTTON_BORDERLESS );
        commonButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        commonButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( commonButton );

        // hide/show total row
        totalButton = new Button();
        totalButton.setData( BUTTON_TOTAL );
        totalButton.setDescription( BUTTON_TOTAL );
        totalButton.setIcon( FontAwesome.OBJECT_GROUP );
        totalButton.addStyleName( ValoTheme.BUTTON_BORDERLESS );
        totalButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        totalButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( totalButton );

        // hide/show sort row
        sortingButton = new Button();
        sortingButton.setData( BUTTON_SORTING );
        sortingButton.setDescription( BUTTON_SORTING );
        sortingButton.setIcon( FontAwesome.SORT_ALPHA_ASC );
        sortingButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        sortingButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        sortingButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( sortingButton );

        // set the function on group of fields
        groupButton = new Button();
        groupButton.setData( BUTTON_GROUP );
        groupButton.setDescription( BUTTON_GROUP );
        groupButton.setEnabled( false );
        groupButton.setIcon( FontAwesome.SEARCH_PLUS );
        groupButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        groupButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        groupButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( groupButton );

        // text field for query name
        queryNameField = new TextField();
        queryNameField.setColumns( TEXT_FIELD_LENGTH );
        queryNameField.setInputPrompt( PROMPT_QUERY_NAME_WITH_GROUP );
        if ( !createNewQueryGroupEnabled )
        {
            queryNameField.setInputPrompt( PROMPT_QUERY_NAME_WITHOUT_GROUP );
        }
        queryNameField.addStyleName( ValoTheme.TEXTFIELD_TINY );
        addComponent( queryNameField );

        // save current query
        saveButton = new Button();
        saveButton.setData( BUTTON_SAVE );
        saveButton.setDescription( BUTTON_SAVE );
        saveButton.setEnabled( false );
        saveButton.setIcon( FontAwesome.SAVE );
        saveButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        saveButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        saveButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( saveButton );

        // remove current saved query
        removeButton = new Button();
        removeButton.setData( BUTTON_DELETE );
        removeButton.setDescription( BUTTON_DELETE );
        removeButton.setEnabled( false );
        removeButton.setIcon( FontAwesome.REMOVE );
        removeButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        removeButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        removeButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( removeButton );

        // clear table container
        clearButton = new Button();
        clearButton.setData( BUTTON_CLEAR );
        clearButton.setDescription( BUTTON_CLEAR );
        clearButton.setEnabled( false );
        clearButton.setIcon( FontAwesome.ERASER );
        clearButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        clearButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        clearButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( clearButton );

        // test current query
        queryButton = new Button();
        queryButton.setData( BUTTON_QUERY );
        queryButton.setDescription( BUTTON_QUERY );
        queryButton.setEnabled( false );
        queryButton.setIcon( FontAwesome.TABLE );
        queryButton.setVisible( showQueryData );
        queryButton.addStyleName( ValoTheme.BUTTON_BORDERLESS_COLORED );
        queryButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        queryButton.addStyleName( ValoTheme.BUTTON_SMALL );
        addComponent( queryButton );
    }

}
