/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.dussan.vaadin.dmenu.DMenu;
import org.dussan.vaadin.dmenu.menuitem.MenuItem;
import org.dussan.vaadin.dmenu.menuitem.MenuItemElement;
import org.dussan.vaadin.dquery.DQuery;
import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.enums.JsonQueryElement;
import org.dussan.vaadin.dquery.enums.SqlDriver;
import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.json.JsonQuery;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONObject;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.ui.ListSelect;

public class DataSourcesMenu
    extends DMenu
{

    private static final long serialVersionUID = -4568749185695417903L;

    private static final String DATABASE_NAME_IS_NOT_SPECIFIED_IN_THE_DATABASE_URL =
        "Database name is not specified in the database url: {0}";

    private static final String COLUMN_NAME = "COLUMN_NAME";

    private static final String DATA_QUERY_ID = "{0}.{1}.{2}.{3}." + TableType.QUERY.toString();

    private static final String DATA_SOURCES_LIST = "data-sources-list";

    private static final String DATA_TABLE_ID = "{0}.{1}.{2}.{3}." + TableType.TABLE.toString();

    private static final String TABLE_ID = "{0}.{1}";

    private static final String TABLE_NAME = "TABLE_NAME";

    private static final String TABLES_AREA = "tables-area";

    private static final String TABLES = "Tables: {0}";

    public static final String DATA_SOURCE = "Data source";

    public static final String DATA_SOURCES = "DATA SOURCES";

    private boolean initialized = false;

    private boolean dataSourcesItemVisible = false;

    private int sourcesPageLength = 0;

    private Map<String, Boolean> dataSourceEnabled = null;

    private Map<String, Boolean> dataSourceTableDisabled = null;

    private Map<String, JdbcUtil> dataSources = null;

    /**
     * Creates a new instance.
     */
    public DataSourcesMenu()
    {
        dataSourceEnabled = new LinkedHashMap<>();
        dataSourceTableDisabled = new LinkedHashMap<>();
        dataSources = new LinkedHashMap<>();

        addStyleName( TABLES_AREA );
        setFloatingMenu( false );
        setSourcesPageLength( 4 );
    }

    /**
     * Check if source menu is initialized.
     * 
     * @return true if source menu is initialized, otherwise false
     */
    public boolean isInitialized()
    {
        return initialized;
    }

    /**
     * Set source menu initialized.
     * 
     * @param initialized
     *            if true source menu is initialized, otherwise is not initialized
     */
    private void setInitialized( boolean initialized )
    {
        this.initialized = initialized;
    }

    /**
     * Check if data sources item is visible.
     * 
     * @return true if data sources item is visible, otherwise false
     */
    public boolean isDataSourcesItemVisible()
    {
        return dataSourcesItemVisible;
    }

    /**
     * Set data sources item visible.
     * 
     * @param visible
     *            if true data sources item is visible, otherwise is hidden
     */
    public void setDataSourcesItemVisible( boolean visible )
    {
        dataSourcesItemVisible = visible;
        for ( int index = 0; index < getComponentCount(); index++ )
        {
            MenuItem menuItem = getMenuItem( index );
            if ( SharedUtil.isNotNullAndNotEmpty( menuItem.getData() ) && DATA_SOURCES.equals( menuItem.getData() ) )
            {
                getTab( index ).setVisible( visible );
                break;
            }
        }
    }

    /**
     * Get sources page length.
     * 
     * @return sources page length
     */
    public int getSourcesPageLength()
    {
        return sourcesPageLength;
    }

    /**
     * Set sources page length.
     * 
     * @param visibleSourcesRows
     *            sources page length
     */
    public void setSourcesPageLength( int sourcesPageLength )
    {
        this.sourcesPageLength = sourcesPageLength;
    }

    /**
     * Get data sources.
     * 
     * @return data sources
     */
    public Map<String, JdbcUtil> getDataSources()
    {
        return dataSources;
    }

    /**
     * Set data sources.
     * 
     * @param dataSources
     *            data sources
     */
    public void setDataSources( Map<String, JdbcUtil> dataSources )
    {
        this.dataSources.putAll( dataSources );
    }

    /**
     * Check if data source is defined.
     * 
     * @param dataSourceId
     *            data source id
     * @return true if data source is defined, otherwise false
     */
    private boolean isDataSourceDefined( String dataSourceId )
    {
        return dataSourceEnabled.containsKey( dataSourceId );
    }

    /**
     * Check if data source is enabled.
     * 
     * @param dataSourceId
     *            data source id
     * @return true data source is enabled, otherwise false
     */
    public boolean isDataSourceEnabled( String dataSourceId )
    {
        if ( isDataSourceDefined( dataSourceId ) )
        {
            return dataSourceEnabled.get( dataSourceId );
        }
        return false;
    }

    /**
     * Set data source enabled. User can later change a state of the data source if data source menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param enabled
     *            if true data source is enabled, otherwise is disabled
     */
    public void setDataSourceEnabled( String dataSourceId, boolean enabled )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            dataSourceEnabled.put( dataSourceId, enabled );
            for ( int index = 0; index < getComponentCount(); index++ )
            {
                MenuItem menuItem = getMenuItem( index );
                if ( getTab( index ).isVisible() != enabled && SharedUtil.isNotNullAndNotEmpty( menuItem.getData() )
                    && dataSourceId.equals( menuItem.getData() ) )
                {
                    getTab( index ).setVisible( enabled );
                    break;
                }
            }
        }
    }

    /**
     * Check if data source table is defined.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @return true if data source table is defined, otherwise false
     */
    private boolean isDataSourceTableDefined( String dataSourceId, String tableName )
    {
        String tableId = MessageFormat.format( TABLE_ID, dataSourceId, tableName );
        return dataSourceTableDisabled.containsKey( tableId );
    }

    /**
     * Check if data source table is disabled. User can later change a state of the data source table if data source
     * menu is visible.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @return true data source table is disabled, otherwise false
     */
    public boolean isDataSourceTableDisabled( String dataSourceId, String tableName )
    {
        String tableId = MessageFormat.format( TABLE_ID, dataSourceId, tableName );
        if ( isDataSourceTableDefined( dataSourceId, tableName ) )
        {
            return dataSourceTableDisabled.get( tableId );
        }
        return false;
    }

    /**
     * Set data source table disabled.
     * 
     * @param dataSourceId
     *            data source id
     * @param tableName
     *            table name
     * @param disabled
     *            if true data source table is disabled, otherwise is enabled
     */
    public void setDataSourceTableDisabled( String dataSourceId, String tableName, boolean disabled )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && SharedUtil.isNotNullAndNotEmpty( tableName ) )
        {
            dataSourceTableDisabled.put( MessageFormat.format( TABLE_ID, dataSourceId, tableName ), disabled );
            for ( int index = 0; index < getComponentCount(); index++ )
            {
                MenuItem menuItem = getMenuItem( index );
                if ( SharedUtil.isNotNullAndNotEmpty( menuItem.getData() )
                    && DATA_SOURCES.equals( menuItem.getData() ) )
                {
                    for ( MenuItemElement menuItemElement : menuItem.getMenuItemElements() )
                    {
                        if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() )
                            && dataSourceId.equals( menuItemElement.getData() ) )
                        {
                            if ( disabled )
                            {
                                ( (ListSelect) menuItemElement.getMenuItemElementComponent() ).unselect( tableName );
                            }
                            else
                            {
                                ( (ListSelect) menuItemElement.getMenuItemElementComponent() ).select( tableName );
                            }
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }

    /**
     * Add table item menu.
     * 
     * @param dataSourceId
     *            data source id
     * @param databaseName
     *            database name
     * @param tableName
     *            table name
     * @param dataSourceDescription
     *            data source description
     * @param tableColumnsName
     *            table columns name
     * @param visible
     *            if true table item is visible, otherwise not
     */
    private void addTableItem( String dataSourceId, String databaseName, String tableName, String dataSourceDescription,
                               List<String> tableColumnsName, boolean visible )
    {
        ListSelect tableColumnsList = new ListSelect();
        tableColumnsList.setRows( getSourcesPageLength() );
        tableColumnsList.setMultiSelect( true );
        tableColumnsList.setNullSelectionAllowed( false );
        tableColumnsList.setSizeFull();

        for ( String columnName : tableColumnsName )
        {
            tableColumnsList.addItem( columnName );
            tableColumnsList.select( columnName );
        }

        if ( !tableColumnsList.isEmpty() )
        {
            // DATATABLE_ID format: data source id, database name, table name, table columns
            MenuItemElement tableItemElement = new MenuItemElement( tableName, tableColumnsList );
            tableItemElement.setVisible( visible );
            tableItemElement.setData( MessageFormat.format( DATA_TABLE_ID, dataSourceId, databaseName, tableName,
                                                            Arrays.toString( tableColumnsList.getItemIds()
                                                                                             .toArray() ) ) );

            MenuItem tableItem = null;
            for ( MenuItem menuItem : getMenuItems() )
            {
                if ( dataSourceId.equals( menuItem.getData() ) )
                {
                    tableItem = menuItem;
                    break;
                }
            }

            if ( SharedUtil.isNull( tableItem ) )
            {
                tableItem = new MenuItem( dataSourceDescription );
                tableItem.setData( dataSourceId );
                tableItem.setMenuItemElementsDraggable( true );
                tableItem.addMenuItemElement( tableItemElement );
                addMenuItem( tableItem );
            }
            else
            {
                tableItem.addMenuItemElement( tableItemElement );
            }
        }
    }

    /**
     * Add data source item menu element to source menu.
     * 
     * @param dataSourcesItem
     *            data source menu item
     * @param dataSourceId
     *            data source id
     * @param dataSource
     *            data source
     */
    private void addDataSourceItemElement( MenuItem dataSourcesItem, String dataSourceId, JdbcUtil dataSource )
    {
        boolean tablesItemElementExist = false;
        for ( MenuItemElement itemElement : dataSourcesItem.getMenuItemElements() )
        {
            if ( dataSourceId.equals( itemElement.getData() ) )
            {
                tablesItemElementExist = true;
                break;
            }
        }

        if ( !tablesItemElementExist )
        {
            // data sources menu item element with table names
            ListSelect tablesList = new ListSelect();
            tablesList.setRows( getSourcesPageLength() );
            tablesList.setSizeFull();
            tablesList.setImmediate( true );
            tablesList.setMultiSelect( true );
            tablesList.setNullSelectionAllowed( false );

            try
            {
                String databaseName;
                Map<String, List<String>> tables = new LinkedHashMap<>();

                if ( SqlDriver.OFFLINE_MODE.equals( dataSource.getDriver() ) )
                {
                    // database name
                    databaseName = dataSource.getCatalog();

                    // save table name and table columns name
                    for ( String table : dataSource.getTables() )
                    {
                        tables.put( table, Arrays.asList( dataSource.getColumns( table ) ) );
                    }
                }
                else
                {
                    Connection connection = dataSource.getConnection();
                    DatabaseMetaData metaData = connection.getMetaData();

                    // catalog or schema cannot be empty
                    if ( connection.getCatalog() == null && connection.getSchema() == null )
                    {
                        throw new IllegalArgumentException( MessageFormat.format( DATABASE_NAME_IS_NOT_SPECIFIED_IN_THE_DATABASE_URL,
                                                                                  dataSource.getUrl() ) );
                    }

                    // database name
                    databaseName = connection.getCatalog();
                    if ( connection.getCatalog() == null )
                    {
                        databaseName = connection.getSchema();
                    }

                    // collect table names from data source
                    ResultSet tempTables =
                        metaData.getTables( connection.getCatalog(), connection.getSchema(), null, null );
                    while ( tempTables.next() )
                    {
                        String tableName = tempTables.getString( TABLE_NAME );
                        List<String> tableColumnsName = new ArrayList<>();

                        // collect table columns name from table
                        ResultSet tableColumns = metaData.getColumns( null, null, tableName, null );
                        while ( tableColumns.next() )
                        {
                            tableColumnsName.add( tableColumns.getString( COLUMN_NAME ) );
                        }
                        tableColumns.close();

                        // save table name and table columns name
                        if ( !tableColumnsName.isEmpty() )
                        {
                            tables.put( tableName, tableColumnsName );
                        }
                    }
                    tempTables.close();
                }

                // collect table names from data source
                for ( Entry<String, List<String>> table : tables.entrySet() )
                {
                    String tableName = table.getKey();
                    List<String> tableColumnsName = table.getValue();

                    if ( !tableColumnsName.isEmpty() )
                    {
                        // data source table is enabled/disabled
                        boolean tableSelected = true;
                        if ( isDataSourceTableDefined( dataSourceId, tableName ) )
                        {
                            tableSelected = isDataSourceTableDisabled( dataSourceId, tableName );
                        }

                        addTableItem( dataSourceId, databaseName, tableName, dataSource.getDescription(),
                                      tableColumnsName, tableSelected );

                        // data source table selected
                        tablesList.addItem( tableName );
                        if ( tableSelected )
                        {
                            tablesList.select( tableName );
                        }
                        else
                        {
                            tablesList.unselect( tableName );
                        }
                    }

                }

                if ( !tablesList.isEmpty() )
                {
                    // add source tables item element to data source item
                    MenuItemElement menuItemElement =
                        new MenuItemElement( MessageFormat.format( TABLES, dataSource.getDescription() ), tablesList );
                    menuItemElement.setData( dataSourceId );
                    dataSourcesItem.addMenuItemElement( menuItemElement );

                    // table list value change event listener
                    tablesList.addValueChangeListener( new ValueChangeListener()
                    {
                        private static final long serialVersionUID = 6761242121554232760L;

                        @Override
                        public void valueChange( ValueChangeEvent event )
                        {
                            Object dataSourceId = ( (MenuItemElement) ( (ListSelect) event.getProperty() ).getParent()
                                                                                                          .getParent()
                                                                                                          .getParent()
                                                                                                          .getParent() ).getData();
                            if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
                            {
                                for ( MenuItem menuItem : getMenuItems() )
                                {
                                    // menu item with data source id
                                    if ( dataSourceId.toString()
                                                     .equals( menuItem.getData() ) )
                                    {
                                        Set<?> values = (Set<?>) event.getProperty()
                                                                      .getValue();
                                        for ( MenuItemElement menuItemElement : menuItem.getMenuItemElements() )
                                        {
                                            if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() ) )
                                            {
                                                // DATATABLE_ID format: data source id, database name, table name
                                                String[] tableName = menuItemElement.getData()
                                                                                    .toString()
                                                                                    .split( StringUtil.STRING_DOT_SPLITTER );
                                                if ( SharedUtil.isNotNull( tableName ) && 3 <= tableName.length )
                                                {
                                                    menuItemElement.setVisible( values.contains( tableName[2] ) );
                                                }
                                            }
                                        }
                                        break;
                                    }
                                }
                            }
                        }
                    } );
                }
            }
            catch ( SQLException e )
            {
                Logger.getLogger( DQuery.class.getName() )
                      .log( Level.INFO, e.getLocalizedMessage(), e );
            }
            finally
            {
                dataSource.close();
            }
        }
    }

    /**
     * Add data source menu item.
     * 
     * @param dataSourceId
     *            data source id
     * @param dataSource
     *            data source
     */
    public void addDataSourceItem( final String dataSourceId, final JdbcUtil dataSource )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            dataSources.put( dataSourceId, dataSource );
        }

        MenuItem dataSourcesItem = null;
        if ( !isInitialized() )
        {
            // create data sources menu item to main menu
            dataSourcesItem = new MenuItem( DATA_SOURCES );
            dataSourcesItem.setData( DATA_SOURCES );
        }
        else
        {
            // data source menu item
            for ( MenuItem menuItem : getMenuItems() )
            {
                if ( DATA_SOURCES.equals( menuItem.getData() ) )
                {
                    dataSourcesItem = menuItem;
                    break;
                }
            }
        }

        // define data source and enable it
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && !isDataSourceDefined( dataSourceId ) )
        {
            setDataSourceEnabled( dataSourceId, true );
        }

        // data source menu item element
        MenuItemElement dataSourcesItemElement = null;
        for ( MenuItemElement menuItemElement : dataSourcesItem.getMenuItemElements() )
        {
            if ( DATA_SOURCE.equals( menuItemElement.getData() ) )
            {
                dataSourcesItemElement = menuItemElement;
                break;
            }
        }

        // data sources list
        ListSelect dataSourcesList = new ListSelect();
        if ( SharedUtil.isNull( dataSourcesItemElement ) )
        {
            // add data sources menu item element to data sources menu
            dataSourcesItemElement = new MenuItemElement( DATA_SOURCE, dataSourcesList );
            dataSourcesItemElement.setData( DATA_SOURCE );
            dataSourcesList.setImmediate( true );
            dataSourcesList.setMultiSelect( true );
            dataSourcesList.setNullSelectionAllowed( false );
            dataSourcesList.setRows( getSourcesPageLength() );
            dataSourcesList.setSizeFull();
            dataSourcesList.addStyleName( DATA_SOURCES_LIST );
            dataSourcesItem.addMenuItemElement( dataSourcesItemElement );
        }
        else
        {
            dataSourcesList = (ListSelect) dataSourcesItemElement.getMenuItemElementComponent();
        }

        // data source list value change event listener
        final MenuItem cloneDataSourcesItem = dataSourcesItem;
        dataSourcesList.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = 1967058519175394515L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                Set<?> values = (Set<?>) event.getProperty()
                                              .getValue();
                for ( MenuItemElement menuItemElement : cloneDataSourcesItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( menuItemElement.getData() ) )
                    {
                        String dataSourceId = menuItemElement.getData()
                                                             .toString();
                        if ( isDataSourceDefined( dataSourceId ) )
                        {
                            String description = getDataSources().get( dataSourceId )
                                                                 .getDescription();
                            menuItemElement.setVisible( values.contains( description ) );
                            setDataSourceEnabled( dataSourceId, values.contains( description ) );
                        }
                    }
                }
            }
        } );

        // add data source to data source list
        String description = dataSource.getDescription();
        if ( SharedUtil.isNotNullAndNotEmpty( description ) )
        {
            dataSourcesList.addItem( description );
            dataSourcesList.select( description );
        }

        // initialize data source menu item
        if ( !isInitialized() )
        {
            addMenuItem( dataSourcesItem );
        }

        // add data sources menu item element to data source menu item
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            addDataSourceItemElement( dataSourcesItem, dataSourceId, dataSource );
        }
    }

    /**
     * Prepare data source menu.
     * 
     * @param dataSources
     *            map of data sources
     */
    public void prepare( Map<String, JdbcUtil> dataSources )
    {
        // enable to initialize data source menu item even if data sources are empty
        if ( dataSources.isEmpty() )
        {
            dataSources.put( StringUtil.EMPTY_VALUE, new JdbcUtil() );
        }

        // data sources
        for ( Entry<String, JdbcUtil> dataSourceEntry : dataSources.entrySet() )
        {
            String dataSourceId = dataSourceEntry.getKey();
            JdbcUtil dataSource = dataSourceEntry.getValue();

            // add data source menu item
            addDataSourceItem( dataSourceId, dataSource );

            // set data source menu initialized
            setInitialized( true );
        }

        // data sources menu item visibility
        setDataSourcesItemVisible( isDataSourcesItemVisible() );
    }

    /**
     * Add query item element.
     * 
     * @param jsonQuery
     *            json query
     * @return table item element
     */
    public void addQueryItemElement( JsonQuery jsonQuery )
    {
        MenuItemElement queryItemElement = null;
        if ( SharedUtil.isNotNull( jsonQuery ) )
        {
            // collect table column names
            ListSelect tableColumnsList = new ListSelect();
            tableColumnsList.setRows( getSourcesPageLength() );
            tableColumnsList.setMultiSelect( true );
            tableColumnsList.setNullSelectionAllowed( false );
            tableColumnsList.setSizeFull();

            List<String> fields = new ArrayList<>();
            JSONArray jsonFields = (JSONArray) jsonQuery.get( JsonQueryElement.FIELDS );
            for ( int index = 0; index < jsonFields.length(); index++ )
            {
                JSONObject field = (JSONObject) jsonFields.get( index );
                fields.add( field.getInt( JsonQuery.TABLE_FIELD ), field.getString( BaseTable.DESCRIPTION_NAME ) );
            }
            for ( String field : fields )
            {
                tableColumnsList.addItem( field );
                tableColumnsList.select( field );
            }

            // DATATABLE_ID format: data source id, query, table name, table columns
            queryItemElement = new MenuItemElement( jsonQuery.get( JsonQueryElement.NAME )
                                                             .toString(),
                                                    tableColumnsList );
            queryItemElement.setData( MessageFormat.format( DATA_QUERY_ID,
                                                            jsonQuery.get( JsonQueryElement.DATA_SOURCE_ID ),
                                                            jsonQuery.get( JsonQueryElement.GROUP ),
                                                            jsonQuery.get( JsonQueryElement.NAME ),
                                                            Arrays.toString( fields.toArray() ) ) );
        }

        // add menu item element to menu item
        if ( SharedUtil.isNotNull( queryItemElement ) )
        {
            String jsonQueryGroup = jsonQuery.get( JsonQueryElement.GROUP )
                                             .toString();

            // get existing menu item
            MenuItem queryMenuItem = null;
            for ( MenuItem menuItem : getMenuItems() )
            {
                if ( menuItem.getCaption()
                             .equals( jsonQueryGroup ) )
                {
                    queryMenuItem = menuItem;
                    break;
                }
            }

            if ( SharedUtil.isNull( queryMenuItem ) )
            {
                // create new menu item
                queryMenuItem = new MenuItem( jsonQueryGroup );
                queryMenuItem.setMenuItemElementsDraggable( true );
                queryMenuItem.addMenuItemElement( queryItemElement );
                queryMenuItem.setData( jsonQueryGroup );
                addMenuItem( queryMenuItem );
            }
            else
            {
                // save menu item
                MenuItemElement menuItemElement = null;
                for ( MenuItemElement itemElement : queryMenuItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( itemElement.getData() ) )
                    {
                        String[] data = itemElement.getData()
                                                   .toString()
                                                   .split( StringUtil.STRING_DOT_SPLITTER );
                        if ( 3 <= data.length && data[0].equals( jsonQuery.get( JsonQueryElement.DATA_SOURCE_ID ) )
                            && data[2].equals( jsonQuery.get( JsonQueryElement.NAME ) ) )
                        {
                            menuItemElement = itemElement;
                            break;
                        }
                    }
                }

                // save menu item element
                queryMenuItem.setData( jsonQueryGroup );
                queryMenuItem.setMenuItemElementsDraggable( true );
                if ( SharedUtil.isNull( menuItemElement ) )
                {
                    // add new menu item element
                    queryMenuItem.addMenuItemElement( queryItemElement );
                }
                else
                {
                    // replace existing menu item element
                    int index = queryMenuItem.getComponentIndex( menuItemElement );
                    queryMenuItem.removeComponent( menuItemElement );
                    queryMenuItem.addMenuItemElement( queryItemElement, index );
                }
            }
        }
    }

    public void removeQueryItemElement( String jsonQueryGroup, String jsonQueryName )
    {
        // get query menu item
        for ( MenuItem menuItem : getMenuItems() )
        {
            if ( menuItem.getCaption()
                         .equals( jsonQueryGroup ) )
            {
                // get and remove query menu item element
                for ( MenuItemElement itemElement : menuItem.getMenuItemElements() )
                {
                    if ( SharedUtil.isNotNullAndNotEmpty( itemElement.getCaption() ) && itemElement.getCaption()
                                                                                                   .equals( jsonQueryName ) )
                    {
                        menuItem.removeComponent( itemElement );
                        break;
                    }
                }

                // if query menu item is empty remove it from data source menu
                if ( menuItem.isEmpty() )
                {
                    removeMenuItem( getMenuItemIndex( menuItem ) );
                }
                break;
            }
        }
    }

}
