/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.json;

import java.io.Serializable;
import java.text.MessageFormat;

import org.dussan.vaadin.dquery.enums.JsonQueryElement;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONObject;

public class JsonQuery
    implements Serializable
{

    private static final long serialVersionUID = 23010531792953775L;

    private static final String QUERY_AS_COLUMN = "({0})";

    public static final String DATA_SOURCE_ID = "datasource_id";

    public static final String DATABASE = "database";

    public static final String TABLE = "table";

    public static final String TABLE_ALIAS = "alias";

    public static final String TABLE_FIELD = "field";

    public static final String TABLE_JOINS_FROM = "from";

    public static final String TABLE_JOINS_FROM_FIELDS = "from_fields";

    public static final String TABLE_JOINS_TYPE = "type";

    public static final String TABLE_JOINS_TO = "to";

    public static final String TABLE_JOINS_TO_FIELDS = "to_fields";

    public static final String TABLE_POSITION = "position";

    public static final String TABLE_POSITION_LEFT = "left";

    public static final String TABLE_POSITION_TOP = "top";

    public static final String TABLE_TYPE = "type";

    private String dataSourceId = null;

    private String queryGroup = null;

    private String queryName = null;

    private JSONObject jsonData = null;

    /**
     * Creates a new instance.
     */
    private JsonQuery()
    {
        jsonData = new JSONObject();
    }

    /**
     * Creates a new instance with query group and query name.
     */
    public JsonQuery( String queryGroup, String queryName )
    {
        this();
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName ) )
        {
            this.queryGroup = queryGroup;
            this.queryName = queryName;
        }
    }

    /**
     * Get json query element object.
     * 
     * @param jsonQueryElement
     *            json query element
     * @return json query element object
     */
    public Object get( JsonQueryElement jsonQueryElement )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( jsonQueryElement ) )
        {
            switch ( jsonQueryElement )
            {
                case DATA_SOURCE_ID:
                case GROUP:
                case NAME:
                case QUERY:
                    return jsonData.getString( jsonQueryElement.toString() );
                case QUERY_AS_COLUMN:
                    String query = jsonData.getString( JsonQueryElement.QUERY.toString() );
                    query = query.replace( StringUtil.SEMI_COLON_VALUE, StringUtil.EMPTY_VALUE );
                    return StringUtil.removeTabsAndNewLines( MessageFormat.format( QUERY_AS_COLUMN, query ) );
                case FIELDS:
                case JOINS:
                case TABLES:
                    return jsonData.getJSONArray( jsonQueryElement.toString() );
                default:
                    break;
            }
        }
        return null;
    }

    /**
     * Get json query object.
     * 
     * @return json query object
     */
    public JSONObject getJsonData()
    {
        return jsonData;
    }

    /**
     * Add table.
     * 
     * @param left
     *            table left position
     * @param top
     *            table top position
     * @param dataSourceId
     *            data source id
     * @param database
     *            table database
     * @param table
     *            table
     * @param tableAlias
     *            table alias
     * @param tableType
     *            tableType
     * @return json query object
     */
    public JsonQuery addTable( int left, int top, String dataSourceId, String database, String table, String tableAlias,
                               String tableType )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && SharedUtil.isNotNullAndNotEmpty( database )
            && SharedUtil.isNotNullAndNotEmpty( table ) && SharedUtil.isNotNullAndNotEmpty( tableAlias ) )
        {
            // table
            JSONObject jsonTable = new JSONObject();
            jsonTable.put( DATA_SOURCE_ID, dataSourceId );
            jsonTable.put( DATABASE, database );
            if ( table.contains( StringUtil.DOT_VALUE ) )
            {
                jsonTable.put( TABLE, table.split( StringUtil.STRING_DOT_SPLITTER )[1] );
            }
            else
            {
                jsonTable.put( TABLE, table );
            }
            jsonTable.put( TABLE_ALIAS, tableAlias );
            jsonTable.put( TABLE_TYPE, tableType );
            this.dataSourceId = dataSourceId;

            // table position
            JSONObject jsonTablePosition = new JSONObject();
            jsonTablePosition.put( TABLE_POSITION_LEFT, left );
            jsonTablePosition.put( TABLE_POSITION_TOP, top );
            jsonTable.put( TABLE_POSITION, jsonTablePosition );

            // tables
            JSONArray jsonTables = new JSONArray();
            if ( jsonData.has( JsonQueryElement.TABLES.toString() ) )
            {
                jsonTables = jsonData.getJSONArray( JsonQueryElement.TABLES.toString() );
            }
            jsonTables.put( jsonTable );

            // save query
            jsonData.put( JsonQueryElement.GROUP.toString(), queryGroup );
            jsonData.put( JsonQueryElement.NAME.toString(), queryName );
            jsonData.put( JsonQueryElement.DATA_SOURCE_ID.toString(), this.dataSourceId );
            jsonData.put( JsonQueryElement.TABLES.toString(), jsonTables );
        }
        return this;
    }

    /**
     * Add table joins.
     * 
     * @param fromTable
     *            from table
     * @param fromFields
     *            from fields
     * @param toTable
     *            to table
     * @param toFields
     *            to fields
     * @param tablesJoin
     *            tables join type
     * @return json query object
     */
    public JsonQuery addTableJoins( String fromTable, String[] fromFields, String toTable, String[] toFields,
                                    TablesJoin tablesJoin )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( fromTable ) && SharedUtil.isNotNullAndNotEmpty( toTable )
            && SharedUtil.isNotNull( fromFields ) && SharedUtil.isNotNull( toFields ) && fromFields.length != 0
            && toFields.length != 0 )
        {
            // table join
            JSONObject jsonJoin = new JSONObject();
            jsonJoin.put( TABLE_JOINS_FROM, fromTable );
            jsonJoin.put( TABLE_JOINS_FROM_FIELDS, fromFields );
            jsonJoin.put( TABLE_JOINS_TO, toTable );
            jsonJoin.put( TABLE_JOINS_TO_FIELDS, toFields );
            jsonJoin.put( TABLE_JOINS_TYPE, tablesJoin.toString() );

            // joins
            JSONArray jsonJoins = new JSONArray();
            if ( jsonData.has( JsonQueryElement.JOINS.toString() ) )
            {
                jsonJoins = jsonData.getJSONArray( JsonQueryElement.JOINS.toString() );
            }
            jsonJoins.put( jsonJoin );

            // save query
            jsonData.put( JsonQueryElement.GROUP.toString(), queryGroup );
            jsonData.put( JsonQueryElement.NAME.toString(), queryName );
            jsonData.put( JsonQueryElement.DATA_SOURCE_ID.toString(), dataSourceId );
            jsonData.put( JsonQueryElement.JOINS.toString(), jsonJoins );
        }
        return this;
    }

    /**
     * Add table fields.
     * 
     * @param columnId
     *            column id
     * @param rowIds
     *            row ids
     * @param rowValues
     *            row values
     * @return json query object
     */
    public JsonQuery addTableField( String columnId, Object[] rowIds, Object[] rowValues )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( columnId ) && SharedUtil.isNotNull( rowIds )
            && SharedUtil.isNotNull( rowValues ) && rowIds.length != 0 && rowValues.length != 0 )
        {
            // table fields
            JSONObject jsonField = new JSONObject();
            for ( int index = 0; index < Math.min( rowIds.length, rowValues.length ); index++ )
            {
                if ( SharedUtil.isNotNullAndNotEmpty( rowIds[index] )
                    && SharedUtil.isNotNullAndNotEmpty( rowValues[index] ) )
                {
                    jsonField.put( rowIds[index].toString(), rowValues[index] );
                }
            }
            jsonField.put( TABLE_FIELD, columnId );

            // fields
            JSONArray jsonFields = new JSONArray();
            if ( jsonData.has( JsonQueryElement.FIELDS.toString() ) )
            {
                jsonFields = jsonData.getJSONArray( JsonQueryElement.FIELDS.toString() );
            }
            jsonFields.put( jsonField );

            // save query
            jsonData.put( JsonQueryElement.GROUP.toString(), queryGroup );
            jsonData.put( JsonQueryElement.NAME.toString(), queryName );
            jsonData.put( JsonQueryElement.DATA_SOURCE_ID.toString(), dataSourceId );
            jsonData.put( JsonQueryElement.FIELDS.toString(), jsonFields );
        }
        return this;
    }

    /**
     * Add query.
     * 
     * @param query
     *            query
     * @return json query object
     */
    public JsonQuery addQuery( String query )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( queryGroup ) && SharedUtil.isNotNullAndNotEmpty( queryName )
            && SharedUtil.isNotNullAndNotEmpty( query ) )
        {
            // save query
            jsonData.put( JsonQueryElement.GROUP.toString(), queryGroup );
            jsonData.put( JsonQueryElement.NAME.toString(), queryName );
            jsonData.put( JsonQueryElement.DATA_SOURCE_ID.toString(), dataSourceId );
            jsonData.put( JsonQueryElement.QUERY.toString(), query );
        }
        return this;
    }

    /**
     * Return json qouery object as string.
     */
    @Override
    public String toString()
    {
        return jsonData.toString();
    }

}
