/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui.windows;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.base.ui.BaseWindow;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.ui.SqlTabs;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.server.FontAwesome;
import com.vaadin.ui.Button;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.ComboBox;
import com.vaadin.ui.CssLayout;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Panel;
import com.vaadin.ui.TwinColSelect;
import com.vaadin.ui.UI;
import com.vaadin.ui.themes.ValoTheme;

public class SqlFunctionsWindow
    extends BaseWindow
{

    private static final long serialVersionUID = 7502337344476902854L;

    private static final String BUTTON_PARAMETERS = "Change function parameter values";

    private static final String FIELDS_AVALIABLE = "Available fields";

    private static final String FIELDS_SELECTED = "Selected fields";

    private static final String MODAL_CAPTION = "Set function on a group of selected fields";

    private static final String SQL_FUNCTION = "Sql";

    private static final String SQL_FUNCTIONS = "{0} functions";

    private SqlTabs sqlTabs;

    /**
     * Creates a new instance.
     */
    private SqlFunctionsWindow()
    {
        setCaption( MODAL_CAPTION );
    }

    /**
     * Creates a new instance.
     * 
     * @param sqlTabs
     *            sqlTabs instance
     */
    public SqlFunctionsWindow( final SqlTabs sqlTabs )
    {
        this();
        setSizeUndefined();
        this.sqlTabs = sqlTabs;

        // sql functions panel
        Panel panel = new Panel( MessageFormat.format( SQL_FUNCTIONS, SQL_FUNCTION ) );
        addContextComponent( panel );

        // common, sorting or total functions
        final ComboBox sqlComboBox = new ComboBox();
        sqlComboBox.setNewItemsAllowed( false );
        sqlComboBox.setNullSelectionAllowed( false );
        sqlComboBox.addStyleName( ValoTheme.COMBOBOX_TINY );

        // sql functions
        final ComboBox functionsComboBox = new ComboBox();
        functionsComboBox.setNewItemsAllowed( false );
        functionsComboBox.setNullSelectionAllowed( false );
        functionsComboBox.setPageLength( 0 );
        functionsComboBox.setWidth( 150, Unit.PIXELS );
        functionsComboBox.addStyleName( ValoTheme.COMBOBOX_TINY );

        // sql function's parameters settings
        final Button functionsButton = new Button();
        functionsButton.setDescription( BUTTON_PARAMETERS );
        functionsButton.setEnabled( false );
        functionsButton.setIcon( FontAwesome.DOT_CIRCLE_O );
        functionsButton.addStyleName( ValoTheme.BUTTON_ICON_ONLY );
        functionsButton.addStyleName( ValoTheme.BUTTON_TINY );

        // sql functions css layout
        CssLayout functionsCssLayout = new CssLayout( functionsComboBox, functionsButton );
        functionsCssLayout.addStyleName( ValoTheme.LAYOUT_COMPONENT_GROUP );

        // sql function's types and sql functions css layout
        HorizontalLayout functionsLayout = new HorizontalLayout( sqlComboBox, functionsCssLayout );
        functionsLayout.setExpandRatio( functionsCssLayout, 1.7f );
        functionsLayout.setMargin( true );
        functionsLayout.setSpacing( true );
        functionsLayout.setWidth( 100, Unit.PERCENTAGE );
        panel.setContent( functionsLayout );

        // available/selected fields
        final TwinColSelect twinColSelect = new TwinColSelect();
        twinColSelect.setMultiSelect( true );
        twinColSelect.setLeftColumnCaption( FIELDS_AVALIABLE );
        twinColSelect.setRightColumnCaption( FIELDS_SELECTED );
        twinColSelect.setRows( 10 );
        twinColSelect.setWidth( 100, Unit.PERCENTAGE );
        addContextComponent( twinColSelect );

        // fill sqlComboBox, functionsComboBox and twinColSelect
        List<String> sqlValues = new ArrayList<>();
        List<String> functionsValues = new ArrayList<>();
        if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_TOTAL ) )
        {
            sqlValues.add( MessageFormat.format( SQL_FUNCTIONS,
                                                 StringUtil.capitalize( BaseTable.DESCRIPTION_TOTAL ) ) );
            functionsValues.addAll( Arrays.asList( sqlTabs.getTotalFunctions()
                                                          .getFunctions() ) );
        }
        if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_SORTING ) )
        {
            sqlValues.add( MessageFormat.format( SQL_FUNCTIONS,
                                                 StringUtil.capitalize( BaseTable.DESCRIPTION_SORTING ) ) );
            functionsValues.clear();
            functionsValues.addAll( Arrays.asList( sqlTabs.getSortingFunctions()
                                                          .getFunctions() ) );
        }
        if ( sqlTabs.isHeaderRowVisible( BaseTable.DESCRIPTION_COMMON ) )
        {
            sqlValues.add( MessageFormat.format( SQL_FUNCTIONS,
                                                 StringUtil.capitalize( BaseTable.DESCRIPTION_COMMON ) ) );
            functionsValues.clear();
            functionsValues.addAll( Arrays.asList( sqlTabs.getCommonFunctions()
                                                          .getFunctions() ) );
        }
        Collections.sort( sqlValues );
        sqlComboBox.addItems( sqlValues );
        sqlComboBox.setValue( sqlValues.get( 0 ) );
        functionsComboBox.addItems( functionsValues );
        functionsComboBox.setValue( functionsValues.get( 0 ) );
        twinColSelect.addItems( getFields() );

        // common, sorting or total functions
        sqlComboBox.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = 8783061967743287586L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                String sqlFunction = StringUtil.trim( sqlComboBox.getValue()
                                                                 .toString()
                                                                 .split( StringUtil.STRING_SPACE_SPLITTER )[0] )
                                               .toUpperCase();
                functionsComboBox.removeAllItems();
                switch ( sqlFunction )
                {
                    case BaseTable.DESCRIPTION_COMMON:
                        functionsComboBox.addItems( (Object[]) sqlTabs.getCommonFunctions()
                                                                      .getFunctions() );
                        break;
                    case BaseTable.DESCRIPTION_SORTING:
                        functionsComboBox.addItems( (Object[]) sqlTabs.getSortingFunctions()
                                                                      .getFunctions() );
                        break;
                    case BaseTable.DESCRIPTION_TOTAL:
                        functionsComboBox.addItems( (Object[]) sqlTabs.getTotalFunctions()
                                                                      .getFunctions() );
                        break;
                    default:
                        break;
                }
                functionsButton.setEnabled( false );
            }
        } );

        // sql functions
        functionsComboBox.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = -9015389061541799353L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                boolean isEnabled = false;
                String sqlFunction = StringUtil.trim( sqlComboBox.getValue()
                                                                 .toString()
                                                                 .split( StringUtil.STRING_SPACE_SPLITTER )[0] )
                                               .toUpperCase();
                Object function = functionsComboBox.getValue();
                if ( SharedUtil.isNotNull( function ) )
                {
                    switch ( sqlFunction )
                    {
                        case BaseTable.DESCRIPTION_COMMON:
                            isEnabled = sqlTabs.getCommonFunctions()
                                               .hasParameters( function.toString() );
                            break;
                        case BaseTable.DESCRIPTION_SORTING:
                            isEnabled = sqlTabs.getSortingFunctions()
                                               .hasParameters( function.toString() );
                            break;
                        case BaseTable.DESCRIPTION_TOTAL:
                            isEnabled = sqlTabs.getTotalFunctions()
                                               .hasParameters( function.toString() );
                            break;
                        default:
                            break;
                    }
                }

                // enable/disable button for change parameters value
                if ( SharedUtil.isNotNull( function ) )
                {
                    functionsButton.setEnabled( isEnabled );
                }
            }
        } );

        // sql function's parameters settings
        functionsButton.addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 2926250399750599682L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                String function = functionsComboBox.getValue()
                                                   .toString();
                if ( SharedUtil.isNotNullAndNotEmpty( function ) )
                {
                    UI.getCurrent()
                      .addWindow( new SqlParametersWindow( sqlTabs, function, null )
                      {
                          private static final long serialVersionUID = 7735994138020106331L;

                          // save function's new parameters
                          @Override
                          public void saveNewParameters( CommonFunctions commonFunctions, String function )
                          {
                              functionsComboBox.removeAllItems();
                              functionsComboBox.addItems( (Object[]) commonFunctions.getFunctions() );
                              functionsComboBox.setValue( commonFunctions.getFunction( function ) );
                          }
                      } );
                }
            }
        } );

        // available/selected fields
        twinColSelect.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = 8783061967743287586L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                if ( SharedUtil.isNotNull( event.getProperty() ) && event.getProperty()
                                                                         .getValue() instanceof Collection )
                {
                    getOkButton().setEnabled( !( (Collection<?>) event.getProperty()
                                                                      .getValue() ).isEmpty() );
                }
            }
        } );

        // "OK" button click
        getOkButton().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 3491165668669637327L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                Collection<?> fields = (Collection<?>) twinColSelect.getValue();
                if ( !fields.isEmpty() )
                {
                    String function = functionsComboBox.getValue()
                                                       .toString();
                    String rowId = StringUtil.trim( sqlComboBox.getValue()
                                                               .toString()
                                                               .split( StringUtil.STRING_SPACE_SPLITTER )[0] )
                                             .toUpperCase();
                    Collection<?> columnIds = sqlTabs.getHeaderFieldsTable()
                                                     .getItem( BaseTable.DESCRIPTION_NAME )
                                                     .getItemPropertyIds();

                    // get function description
                    String description = null;
                    switch ( rowId )
                    {
                        case BaseTable.DESCRIPTION_COMMON:
                            description = sqlTabs.getCommonFunctions()
                                                 .getDescription( function );
                            break;
                        case BaseTable.DESCRIPTION_SORTING:
                            description = sqlTabs.getSortingFunctions()
                                                 .getDescription( function );
                            break;
                        case BaseTable.DESCRIPTION_TOTAL:
                            description = sqlTabs.getTotalFunctions()
                                                 .getDescription( function );
                            break;
                        default:
                            break;
                    }
                    if ( SharedUtil.isNullOrEmpty( function ) )
                    {
                        description = StringUtil.EMPTY_VALUE;
                    }

                    // get all functions
                    int functionsLength = functionsComboBox.getItemIds()
                                                           .size();
                    String[] functions = functionsComboBox.getItemIds()
                                                          .toArray( new String[functionsLength] );

                    // fill header fields with new values
                    for ( Object field : fields )
                    {
                        for ( Object columnId : columnIds )
                        {
                            if ( sqlTabs.getHeaderFieldsTable()
                                        .getCellValue( columnId.toString(), BaseTable.DESCRIPTION_NAME )
                                        .equals( field ) )
                            {
                                sqlTabs.getHeaderFieldsTable()
                                       .resetCell( columnId.toString(), rowId );
                                sqlTabs.getHeaderFieldsTable()
                                       .setCellValues( columnId.toString(), rowId, functions );
                                sqlTabs.getHeaderFieldsTable()
                                       .setCellValue( columnId.toString(), rowId, function );
                                sqlTabs.getHeaderFieldsTable()
                                       .setCellDescription( columnId.toString(), rowId, description );
                                break;
                            }
                        }
                    }
                }

                // close dialog window
                close();
            }
        } );
    }

    /**
     * Get fields.
     * 
     * @return fields
     */
    private List<String> getFields()
    {
        List<String> fields = new ArrayList<>();
        for ( Object columnId : sqlTabs.getHeaderFieldsTable()
                                       .getVisibleColumns() )
        {
            if ( !sqlTabs.isFieldColumnEmpty( columnId.toString() ) && !BaseTable.COLUMN_LAST_FINAL.equals( columnId ) )
            {
                fields.add( sqlTabs.getHeaderFieldsTable()
                                   .getCellValue( columnId.toString(), BaseTable.DESCRIPTION_NAME ) );
            }
        }
        return fields;
    }

}
