/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.dussan.vaadin.dquery.base.ui.BaseTable;
import org.dussan.vaadin.dquery.enums.CellType;
import org.dussan.vaadin.dquery.sql.functions.CommonFunctions;
import org.dussan.vaadin.dquery.sql.functions.SortingFunctions;
import org.dussan.vaadin.dquery.sql.functions.TotalFunctions;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.dussan.vaadin.dtabs.DTabs;
import org.vaadin.aceeditor.AceEditor;
import org.vaadin.aceeditor.AceMode;
import org.vaadin.aceeditor.AceTheme;

import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.event.FieldEvents.BlurListener;
import com.vaadin.event.FieldEvents.FocusListener;
import com.vaadin.event.FieldEvents.TextChangeListener;
import com.vaadin.event.dd.DropHandler;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.HorizontalLayout;
import com.vaadin.ui.Table;
import com.vaadin.ui.Table.CellStyleGenerator;
import com.vaadin.ui.VerticalLayout;

public class SqlTabs
    extends DTabs
{

    private static final long serialVersionUID = 8785730435839578561L;

    private static final int MINIMAL_CRITERIA_ROWS = 3;

    private static final int MINIMAL_FIELD_COLUMNS = 30;

    private static final int MINIMAL_FIELD_COLUMN_WIDTH = 100;

    private static final String FIELDS_AREA = "fields-area";

    private static final String FIELDS_AREA_BACKGROUND = FIELDS_AREA + "-background";

    private static final String FIELDS_AREA_HEADER = FIELDS_AREA + "-header";

    private static final String FIELDS_AREA_CRITERIA = FIELDS_AREA + "-criteria";

    private static final String SQL_STACK = "SQL STACK";

    private static final String SQL_QUERY = "SQL QUERY";

    private int allFieldColumns = 0;

    private int fieldColumns = 0;

    private int criteriaRows = 0;

    private int criteriaMaxPageLength = 0;

    private int criteriaMinPageLength = 0;

    private int criteriaPageLength = 0;

    private int descriptionColumnWidth = 0;

    private int fieldColumnWidth = 0;

    private BaseTable headerDescriptionTable = null;

    private BaseTable headerFieldsTable = null;

    private BaseTable criteriaDescriptionTable = null;

    private BaseTable criteriaFieldsTable = null;

    private AceEditor sqlQueryEditor = null;

    private CommonFunctions commonFunctions = null;

    private SortingFunctions sortingFunctions = null;

    private TotalFunctions totalFunctions = null;

    private List<String> headerRows = null;

    private Map<String, CellType> headerRowTypes = null;

    private List<Integer> invisibleHeaderRows = null;

    /**
     * Creates a new instance.
     */
    public SqlTabs()
    {
        addStyleName( FIELDS_AREA );
        setTabBarBottom( true );
        setCompactedTabBar( true );
        setFramedTabs( true );

        // table dimensions
        setDescriptionColumnWidth( 100 );
        setAllFieldColumns( 100 );
        setFieldColumns( 30 );
        setFieldColumnWidth( 150 );
        setCriteriaRows( 20 );
        setCriteriaMinPageLength( 3 );
        setCriteriaMaxPageLength( 10 );
        setCriteriaPageLength( 5 );

        // initialization of fields and criteria part
        headerDescriptionTable = new BaseTable( true );
        headerFieldsTable = new BaseTable( false );
        criteriaDescriptionTable = new BaseTable( true );
        criteriaFieldsTable = new BaseTable( false );

        // sql functions
        commonFunctions = new CommonFunctions();
        sortingFunctions = new SortingFunctions();
        totalFunctions = new TotalFunctions();

        // header row names
        headerRows = new ArrayList<>( Arrays.asList( BaseTable.DESCRIPTION_NAME, BaseTable.DESCRIPTION_FIELD,
                                                     BaseTable.DESCRIPTION_TABLE, BaseTable.DESCRIPTION_COMMON,
                                                     BaseTable.DESCRIPTION_TOTAL, BaseTable.DESCRIPTION_SORTING,
                                                     BaseTable.DESCRIPTION_SHOW ) );

        // header row types
        headerRowTypes = new HashMap<>();
        headerRowTypes.put( BaseTable.DESCRIPTION_NAME, CellType.HEADER );
        headerRowTypes.put( BaseTable.DESCRIPTION_FIELD, CellType.TEXT_FIELD );
        headerRowTypes.put( BaseTable.DESCRIPTION_TABLE, CellType.TEXT_FIELD );
        headerRowTypes.put( BaseTable.DESCRIPTION_COMMON, CellType.COMBO_BOX_PARAMETER );
        headerRowTypes.put( BaseTable.DESCRIPTION_TOTAL, CellType.COMBO_BOX );
        headerRowTypes.put( BaseTable.DESCRIPTION_SORTING, CellType.COMBO_BOX );
        headerRowTypes.put( BaseTable.DESCRIPTION_SHOW, CellType.CHECK_BOX );

        // invisible header rows: common, sorting or total row
        invisibleHeaderRows = new ArrayList<>();
    }

    /**
     * Get sql query editor.
     * 
     * @return sql query editor
     */
    public AceEditor getSqlQueryEditor()
    {
        return sqlQueryEditor;
    }

    /**
     * Get number of max allowed visible field columns.
     * 
     * @return number of max allowed visible field columns
     */
    public int getAllFieldColumns()
    {
        return allFieldColumns;
    }

    /**
     * Set number of max allowed visible field columns.
     * 
     * @param allFieldColumns
     *            number of max allowed visible field columns
     */
    public void setAllFieldColumns( int allFieldColumns )
    {
        this.allFieldColumns = allFieldColumns;
    }

    /**
     * Get number of visible field columns.
     * 
     * @return number of visible filed columns
     */
    public int getFieldColumns()
    {
        return fieldColumns;
    }

    /**
     * Set number of visible field columns.
     * 
     * @param fieldColumns
     *            number of visible field columns
     */
    public void setFieldColumns( int fieldColumns )
    {
        this.fieldColumns = Math.max( fieldColumns, MINIMAL_FIELD_COLUMNS );
        this.fieldColumns = Math.min( this.fieldColumns, this.allFieldColumns );
    }

    /**
     * Get number of criteria rows.
     * 
     * @return number of criteria rows
     */
    public int getCriteriaRows()
    {
        return criteriaRows;
    }

    /**
     * Set number of criteria rows.
     * 
     * @param criteriaRows
     *            number of criteria rows
     */
    public void setCriteriaRows( int criteriaRows )
    {
        this.criteriaRows = Math.max( this.criteriaRows, criteriaRows );
        this.criteriaRows = Math.max( MINIMAL_CRITERIA_ROWS, this.criteriaRows );
    }

    /**
     * Get criteria maximum page length.
     * 
     * @return criteria maximum page length
     */
    public int getCriteriaMaxPageLength()
    {
        return criteriaMaxPageLength;
    }

    /**
     * Set criteria maximum page length.
     * 
     * @param criteriaMaxPageLength
     *            criteria maximum page length
     */
    public void setCriteriaMaxPageLength( int criteriaMaxPageLength )
    {
        this.criteriaMaxPageLength = Math.min( this.criteriaRows, criteriaMaxPageLength );
        this.criteriaMaxPageLength = Math.max( MINIMAL_CRITERIA_ROWS, this.criteriaMaxPageLength );
    }

    /**
     * Get criteria minimum page length.
     * 
     * @return criteria minimum page length
     */
    public int getCriteriaMinPageLength()
    {
        return criteriaMinPageLength;
    }

    /**
     * Set criteria minimum page length.
     * 
     * @param criteriaMinPageLength
     *            criteria minimum page length
     */
    public void setCriteriaMinPageLength( int criteriaMinPageLength )
    {
        this.criteriaMinPageLength = Math.min( this.criteriaRows, criteriaMinPageLength );
        this.criteriaMinPageLength = Math.max( MINIMAL_CRITERIA_ROWS, this.criteriaMinPageLength );
    }

    /**
     * Get criteria page length.
     * 
     * @return criteria page length
     */
    public int getCriteriaPageLength()
    {
        return criteriaPageLength;
    }

    /**
     * Set criteria page length.
     * 
     * @param criteriaPageLength
     *            criteria page length
     */
    public void setCriteriaPageLength( int criteriaPageLength )
    {
        this.criteriaPageLength = criteriaPageLength;
        setCriteriaMinPageLength( Math.min( this.criteriaPageLength, getCriteriaMinPageLength() ) );
        setCriteriaMaxPageLength( Math.max( this.criteriaPageLength, getCriteriaMaxPageLength() ) );

        if ( SharedUtil.isNotNull( criteriaDescriptionTable ) && SharedUtil.isNotNull( criteriaFieldsTable ) )
        {
            criteriaDescriptionTable.setPageLength( criteriaPageLength );
            criteriaFieldsTable.setPageLength( criteriaPageLength );
        }
    }

    /**
     * Increase/decrease visible criteria rows.
     * 
     * @param visibled
     *            if true more criteria rows is visible, otherwise less
     */
    public void setMoreCriteriaRowsVisibled( boolean visibled )
    {
        int visibleRows = criteriaDescriptionTable.getPageLength();
        if ( visibled )
        {
            visibleRows = Math.min( getCriteriaMaxPageLength(), visibleRows + 1 );
        }
        else
        {
            visibleRows = Math.max( getCriteriaMinPageLength(), visibleRows - 1 );
        }
        criteriaDescriptionTable.setPageLength( visibleRows );
        criteriaFieldsTable.setPageLength( visibleRows );
        setCriteriaPageLength( visibleRows );
    }

    /**
     * Get width of description column in pixels.
     * 
     * @return width of description column in pixels.
     */
    public int getDescriptionColumnWidth()
    {
        return descriptionColumnWidth;
    }

    /**
     * Set width of description column in pixels.
     * 
     * @param descriptionColumnWidth
     *            width of description column in pixels
     */
    public void setDescriptionColumnWidth( int descriptionColumnWidth )
    {
        this.descriptionColumnWidth = descriptionColumnWidth;
    }

    /**
     * Get width of field column in pixels.
     * 
     * @return width of field column in pixels.
     */
    public int getFieldColumnWidth()
    {
        return fieldColumnWidth;
    }

    /**
     * Set width of field column in pixels.
     * 
     * @param fieldColumnWidth
     *            width of field column in pixels
     */
    public void setFieldColumnWidth( int fieldColumnWidth )
    {
        this.fieldColumnWidth = Math.max( fieldColumnWidth, MINIMAL_FIELD_COLUMN_WIDTH );
    }

    /**
     * Enable or disable all components in field column for interaction.
     * 
     * @param columnId
     *            column id
     * @param enabled
     *            if true component is enable for interaction; false otherwise
     */
    public void setFieldColumnEnabled( String columnId, boolean enabled )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
        {
            headerFieldsTable.setColumnEnabled( columnId, enabled );
            criteriaFieldsTable.setColumnEnabled( columnId, enabled );
        }
    }

    /**
     * Check if field column has no values in rows: name, field or table.
     * 
     * @param columnId
     *            column id
     * @return true if field column has no values; false otherwise
     */
    public boolean isFieldColumnEmpty( String columnId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
        {
            return headerFieldsTable.isColumnEmpty( columnId );
        }
        return true;
    }

    /**
     * Check if all fields columns have no values.
     * 
     * @return true if all fields columns have no values; false otherwise
     */
    public boolean isAllFieldColumnsEmpty()
    {
        List<Object> columnsId = new ArrayList<>( headerFieldsTable.getContainerPropertyIds() );
        columnsId.remove( BaseTable.COLUMN_LAST_FINAL );
        for ( Object columnId : columnsId )
        {
            if ( !isFieldColumnEmpty( columnId.toString() ) )
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Get column id of first empty field column.
     * 
     * @return column id of first empty field column
     */
    public String getFirstEmptyFieldColumn()
    {
        for ( Object columnId : headerFieldsTable.getVisibleColumns() )
        {
            if ( isFieldColumnEmpty( columnId.toString() ) )
            {
                return columnId.toString();
            }
        }
        return null;
    }

    /**
     * Get next field column id.
     * 
     * @param columnId
     *            column id
     * @return next column id
     */
    public String getNextFieldColumnId( String columnId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
        {
            List<Object> columnsId = new ArrayList<>( Arrays.asList( headerFieldsTable.getVisibleColumns() ) );
            int columnIndex = columnsId.indexOf( columnId ) + 1;
            if ( columnIndex <= allFieldColumns )
            {
                return columnsId.get( columnIndex )
                                .toString();
            }
        }
        return null;
    }

    /**
     * Add missing visible field columns to end of the fields table.
     */
    public void addMissingFieldColumns()
    {
        String nonEmptyColumnId = null;
        List<Object> visibleColumns = new ArrayList<>( Arrays.asList( headerFieldsTable.getVisibleColumns() ) );
        Collections.reverse( visibleColumns );
        visibleColumns.remove( 0 );

        // get the last non-empty column (from left to right)
        for ( Object columnId : visibleColumns )
        {
            String id = columnId.toString();
            if ( !headerFieldsTable.isColumnCollapsed( id ) )
            {
                nonEmptyColumnId = id;
                if ( !isFieldColumnEmpty( id ) )
                {
                    break;
                }
            }
        }

        // set at least 10 columns visible after the last non-empty column
        if ( SharedUtil.isNotNullAndNotEmpty( nonEmptyColumnId ) )
        {
            Collections.reverse( visibleColumns );
            int index = visibleColumns.indexOf( nonEmptyColumnId );
            for ( int column = index; column < ( index + 10 ) && column < allFieldColumns; column++ )
            {
                String id = visibleColumns.get( column )
                                          .toString();
                if ( headerFieldsTable.isColumnCollapsed( id ) )
                {
                    headerFieldsTable.setColumnCollapsed( id, false );
                    criteriaFieldsTable.setColumnCollapsed( id, false );

                    // set visible field columns
                    setFieldColumns( getFieldColumns() + 1 );
                    headerFieldsTable.setColumns( getFieldColumns() );
                    criteriaFieldsTable.setColumns( getFieldColumns() );
                }
            }
        }
    }

    /**
     * Add field column with values from database and table definitions.
     * 
     * @param tableName
     *            table name
     * @param fieldName
     *            field name
     * @param name
     *            column name
     * @param show
     *            if true new column will be included in generated sql clause
     * @return the column id of newly added column
     */
    public String addFieldColumn( String tableName, String fieldName, String name, boolean show )
    {
        String columnId = headerFieldsTable.addColumn( tableName, fieldName, name, show );
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
        {
            criteriaFieldsTable.setColumnEnabled( columnId, true );

            // add values to common combo box
            headerFieldsTable.setCellValues( columnId, BaseTable.DESCRIPTION_COMMON, commonFunctions.getFunctions() );

            // add values to total combo box
            headerFieldsTable.setCellValues( columnId, BaseTable.DESCRIPTION_TOTAL, totalFunctions.getFunctions() );

            // add values to sorting combo box
            headerFieldsTable.setCellValues( columnId, BaseTable.DESCRIPTION_SORTING, sortingFunctions.getFunctions() );
        }
        return columnId;
    }

    /**
     * Add field column before other field column.
     * 
     * @param columnId
     *            column which will be moved to new location
     * @param insertBeforeColumnId
     *            column on which places will be inserted new column
     */
    public void addFieldColumnBefore( String columnId, String insertBeforeColumnId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) && SharedUtil.isNotNullAndNotEmpty( insertBeforeColumnId ) )
        {
            headerFieldsTable.moveColumn( columnId, insertBeforeColumnId );
            criteriaFieldsTable.moveColumn( columnId, insertBeforeColumnId );
        }
    }

    /**
     * Reverse field column with other field column.
     * 
     * @param columnId
     *            column which will be moved to new location
     * @param reversedColumnId
     *            column on which places will be inserted new column
     */
    public void reverseFieldColumns( String columnId, String reversedColumnId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) && SharedUtil.isNotNullAndNotEmpty( reversedColumnId ) )
        {
            headerFieldsTable.reverseColumns( columnId, reversedColumnId );
            criteriaFieldsTable.reverseColumns( columnId, reversedColumnId );
        }
    }

    /**
     * Remove field column.
     * 
     * @param columnId
     *            column id
     */
    public void removeFieldColumn( String columnId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( columnId ) )
        {
            headerFieldsTable.removeColumn( columnId );
            criteriaFieldsTable.removeColumn( columnId );
        }
    }

    /**
     * Remove field columns for specified table.
     * 
     * @param databaseName
     *            database name
     * @param tableName
     *            table name
     */
    public void removeFieldColumns( String tableName )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( tableName ) )
        {
            for ( Iterator<?> iterator = headerFieldsTable.getItem( BaseTable.DESCRIPTION_NAME )
                                                          .getItemPropertyIds()
                                                          .iterator(); iterator.hasNext(); )
            {
                String columnId = iterator.next()
                                          .toString();
                if ( tableName.equals( headerFieldsTable.getCellValue( columnId, BaseTable.DESCRIPTION_TABLE ) )
                    && !isFieldColumnEmpty( columnId ) )
                {
                    removeFieldColumn( columnId );
                }
            }
        }
    }

    /**
     * Get header description table.
     * 
     * @return header description table
     */
    public BaseTable getHeaderDescriptionTable()
    {
        return headerDescriptionTable;
    }

    /**
     * Get header fields table.
     * 
     * @return header fields table
     */
    public BaseTable getHeaderFieldsTable()
    {
        return headerFieldsTable;
    }

    /**
     * Get criteria description table.
     * 
     * @return criteria description table
     */
    public BaseTable getCriteriaDescritpionTable()
    {
        return criteriaDescriptionTable;
    }

    /**
     * Get criteria fields table.
     * 
     * @return criteria fields table
     */
    public BaseTable getCriteriaFieldsTable()
    {
        return criteriaFieldsTable;
    }

    /**
     * Get common functions.
     * 
     * @return common functions
     */
    public CommonFunctions getCommonFunctions()
    {
        return commonFunctions;
    }

    /**
     * Get sorting functions.
     * 
     * @return sorting functions
     */
    public SortingFunctions getSortingFunctions()
    {
        return sortingFunctions;
    }

    /**
     * Get total functions.
     * 
     * @return total functions
     */
    public TotalFunctions getTotalFunctions()
    {
        return totalFunctions;
    }

    /**
     * Get invisible header rows.
     * 
     * @return invisible header rows
     */
    public List<Integer> getInvisibleHeaderRows()
    {
        return invisibleHeaderRows;
    }

    /**
     * Get if is header row visible.
     * 
     * @param rowId
     *            row id
     * @return return true if it is visible, otherwise false
     */
    public boolean isHeaderRowVisible( String rowId )
    {
        int invisibleHeaderRow = -1;
        switch ( StringUtil.trim( rowId ) )
        {
            case BaseTable.DESCRIPTION_COMMON:
                invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_COMMON_ID );
                break;
            case BaseTable.DESCRIPTION_SORTING:
                invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_SORTING_ID );
                break;
            case BaseTable.DESCRIPTION_TOTAL:
                invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_TOTAL_ID );
                break;
            default:
                break;
        }
        return invisibleHeaderRow < 0;
    }

    /**
     * Set header row visible.
     * 
     * @param rowId
     *            header row
     * @param visible
     *            if true header row is visible, otherwise is hidden
     */
    public void setHeaderRowVisible( String rowId, boolean visible )
    {
        int invisibleHeaderRow = -1;
        if ( visible )
        {
            switch ( StringUtil.trim( rowId ) )
            {
                case BaseTable.DESCRIPTION_COMMON:
                    invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_COMMON_ID );
                    break;
                case BaseTable.DESCRIPTION_SORTING:
                    invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_SORTING_ID );
                    break;
                case BaseTable.DESCRIPTION_TOTAL:
                    invisibleHeaderRow = invisibleHeaderRows.indexOf( BaseTable.DESCRIPTION_TOTAL_ID );
                    break;
                default:
                    break;
            }

            // set header row visible
            if ( 0 <= invisibleHeaderRow )
            {
                invisibleHeaderRows.remove( invisibleHeaderRow );
            }
        }
        else
        {
            switch ( StringUtil.trim( rowId ) )
            {
                case BaseTable.DESCRIPTION_COMMON:
                    invisibleHeaderRow = BaseTable.DESCRIPTION_COMMON_ID;
                    break;
                case BaseTable.DESCRIPTION_SORTING:
                    invisibleHeaderRow = BaseTable.DESCRIPTION_SORTING_ID;
                    break;
                case BaseTable.DESCRIPTION_TOTAL:
                    invisibleHeaderRow = BaseTable.DESCRIPTION_TOTAL_ID;
                    break;
                default:
                    break;
            }

            // set header row invisible
            if ( 0 <= invisibleHeaderRow && !invisibleHeaderRows.contains( invisibleHeaderRow ) )
            {
                invisibleHeaderRows.add( invisibleHeaderRow );
            }
        }
    }

    /**
     * Add blur listener.
     * 
     * @param blurListener
     *            blur listener
     */
    public void addBlurListener( BlurListener blurListener )
    {
        headerFieldsTable.addBlurListener( blurListener );
    }

    /**
     * Add click listener.
     * 
     * @param clickListener
     *            click listener
     */
    public void addClickListener( ClickListener clickListener )
    {
        headerFieldsTable.addClickListener( clickListener );
        criteriaDescriptionTable.addClickListener( clickListener );
    }

    /**
     * Add drop handler.
     * 
     * @param dropHandler
     *            drop handler
     */
    public void addDropHandler( DropHandler dropHandler )
    {
        headerFieldsTable.addDropHandler( dropHandler );
    }

    /**
     * Add focus listener.
     * 
     * @param focusListener
     *            focus listener
     */
    public void addFocusListener( FocusListener focusListener )
    {
        headerFieldsTable.addFocusListener( focusListener );
    }

    /**
     * Add text change listener.
     * 
     * @param textChangeListener
     *            text change listener
     */
    public void addTextChangeListener( TextChangeListener textChangeListener )
    {
        headerFieldsTable.addTextChangeListener( textChangeListener );
    }

    /**
     * Add value change listener.
     * 
     * @param valueChangeListener
     *            value change listener
     */
    public void addValueChangeListener( ValueChangeListener valueChangeListener )
    {
        headerFieldsTable.addValueChangeListener( valueChangeListener );
        criteriaFieldsTable.addValueChangeListener( valueChangeListener );
    }

    /**
     * Prepare: create header section and criteria section. Both sections includes one description column and one or
     * more field columns.
     */
    public void prepare()
    {
        // header description table
        headerDescriptionTable.addStyleName( BaseTable.TABLE_NO_SCROLL );
        headerDescriptionTable.setColumnWidth( descriptionColumnWidth );
        headerDescriptionTable.setWidth( descriptionColumnWidth, Unit.PIXELS );
        headerDescriptionTable.setPageLength( headerRows.size() );
        headerDescriptionTable.setCellStyleGenerator( new CellStyleGenerator()
        {
            private static final long serialVersionUID = -4818757050175929609L;

            @Override
            public String getStyle( Table source, Object itemId, Object propertyId )
            {
                if ( SharedUtil.isNull( propertyId )
                    && invisibleHeaderRows.contains( headerRows.indexOf( itemId.toString() ) ) )
                {
                    return BaseTable.TABLE_ROW_FIELD_HIDDEN;
                }
                return BaseTable.DESCRIPTION_FIELD_STYLE;
            }
        } );

        // header field table
        headerFieldsTable.addStyleName( BaseTable.TABLE_NO_SCROLL );
        headerFieldsTable.setColumns( fieldColumns );
        headerFieldsTable.setColumnWidth( fieldColumnWidth );
        headerFieldsTable.setPageLength( headerFieldsTable.size() );
        headerFieldsTable.setSizeFull();
        headerFieldsTable.setCellStyleGenerator( new CellStyleGenerator()
        {
            private static final long serialVersionUID = -4818757050175929609L;

            @Override
            public String getStyle( Table source, Object itemId, Object propertyId )
            {
                if ( SharedUtil.isNull( propertyId ) && SharedUtil.isNotNullAndNotEmpty( itemId ) )
                {
                    String rowStyle = null;
                    if ( invisibleHeaderRows.contains( headerRows.indexOf( itemId.toString() ) ) )
                    {
                        rowStyle = BaseTable.TABLE_ROW_FIELD_HIDDEN;
                    }
                    else
                    {
                        // all visible rows
                        List<String> visibleRows =
                            new ArrayList<>( headerRows.subList( 0, headerRows.indexOf( itemId.toString() ) + 1 ) );

                        // remove invisible rows
                        for ( int row : invisibleHeaderRows )
                        {
                            visibleRows.remove( headerRows.get( row ) );
                        }

                        // set row style based on visible rows size
                        switch ( visibleRows.size() % 2 )
                        {
                            case 0:
                                rowStyle = BaseTable.TABLE_ROW_FIELD_ODD;
                                break;
                            case 1:
                                rowStyle = BaseTable.TABLE_ROW_FIELD;
                                break;
                            default:
                                break;
                        }
                    }
                    return rowStyle;
                }
                return null;
            }
        } );

        // header description column and header's first empty field column with default components
        headerDescriptionTable.addEmptyColumn( BaseTable.COLUMN_DESCRIPTION );
        headerFieldsTable.addEmptyColumn( BaseTable.COLUMN_LAST_FINAL );
        headerFieldsTable.addEmptyColumn( Integer.toString( 0 ) );
        for ( int row = 0; row < headerRows.size(); row++ )
        {
            headerDescriptionTable.addEmptyRow( headerRows.get( row ) );
            headerDescriptionTable.setCellComponent( BaseTable.COLUMN_DESCRIPTION, headerRows.get( row ),
                                                     CellType.LABEL );
            headerDescriptionTable.setCellValue( BaseTable.COLUMN_DESCRIPTION, headerRows.get( row ),
                                                 headerRows.get( row ) );
            headerFieldsTable.addEmptyRow( headerRows.get( row ) );
            headerFieldsTable.setCellComponent( Integer.toString( 0 ), headerRows.get( row ),
                                                headerRowTypes.get( headerRows.get( row ) ) );
        }

        // criteria description table
        criteriaDescriptionTable.addStyleName( BaseTable.TABLE_NO_SCROLL );
        criteriaDescriptionTable.setColumnWidth( descriptionColumnWidth );
        criteriaDescriptionTable.setWidth( descriptionColumnWidth, Unit.PIXELS );
        criteriaDescriptionTable.setPageLength( criteriaPageLength );

        // criteria field table
        criteriaFieldsTable.setColumns( fieldColumns );
        criteriaFieldsTable.setColumnWidth( fieldColumnWidth );
        criteriaFieldsTable.setPageLength( criteriaPageLength );
        criteriaFieldsTable.setSizeFull();

        // criteria description column and criteria first empty field column with default components
        criteriaDescriptionTable.addEmptyColumn( BaseTable.COLUMN_DESCRIPTION );
        criteriaDescriptionTable.setColumnWidth( BaseTable.COLUMN_DESCRIPTION, descriptionColumnWidth );
        criteriaFieldsTable.addEmptyColumn( Integer.toString( 0 ) );
        for ( int row = 0; row < criteriaRows; row++ )
        {
            String rowId = Integer.toString( row );
            criteriaDescriptionTable.addEmptyRow( rowId );
            if ( row == 0 )
            {
                criteriaDescriptionTable.setCellComponent( BaseTable.COLUMN_DESCRIPTION, rowId,
                                                           CellType.LABEL_WITH_BUTTONS );
                criteriaDescriptionTable.setCellValue( BaseTable.COLUMN_DESCRIPTION, rowId,
                                                       BaseTable.DESCRIPTION_CRITERIA );
            }
            else
            {
                criteriaDescriptionTable.setCellComponent( BaseTable.COLUMN_DESCRIPTION, rowId,
                                                           CellType.LABEL_RIGHT_ALIGNED );
                criteriaDescriptionTable.setCellValue( BaseTable.COLUMN_DESCRIPTION, rowId,
                                                       BaseTable.DESCRIPTION_CRITERIA_OR );
            }
            criteriaFieldsTable.addEmptyRow( rowId );
            criteriaFieldsTable.setCellComponent( Integer.toString( 0 ), rowId, CellType.TEXT_FIELD );
        }
        criteriaDescriptionTable.addEmptyRow( Integer.toString( criteriaRows ) );

        // add field columns to header and criteria field section
        for ( int column = 0; column < allFieldColumns; column++ )
        {
            String columnId = Integer.toString( column );
            headerFieldsTable.addColumn( columnId );
            criteriaFieldsTable.addColumn( columnId );
        }

        // set field columns collapsed - invisible
        for ( int column = fieldColumns; column < allFieldColumns; column++ )
        {
            String columnId = Integer.toString( column );
            headerFieldsTable.setColumnCollapsed( columnId, true );
            criteriaFieldsTable.setColumnCollapsed( columnId, true );
        }

        // header layout
        HorizontalLayout headerLayout = new HorizontalLayout( headerDescriptionTable, headerFieldsTable );
        headerLayout.addStyleName( FIELDS_AREA_HEADER );
        headerLayout.setSizeFull();
        headerLayout.setMargin( false );
        headerLayout.setSpacing( false );
        headerLayout.setExpandRatio( headerFieldsTable, 1 );

        // criteria layout
        HorizontalLayout criteriaLayout = new HorizontalLayout( criteriaDescriptionTable, criteriaFieldsTable );
        criteriaLayout.addStyleName( FIELDS_AREA_CRITERIA );
        criteriaLayout.setSizeFull();
        criteriaLayout.setMargin( false );
        criteriaLayout.setSpacing( false );
        criteriaLayout.setExpandRatio( criteriaFieldsTable, 1 );

        // source and criteria tab
        VerticalLayout sqlStackLayout = new VerticalLayout( headerLayout, criteriaLayout );
        sqlStackLayout.setCaption( null );
        sqlStackLayout.setSizeFull();
        sqlStackLayout.setMargin( false );
        sqlStackLayout.addStyleName( FIELDS_AREA_BACKGROUND );
        addTab( sqlStackLayout, SQL_STACK );

        // sql editor tab
        sqlQueryEditor = new AceEditor();
        sqlQueryEditor.setSizeFull();
        sqlQueryEditor.setHeight( 200, Unit.PIXELS );
        sqlQueryEditor.setHighlightActiveLine( false );
        sqlQueryEditor.setMode( AceMode.pgsql );
        sqlQueryEditor.setShowPrintMargin( false );
        sqlQueryEditor.setTheme( AceTheme.github );
        sqlQueryEditor.setReadOnly( true );
        sqlQueryEditor.setWordWrap( true );
        addTab( sqlQueryEditor, SQL_QUERY );
    }

}
