/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.component;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.dussan.vaadin.dquery.base.ui.DroppedTable;
import org.dussan.vaadin.dquery.client.state.QueryState;
import org.dussan.vaadin.dquery.enums.TableType;
import org.dussan.vaadin.dquery.helper.ManifestHelper;
import org.dussan.vaadin.dquery.json.JsonBuilder;
import org.dussan.vaadin.dquery.sql.builder.SqlBuilder;
import org.dussan.vaadin.dquery.ui.DataSourcesMenu;
import org.dussan.vaadin.dquery.ui.DroppedTableContainer;
import org.dussan.vaadin.dquery.ui.SqlMenu;
import org.dussan.vaadin.dquery.ui.SqlTabs;
import org.dussan.vaadin.dquery.utils.JdbcUtil;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.shared.ui.MarginInfo;
import com.vaadin.ui.Component;
import com.vaadin.ui.VerticalLayout;

public class DQueryComponent
    extends VerticalLayout
{

    private static final long serialVersionUID = 8504642832573585503L;

    private static final String DQUERY = "dquery";

    private static final String GIT_VERSION = "Git-Version";

    private static final String VERSION = "Implementation-Version";

    protected static final String CONTENT = "content";

    protected static final String DATA_SOURCE_MENU = "data-source-menu";

    protected static final String DATA_SOURCES_ALREADY_HAVE_THIS_IDENTIFIER =
        "Data sources already have this identifier: {0}.";

    protected static final String FIELDS = "fields";

    protected static final String IDENTIFIER_IS_EMPTY = "Identifier is empty.";

    protected static final String SQL_MENU = "sql-menu";

    protected static final String TABLES = "tables";

    private transient Map<String, JdbcUtil> dataSources = null;

    private transient Map<String, Boolean> showDroppedTableData = null;

    private transient Map<String, Boolean> showQueryData = null;

    private DataSourcesMenu dataSourcesMenu = null;

    private DroppedTableContainer droppedTableContainer = null;

    private JsonBuilder jsonBuilder = null;

    private SqlBuilder sqlBuilder = null;

    private SqlMenu sqlMenu = null;

    private SqlTabs sqlTabs = null;

    /**
     * Creates a new instance.
     */
    public DQueryComponent()
    {
        setMargin( false );
        setSpacing( false );
        setSizeFull();
        setHeightUndefined();
        addStyleName( DQUERY );

        // data sources
        dataSources = new LinkedHashMap<>();

        // show table data for data source
        showDroppedTableData = new HashMap<>();

        // show query data
        showQueryData = new HashMap<>();

        // data sources menu
        dataSourcesMenu = new DataSourcesMenu();
        dataSourcesMenu.setCenteredTabs( true );
        dataSourcesMenu.setFloatingMenuLockEnabled( true );
        dataSourcesMenu.setTabNotSorted( DataSourcesMenu.DATA_SOURCES );
        dataSourcesMenu.setTabsSortEnabled( true );

        // container for dropped tables
        droppedTableContainer = new DroppedTableContainer();

        // json builder
        jsonBuilder = new JsonBuilder();

        // sql menu
        sqlMenu = new SqlMenu();

        // sql tabs
        sqlTabs = new SqlTabs();
        sqlTabs.setCenteredTabs( true );
    }

    /**
     * Setting margins is disabled.
     */
    @Override
    public void setMargin( boolean enabled )
    {
        super.setMargin( false );
    }

    /**
     * Setting margins is disabled.
     */
    @Override
    public void setMargin( MarginInfo marginInfo )
    {
        super.setMargin( new MarginInfo( false ) );
    }

    /**
     * Get dQuery version.
     * 
     * @return dQuery version
     */
    public static String getVersion()
    {
        if ( !SharedUtil.isNull( ManifestHelper.getManifest() ) )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( VERSION );
        }
        return null;
    }

    /**
     * Get dQuery git version.
     * 
     * @return dQuery git version
     */
    public static String getGitVersion()
    {
        if ( !SharedUtil.isNull( ManifestHelper.getManifest() ) )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( GIT_VERSION );
        }
        return null;
    }

    /**
     * Get the shared state.
     */
    @Override
    protected QueryState getState()
    {
        return (QueryState) super.getState();
    }

    /**
     * Get the shared state.
     */
    @Override
    protected QueryState getState( boolean markAsDirty )
    {
        return (QueryState) super.getState( markAsDirty );
    }

    /**
     * Get data sources.
     * 
     * @return data sources
     */
    protected Map<String, JdbcUtil> getDataSources()
    {
        return dataSources;
    }

    /**
     * Get data sources menu.
     * 
     * @return data sources menu
     */
    protected DataSourcesMenu getDataSourcesMenu()
    {
        return dataSourcesMenu;
    }

    /**
     * Get container for dropped tables.
     * 
     * @return container for dropped tables
     */
    protected DroppedTableContainer getDroppedTableContainer()
    {
        return droppedTableContainer;
    }

    /**
     * Get Json builder.
     * 
     * @return Json builder
     */
    protected JsonBuilder getJsonBuilder()
    {
        return jsonBuilder;
    }

    /**
     * Get Sql builder.
     * 
     * @return Sql builder
     */
    protected SqlBuilder getSqlBuilder()
    {
        return sqlBuilder;
    }

    /**
     * Reset Sql builder.
     */
    protected void resetSqlBuilder()
    {
        sqlBuilder = new SqlBuilder();
    }

    /**
     * Get Sql menu.
     * 
     * @return Sql menu
     */
    protected SqlMenu getSqlMenu()
    {
        return sqlMenu;
    }

    /**
     * Get Sql tabs.
     * 
     * @return Sql tabs
     */
    protected SqlTabs getSqlTabs()
    {
        return sqlTabs;
    }

    /**
     * Get dropped table data showing enabled.
     * 
     * @return dropped table data showing enabled
     */
    protected Map<String, Boolean> getShowDroppedTableDataEnabled()
    {
        return showDroppedTableData;
    }

    /**
     * Check if can show dropped table data per data source.
     * 
     * @param dataSourceId
     *            data source id
     * @return true if can show dropped table data for specified data source, otherwise false
     */
    protected boolean isShowDroppedTableDataEnabled( String dataSourceId )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) && showDroppedTableData.containsKey( dataSourceId ) )
        {
            return showDroppedTableData.get( dataSourceId );
        }
        return false;
    }

    /**
     * Show dropped table data per data source.
     * 
     * @param dataSourceId
     *            data source id
     * @param showTableData
     *            if true show dropped table data, otherwise not
     */
    protected void setShowDroppedTableDataEnabled( String dataSourceId, boolean showTableData )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( dataSourceId ) )
        {
            this.showDroppedTableData.put( dataSourceId, showTableData );
            if ( dataSources.containsKey( dataSourceId ) && 0 < droppedTableContainer.getContainer()
                                                                                     .getComponentCount() )
            {
                for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                          .iterator(); iterator.hasNext(); )
                {
                    Component component = iterator.next();
                    if ( component instanceof DroppedTable )
                    {
                        DroppedTable droppedTable = (DroppedTable) component;
                        if ( TableType.TABLE.equals( droppedTable.getTableType() )
                            && dataSourceId.equals( droppedTable.getDataSourceId() ) )
                        {
                            droppedTable.setShowTableDataEnabled( showTableData );
                        }
                    }
                }
            }
        }
    }

    /**
     * Get query data showing enabled.
     * 
     * @return dropped table data showing enabled
     */
    protected Map<String, Boolean> getShowQueryDataEnabled()
    {
        return showQueryData;
    }

    /**
     * Check if can show query data.
     * 
     * @param group
     *            query group
     * @param name
     *            query name
     * @return true if can show query data, otherwise false
     */
    protected boolean isShowQueryDataEnabled( String group, String name )
    {
        String queryGroup = group;
        if ( SharedUtil.isNullOrEmpty( group ) )
        {
            queryGroup = SqlMenu.QUERY_GROUP;
        }

        String queryName = name;
        if ( SharedUtil.isNullOrEmpty( name ) )
        {
            queryName = SqlMenu.QUERY_NAME;
        }

        String queryGroupName = MessageFormat.format( SqlMenu.QUERY_GROUP_NAME, queryGroup, queryName );
        if ( SharedUtil.isNotNullAndNotEmpty( name ) && showQueryData.containsKey( queryGroupName ) )
        {
            return showQueryData.get( queryGroupName );
        }
        return false;
    }

    /**
     * Show query data per data source.
     * 
     * @param group
     *            query group
     * @param name
     *            query name
     * @param showQueryData
     *            if true show query data, otherwise not
     */
    protected void setShowQueryDataEnabled( String group, String name, boolean showQueryData )
    {
        String queryGroup = group;
        if ( SharedUtil.isNullOrEmpty( group ) )
        {
            queryGroup = SqlMenu.QUERY_GROUP;
        }

        String queryName = MessageFormat.format( SqlMenu.QUERY_GROUP_NAME, queryGroup, name );
        if ( SharedUtil.isNotNullAndNotEmpty( name ) )
        {
            this.showQueryData.put( queryName, showQueryData );
            if ( 0 < droppedTableContainer.getContainer()
                                          .getComponentCount() )
            {
                for ( Iterator<Component> iterator = droppedTableContainer.getContainer()
                                                                          .iterator(); iterator.hasNext(); )
                {
                    Component component = iterator.next();
                    if ( component instanceof DroppedTable )
                    {
                        DroppedTable droppedTable = (DroppedTable) component;
                        String tableName = droppedTable.getTableName()
                                                       .replace( StringUtil.DOT_VALUE, StringUtil.FORWARD_SLASH_VALUE );
                        if ( TableType.QUERY.equals( droppedTable.getTableType() ) && queryName.equals( tableName ) )
                        {
                            droppedTable.setShowQueryDefinitionEnabled( showQueryData );
                        }
                    }
                }
            }
        }
    }

}
