/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.utils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;

public class StringUtil
    implements Serializable
{

    private static final long serialVersionUID = 937947770423184349L;

    private static final String BRACKET_LEFT_VALUE = "[";

    private static final String BRACKET_RIGHT_VALUE = "]";

    public static final String COMMA_VALUE = ",";

    public static final String COMMA_WITH_SPACE = ", ";

    public static final String COUNTER_STRING = "_{0}";

    public static final String DOT_VALUE = ".";

    public static final String[] EMPTY_ARRAY = {};

    public static final String FORWARD_SLASH_VALUE = "/";

    public static final String EMPTY_VALUE = "";

    public static final String QUOTATION_MARK = "\"";

    public static final String SEMI_COLON_VALUE = ";";

    public static final String SINGLE_QOUTE_VALUE = "'";

    public static final String SPACE_VALUE = " ";

    public static final String STRING_COLON_SPLITTER = ":";

    public static final String STRING_COMMA_SPLITTER = ",";

    public static final String STRING_DASH_SPLITTER = "-";

    public static final String STRING_DOT_SPLITTER = "\\.";

    public static final String STRING_FORWARD_SLASH_SPLITTER = FORWARD_SLASH_VALUE;

    public static final String STRING_SPACE_SPLITTER = " ";

    public static final String UNDERSCORE_VALUE = "_";

    /**
     * Creates a new instance.
     */
    private StringUtil()
    {
        // Auto-generated constructor stub
    }

    /**
     * Check if object is null.
     * 
     * @param value
     *            object value
     * @return true if object is null; false otherwise
     */
    public static String capitalize( String value )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( value ) )
        {
            String capitalize = value.substring( 0, 1 )
                                     .toUpperCase();
            capitalize += value.substring( 1 )
                               .toLowerCase();
            return capitalize;
        }
        return value;
    }

    /**
     * Remove double blank spaces from value, remove any leading and trailing whitespace.
     * 
     * @param value
     *            value to trim
     * @return trimmed value
     */
    public static String trim( String value )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( value ) )
        {
            return value.trim()
                        .replaceAll( "\\s+", " " );
        }
        return value;
    }

    /**
     * Remove tabs and new lines from string value.
     * 
     * @param value
     *            value with tabs and new lines
     * @return value without tabs and new lines
     */
    public static String removeTabsAndNewLines( String value )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( value ) )
        {
            return trim( value.replaceAll( "\\r|\\n|\\t", SPACE_VALUE ) );
        }
        return value;
    }

    /**
     * Replace spaces with underscore character.
     * 
     * @param value
     *            value with spaces
     * @return value without spaces
     */
    public static String replaceSpacesWithUnderscores( String value )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( value ) )
        {
            return trim( value ).replaceAll( SPACE_VALUE, UNDERSCORE_VALUE );
        }
        return value;
    }

    /**
     * Convert object array to string array.
     * 
     * @param objects
     *            object array
     * @return string array
     */
    public static String[] asStringArray( Object... objects )
    {
        if ( SharedUtil.isNotNull( objects ) )
        {
            Object[] stringObject = objects;
            if ( objects.length == 1 )
            {
                String string = objects[0].toString()
                                          .replace( BRACKET_LEFT_VALUE, EMPTY_VALUE )
                                          .replace( BRACKET_RIGHT_VALUE, EMPTY_VALUE );
                string = trim( string );
                stringObject = string.split( STRING_COMMA_SPLITTER );
            }

            List<String> strings = new ArrayList<>();
            for ( Object object : stringObject )
            {
                strings.add( trim( object.toString() ) );
            }
            return strings.toArray( new String[strings.size()] );
        }
        return EMPTY_ARRAY;
    }

    /**
     * Convert json array to string array.
     * 
     * @param jsonArray
     *            json array
     * @return string array
     */
    public static String[] asStringArray( JSONArray jsonArray )
    {
        if ( SharedUtil.isNotNull( jsonArray ) )
        {
            List<String> list = new ArrayList<>();
            for ( int index = 0; index < jsonArray.length(); index++ )
            {
                list.add( jsonArray.getString( index ) );
            }
            return list.toArray( new String[list.size()] );
        }
        return EMPTY_ARRAY;
    }

}
