/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.json;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;

import org.dussan.vaadin.dquery.enums.QueryElements;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class JsonBuilder
    implements Serializable
{

    private static final long serialVersionUID = 23010531792953775L;

    private transient Map<String, Map<String, JsonQuery>> jsonQueries = null;

    private transient Map<String, Map<String, String>> sqlQueries = null;

    /**
     * Creates a new instance.
     */
    public JsonBuilder()
    {
        jsonQueries = new HashMap<>();
        sqlQueries = new HashMap<>();
    }

    /**
     * Get sql queries for specified query group.
     * 
     * @param queryGroup
     *            query group
     * @return sql queries for specified query group
     */
    private Map<String, String> getGroupSqlQueries( String queryGroup )
    {
        return sqlQueries.get( queryGroup );
    }

    /**
     * Get sql query for specified query group and query name.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     * @return sql query for specified query group and query name
     */
    public String getSqlQuery( String queryGroup, String queryName )
    {
        Map<String, String> queries = getGroupSqlQueries( queryGroup );
        if ( SharedUtil.isNotNull( queries ) && SharedUtil.isNotNull( queries.get( queryName ) ) )
        {
            String query = queries.get( queryName );
            addSqlQuery( queryGroup, queryName, query );
            return query;
        }
        return StringUtil.EMPTY_VALUE;
    }

    /**
     * Get sql query as column for specified query group and query name.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     * @return sql query as column for specified query group and query name
     */
    public String getSqlQueryAsColumn( String queryGroup, String queryName )
    {
        String query = getSqlQuery( queryGroup, queryName );
        if ( !query.isEmpty() )
        {
            query = query.replace( StringUtil.SEMI_COLON_VALUE, StringUtil.EMPTY_VALUE );
            return StringUtil.removeTabsAndNewLines( MessageFormat.format( JsonQuery.QUERY_AS_COLUMN, query ) );
        }
        return query;
    }

    /**
     * Add sql query.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     * @param sqlQuery
     *            sql query
     */
    public void addSqlQuery( String queryGroup, String queryName, String sqlQuery )
    {
        JsonQuery jsonQuery = getQuery( queryGroup, queryName );
        if ( SharedUtil.isNotNull( jsonQuery ) )
        {
            jsonQuery.addQuery( sqlQuery );
            jsonQueries.get( queryGroup )
                       .put( queryName, jsonQuery );
        }
    }

    /**
     * Get all json queries.
     * 
     * @return all json queries
     */
    public Map<String, Map<String, JsonQuery>> getQueries()
    {
        return jsonQueries;
    }

    /**
     * Get json queries for specified query group.
     * 
     * @param queryGroup
     *            query group
     * @return json queries for specified query group
     */
    public Map<String, JsonQuery> getGroupQueries( String queryGroup )
    {
        return jsonQueries.get( queryGroup );
    }

    /**
     * Get json query for specified query group and query name.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     * @return json query for specified query group and query name
     */
    public JsonQuery getQuery( String queryGroup, String queryName )
    {
        Map<String, JsonQuery> queries = getGroupQueries( queryGroup );
        if ( SharedUtil.isNotNull( queries ) )
        {
            return queries.get( queryName );
        }
        return null;
    }

    /**
     * Add json query.
     * 
     * @param jsonQuery
     *            json query.
     */
    public void addQuery( JsonQuery jsonQuery )
    {
        String queryGroup = jsonQuery.get( QueryElements.GROUP )
                                     .toString();
        String queryName = jsonQuery.get( QueryElements.NAME )
                                    .toString();

        // save json query
        Map<String, JsonQuery> tempJsonQueries = getGroupQueries( queryGroup );
        if ( SharedUtil.isNull( tempJsonQueries ) )
        {
            tempJsonQueries = new HashMap<>();
        }
        tempJsonQueries.put( queryName, jsonQuery );
        jsonQueries.put( queryGroup, tempJsonQueries );

        // save sql query
        if ( !jsonQuery.get( QueryElements.QUERY_AS_COLUMN )
                       .equals( MessageFormat.format( JsonQuery.QUERY_AS_COLUMN, StringUtil.EMPTY_VALUE ) ) )
        {
            Map<String, String> tempSqlQueries = getGroupSqlQueries( queryGroup );
            if ( SharedUtil.isNull( tempSqlQueries ) )
            {
                tempSqlQueries = new HashMap<>();
            }
            tempSqlQueries.put( queryName, jsonQuery.get( QueryElements.QUERY_AS_COLUMN )
                                                    .toString() );
            sqlQueries.put( queryGroup, tempSqlQueries );
        }
    }

    /**
     * Remove json query from specified query group.
     * 
     * @param queryGroup
     *            query group
     * @param queryName
     *            query name
     */
    public void removeQuery( String queryGroup, String queryName )
    {
        Map<String, JsonQuery> tempJsonQueries = getGroupQueries( queryGroup );
        if ( SharedUtil.isNotNull( tempJsonQueries ) )
        {
            tempJsonQueries.remove( queryName );
            jsonQueries.put( queryGroup, tempJsonQueries );
        }

        Map<String, String> tempSqlQueries = getGroupSqlQueries( queryGroup );
        if ( SharedUtil.isNotNull( tempSqlQueries ) )
        {
            tempSqlQueries.remove( queryName );
            sqlQueries.put( queryGroup, tempSqlQueries );
        }
    }

}
