/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.enums;

/**
 * Query elements.
 */
public enum QueryElements
{
    DATA_SOURCE_ID( "datasource_id" ),
    GROUP( "group" ),
    NAME( "name" ),
    FIELDS( "fields" ),
    JOINS( "joins" ),
    QUERY( "query" ),
    QUERY_AS_COLUMN( "query_as_column" ),
    TABLES( "tables" );

    private String element;

    /**
     * Create a new instance.
     * 
     * @param element
     *            query element
     */
    private QueryElements( String element )
    {
        this.element = element;
    }

    /**
     * @return query element as string
     */
    @Override
    public String toString()
    {
        return element;
    }

    /**
     * Query's element: join
     */
    public enum Join
    {

        FROM( "from" ), FROM_FIELDS( "from_fields" ), TO( "to" ), TO_FIELDS( "to_fields" ), TYPE( "type" );

        private String join;

        private Join( String join )
        {
            this.join = join;
        }

        /**
         * @return table query's join element as string
         */
        @Override
        public String toString()
        {
            return join;
        }

    }

    /**
     * Query's element: field
     */
    public enum Field
    {

        COLUMN( "column" ),
        COMMON( "common" ),
        FIELD( "field" ),
        NAME( "name" ),
        SHOW( "show" ),
        SORTING( "sorting" ),
        TABLE( "table" ),
        TOTAL( "total" ),
        UNDEFINED( "undefined" );

        private String field;

        private Field( String field )
        {
            this.field = field;
        }

        /**
         * Retrieve field from string value.
         * 
         * @param field
         *            value from which retrieve field
         * @return field
         */
        public static Field get( Object field )
        {
            for ( Field value : Field.values() )
            {
                if ( value.toString()
                          .equals( field.toString()
                                        .toLowerCase() ) )
                {
                    return value;
                }
            }

            // not found
            return UNDEFINED;
        }

        /**
         * Get query's field element as upper case string.
         * 
         * @return query's field element as upper case string
         */
        public String toUpperCaseString()
        {
            return field.toUpperCase();
        }

        /**
         * @return query's field element as string
         */
        @Override
        public String toString()
        {
            return field;
        }

    }

    /**
     * Query's element: table
     */
    public enum Table
    {

        ALIAS( "alias" ),
        DATA_SOURCE_ID( "datasource_id" ),
        DATABASE( "database" ),
        NAME( "name" ),
        POSITION( "position" ),
        TYPE( "type" );

        private String table;

        /**
         * Create a new instance.
         * 
         * @param table
         *            query's table element
         */
        private Table( String table )
        {
            this.table = table;
        }

        /**
         * @return query's table element as string
         */
        @Override
        public String toString()
        {
            return table;
        }

        /**
         * Table query's element: position
         */
        public enum Position
        {

            LEFT( "left" ), TOP( "top" );

            private String position;

            /**
             * Create a new instance.
             * 
             * @param position
             *            query's table element
             */
            private Position( String position )
            {
                this.position = position;
            }

            /**
             * @return table query's position element as string
             */
            @Override
            public String toString()
            {
                return position;
            }

        }

    }

}
