/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import org.dussan.vaadin.dquery.enums.QueryElements;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;
import org.json.JSONArray;
import org.json.JSONObject;

public class QueryElement
    implements Serializable
{

    private static final long serialVersionUID = -5140261085797799721L;

    private static final String QUERY_AS_COLUMN = "({0})";

    public static final JSONArray EMPTY_JSON_ARRAY = null;

    private JSONObject queryElement = null;

    /**
     * Creates a new instance.
     */
    public QueryElement()
    {
        queryElement = new JSONObject();
    }

    /**
     * Creates a new instance.
     */
    public QueryElement( Object queryElement )
    {
        this();
        if ( SharedUtil.isNotNullAndNotEmpty( queryElement ) )
        {
            this.queryElement = new JSONObject( queryElement.toString() );
        }
    }

    /**
     * Get query element.
     * 
     * @param element
     *            query element
     * @return query element
     */
    public String get( QueryElements element )
    {
        if ( SharedUtil.isNotNull( element ) )
        {
            switch ( element )
            {
                case DATA_SOURCE_ID:
                case GROUP:
                case NAME:
                case QUERY:
                    return queryElement.optString( element.toString() );
                case QUERY_AS_COLUMN:
                    String query = queryElement.optString( QueryElements.QUERY.toString() );
                    query = query.replace( StringUtil.SEMI_COLON_VALUE, StringUtil.EMPTY_VALUE );
                    return StringUtil.removeTabsAndNewLines( MessageFormat.format( QUERY_AS_COLUMN, query ) );
                default:
                    break;
            }
        }
        return StringUtil.EMPTY_VALUE;
    }

    /**
     * Get query element array.
     * 
     * @param element
     *            query element
     * @return query element array
     */
    public JSONArray gets( QueryElements element )
    {
        if ( SharedUtil.isNotNull( element ) )
        {
            switch ( element )
            {
                case FIELDS:
                case JOINS:
                case TABLES:
                    return queryElement.optJSONArray( element.toString() );
                default:
                    break;
            }
        }
        return EMPTY_JSON_ARRAY;
    }

    public static class Table
    {

        private JSONObject table = null;

        /**
         * Creates a new table instance.
         */
        public Table()
        {
            table = new JSONObject();
        }

        /**
         * Creates a new table instance.
         */
        public Table( Object table )
        {
            this();
            if ( SharedUtil.isNotNullAndNotEmpty( table ) )
            {
                this.table = new JSONObject( table.toString() );
            }
        }

        /**
         * Get table element.
         * 
         * @param element
         *            query table element
         * @return table element
         */
        public String get( QueryElements.Table element )
        {
            if ( SharedUtil.isNotNull( element ) )
            {
                switch ( element )
                {
                    case ALIAS:
                    case DATA_SOURCE_ID:
                    case DATABASE:
                    case NAME:
                    case TYPE:
                    case POSITION:
                        return table.optString( element.toString() );
                    default:
                        break;
                }
            }
            return StringUtil.EMPTY_VALUE;
        }

        public static class Position
        {

            private static final int NULL_POSITION = 0;

            private JSONObject position = null;

            /**
             * Creates a new table position instance.
             */
            public Position()
            {
                position = new JSONObject();
            }

            /**
             * Creates a new table position instance.
             */
            public Position( Object position )
            {
                this();
                if ( SharedUtil.isNotNullAndNotEmpty( position ) )
                {
                    this.position = new JSONObject( position.toString() );
                }
            }

            /**
             * Get table position.
             * 
             * @param element
             *            query table position element
             * @return table position
             */
            public int get( QueryElements.Table.Position element )
            {
                if ( SharedUtil.isNotNull( element ) )
                {
                    switch ( element )
                    {
                        case LEFT:
                        case TOP:
                            return position.optInt( element.toString() );
                        default:
                            break;
                    }
                }
                return NULL_POSITION;
            }

        }

    }

    public static class Field
    {

        private JSONObject field = null;

        /**
         * Creates a new field instance.
         */
        public Field()
        {
            field = new JSONObject();
        }

        /**
         * Creates a new field instance.
         */
        public Field( Object field )
        {
            this();
            if ( SharedUtil.isNotNullAndNotEmpty( field ) )
            {
                this.field = new JSONObject( field.toString() );
            }
        }

        /**
         * Get field.
         * 
         * @param element
         *            query field element
         * @return field
         */
        public String get( QueryElements.Field element )
        {
            if ( SharedUtil.isNotNull( element ) )
            {
                switch ( element )
                {
                    case COLUMN:
                    case COMMON:
                    case FIELD:
                    case NAME:
                    case SHOW:
                    case SORTING:
                    case TABLE:
                    case TOTAL:
                        return field.optString( element.toString() );
                    default:
                        break;
                }
            }
            return StringUtil.EMPTY_VALUE;
        }

        /**
         * Get field elements.
         * 
         * @return field elements
         */
        public List<QueryElements.Field> getElements()
        {
            List<QueryElements.Field> elements = new ArrayList<>();
            for ( String element : field.keySet() )
            {
                elements.add( QueryElements.Field.get( element ) );
            }
            return elements;
        }

    }

    public static class Join
    {

        private JSONObject join = null;

        /**
         * Creates a new join instance.
         */
        public Join()
        {
            join = new JSONObject();
        }

        /**
         * Creates a new join instance.
         */
        public Join( Object field )
        {
            this();
            if ( SharedUtil.isNotNullAndNotEmpty( field ) )
            {
                this.join = new JSONObject( field.toString() );
            }
        }

        /**
         * Get join.
         * 
         * @param element
         *            query join element
         * @return join
         */
        public String get( QueryElements.Join element )
        {
            if ( SharedUtil.isNotNull( element ) )
            {
                switch ( element )
                {
                    case FROM:
                    case TO:
                        return join.optString( element.toString() );
                    default:
                        break;
                }
            }
            return StringUtil.EMPTY_VALUE;
        }

        /**
         * Get join array.
         * 
         * @param element
         *            query join element
         * @return join array
         */
        public JSONArray gets( QueryElements.Join element )
        {
            if ( SharedUtil.isNotNull( element ) )
            {
                switch ( element )
                {
                    case FROM_FIELDS:
                    case TO_FIELDS:
                        return join.optJSONArray( element.toString() );
                    default:
                        break;
                }
            }
            return EMPTY_JSON_ARRAY;
        }

    }

}
