/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.ui.windows;

import java.text.MessageFormat;

import org.dussan.vaadin.dquery.base.ui.BaseWindow;
import org.dussan.vaadin.dquery.enums.TablesJoin;
import org.dussan.vaadin.dquery.utils.StringUtil;

import com.vaadin.data.Property.ValueChangeEvent;
import com.vaadin.data.Property.ValueChangeListener;
import com.vaadin.shared.ui.label.ContentMode;
import com.vaadin.ui.Button.ClickEvent;
import com.vaadin.ui.Button.ClickListener;
import com.vaadin.ui.Label;
import com.vaadin.ui.OptionGroup;
import com.vaadin.ui.themes.ValoTheme;

public abstract class JoinedTablesWindow
    extends BaseWindow
{

    private static final long serialVersionUID = -2928434392869716486L;

    private static final String JOIN_TYPE_FROM = "<b>FROM TABLE:</b> {0}</br><b>FROM FIELD:</b> {1}";

    private static final String JOIN_TYPE_INNER = "Inner join";

    private static final String JOIN_TYPE_LEFT = "Left join";

    private static final String JOIN_TYPE_REMOVE = "Remove join";

    private static final String JOIN_TYPE_RIGHT = "Right join";

    private static final String JOIN_TYPE_TO = "<b>TO TABLE:</b> {0}</br><b>TO FIELD:</b> {1}";

    private static final String MODAL = "modal";

    private static final String MODAL_CAPTION = "Change join type";

    private static final String MODAL_GROUP = MODAL + "-group";

    private static final String MODAL_LABEL = MODAL + "-label";

    /**
     * Creates a new instance.
     */
    private JoinedTablesWindow()
    {
        setCaption( MODAL_CAPTION );
        setWidth( 400, Unit.PIXELS );
    }

    /**
     * Creates a new instance.
     * 
     * @param joinedTablesId
     *            index of joined tables
     * @param joinedFieldsIndex
     *            index of joined fields
     */
    public JoinedTablesWindow( final String joinedTablesId, final int joinedFieldsIndex )
    {
        this();

        final String[] joinedData = joinedTablesId.split( StringUtil.STRING_COLON_SPLITTER );
        String[] fromToFileds = joinedData[3 + joinedFieldsIndex].split( StringUtil.STRING_DASH_SPLITTER );

        Label fromTableLabel = new Label( MessageFormat.format( JOIN_TYPE_FROM, joinedData[0], fromToFileds[0] ) );
        fromTableLabel.setContentMode( ContentMode.HTML );
        fromTableLabel.addStyleName( ValoTheme.LABEL_SMALL );
        fromTableLabel.addStyleName( MODAL_LABEL );
        addContextComponent( fromTableLabel );

        Label toTableLabel = new Label( MessageFormat.format( JOIN_TYPE_TO, joinedData[1], fromToFileds[1] ) );
        toTableLabel.setContentMode( ContentMode.HTML );
        toTableLabel.addStyleName( ValoTheme.LABEL_SMALL );
        toTableLabel.addStyleName( MODAL_LABEL );
        addContextComponent( toTableLabel );

        // tables join types
        final OptionGroup joinedTablesGroup = new OptionGroup();
        joinedTablesGroup.addItems( TablesJoin.INNER, TablesJoin.LEFT, TablesJoin.RIGHT, TablesJoin.REMOVE );
        joinedTablesGroup.setItemCaption( TablesJoin.INNER, JOIN_TYPE_INNER );
        joinedTablesGroup.setItemCaption( TablesJoin.LEFT, JOIN_TYPE_LEFT );
        joinedTablesGroup.setItemCaption( TablesJoin.RIGHT, JOIN_TYPE_RIGHT );
        joinedTablesGroup.setItemCaption( TablesJoin.REMOVE, JOIN_TYPE_REMOVE );
        joinedTablesGroup.setNullSelectionAllowed( false );
        joinedTablesGroup.setValue( TablesJoin.getJoin( joinedData[2] ) );
        joinedTablesGroup.addStyleName( ValoTheme.OPTIONGROUP_SMALL );
        joinedTablesGroup.addStyleName( MODAL_GROUP );
        addContextComponent( joinedTablesGroup );

        // "OK" button click
        getOkButton().addClickListener( new ClickListener()
        {
            private static final long serialVersionUID = 3549659948212739081L;

            @Override
            public void buttonClick( ClickEvent event )
            {
                // format: fromTable, toTable, tablesJoin, fromToFields
                setJoinedTablesJoinType( joinedData[0], joinedData[1], (TablesJoin) joinedTablesGroup.getValue(),
                                         joinedFieldsIndex );

                // close dialog window
                close();
            }
        } );

        // group options changed
        joinedTablesGroup.addValueChangeListener( new ValueChangeListener()
        {
            private static final long serialVersionUID = -2939791155589862392L;

            @Override
            public void valueChange( ValueChangeEvent event )
            {
                getOkButton().setEnabled( true );
            }
        } );
    }

    /**
     * Set joined tables join's type.
     * 
     * @param fromTable
     *            table from which connections will be established
     * @param toTable
     *            table to which connections will be established
     * @param tablesJoin
     *            tables join type
     * @param joinedFieldsIndex
     *            joined table fields index
     */
    protected abstract void setJoinedTablesJoinType( String fromTable, String toTable, TablesJoin tablesJoin,
                                                     int joinedFieldsIndex );

}
