/**
 * Copyright (C) 2016-2017  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dquery.base.sql.functions;

import java.io.Serializable;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dussan.vaadin.dquery.enums.SqlFunctionType;
import org.dussan.vaadin.dquery.utils.SharedUtil;
import org.dussan.vaadin.dquery.utils.StringUtil;

public class SqlFunctions
    implements Serializable
{

    private static final long serialVersionUID = 1962753384800249525L;

    private static final String LEFT_PARENTHESIS = "(";

    private static final String FUNCTION_DEFAULT = "{0}()";

    private static final String FUNCTION_WITH_PARAMETERS = "{0}( {1} )";

    public static final String LEFT_PARENTHESIS_ESCAPED = "\\(";

    public static final String RIGHT_PARENTHESIS_ESCAPED = "\\)";

    protected transient Map<String, String> functions = null;

    protected transient Map<String, Object[]> parameterValues = null;

    protected transient Map<String, String[]> parameters = null;

    protected transient Map<String, SqlFunctionType> types = null;

    /**
     * Creates a new instance.
     */
    public SqlFunctions()
    {
        functions = new HashMap<>();
        parameterValues = new HashMap<>();
        parameters = new HashMap<>();
        types = new HashMap<>();

        // default empty function
        functions.put( StringUtil.EMPTY_VALUE, StringUtil.EMPTY_VALUE );
    }

    /**
     * Get function name from function with or without parenthesis.
     * 
     * @param function
     *            function name with or without parenthesis
     * @return function name
     */
    private String getFunctionName( String function )
    {
        if ( SharedUtil.isNotNullAndNotEmpty( function ) && function.contains( LEFT_PARENTHESIS ) )
        {
            return function.split( LEFT_PARENTHESIS_ESCAPED )[0];
        }
        return function;
    }

    /**
     * Get all sql functions.
     * 
     * @return all sql functions
     */
    public String[] getFunctions()
    {
        List<String> tempFunctions = new ArrayList<>();
        for ( String function : functions.keySet() )
        {
            if ( SharedUtil.isNullOrEmpty( function ) )
            {
                tempFunctions.add( function );
            }
            else
            {
                String func = getFunction( function );
                if ( SharedUtil.isNotNullAndNotEmpty( func ) )
                {
                    tempFunctions.add( func );
                }
            }
        }

        Collections.sort( tempFunctions );
        return tempFunctions.toArray( new String[tempFunctions.size()] );
    }

    /**
     * Get sql function.
     * 
     * @param function
     *            function name
     * @return sql function
     */
    public String getFunction( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNull( types.get( functionName ) ) )
        {
            switch ( types.get( functionName ) )
            {
                case NULL:
                    return function;
                case WITH_PARAMETERS:
                    if ( SharedUtil.isNotNull( parameters.get( functionName ) )
                        && SharedUtil.isNotNull( parameterValues.get( functionName ) ) )
                    {
                        List<String> paramValues = new ArrayList<>();
                        for ( int index = 0; index < parameters.get( functionName ).length
                            && index < parameterValues.get( functionName ).length; index++ )
                        {
                            if ( SharedUtil.isNotNull( parameterValues.get( functionName )[index] ) )
                            {
                                paramValues.add( parameterValues.get( functionName )[index].toString() );
                            }
                        }
                        String values = Arrays.toString( paramValues.toArray( new String[paramValues.size()] ) );
                        values = values.substring( 1, values.length() - 1 );
                        return MessageFormat.format( FUNCTION_WITH_PARAMETERS, functionName, values );
                    }
                case DEFAULT:
                case WITHOUT_PARAMETERS:
                default:
                    return MessageFormat.format( FUNCTION_DEFAULT, functionName );
            }
        }
        return null;
    }

    /**
     * Add sql functions.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void addFunction( String function, String description )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNullAndNotEmpty( description ) )
        {
            functions.put( functionName, description );
            types.put( functionName, SqlFunctionType.DEFAULT );
        }
    }

    /**
     * Get function description.
     * 
     * @param function
     *            function name
     * @return function description
     */
    public String getDescription( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) )
        {
            return functions.get( getFunctionName( functionName ) );
        }
        return null;
    }

    /**
     * Set function description.
     * 
     * @param function
     *            function name
     * @param description
     *            function description
     */
    public void setDescription( String function, String description )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNullAndNotEmpty( description )
            && SharedUtil.isNotNull( functions.get( functionName ) ) )
        {
            functions.put( functionName, description );
        }
    }

    /**
     * Get function default values.
     * 
     * @param function
     *            function name
     * @return function's parameter values
     */
    public Object[] getParameterValues( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) )
        {
            if ( hasParameters( functionName ) && !functionName.equals( function ) )
            {
                return function.split( LEFT_PARENTHESIS_ESCAPED )[1].split( RIGHT_PARENTHESIS_ESCAPED )[0].split( StringUtil.STRING_COMMA_SPLITTER );
            }
            return parameterValues.get( functionName );
        }
        return SharedUtil.EMPTY_OBJECT_ARRAY;
    }

    /**
     * Set function's parameter values.
     * 
     * @param function
     *            function name
     * @param parameterValues
     *            function's parameter values
     */
    public void setParameterValues( String function, Object... parameterValues )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNull( parameterValues ) )
        {
            this.parameterValues.put( functionName, parameterValues );
        }
    }

    /**
     * Check if function has parameters.
     * 
     * @param function
     *            function name
     * @return true if function has parameters, otherwise false
     */
    public boolean hasParameters( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) )
        {
            return SharedUtil.isNotNull( parameters.get( functionName ) );
        }
        return false;
    }

    /**
     * Get function parameters.
     * 
     * @param function
     *            function name
     * @return function parameters
     */
    public String[] getParameters( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) )
        {
            return parameters.get( functionName );
        }
        return StringUtil.EMPTY_ARRAY;
    }

    /**
     * Set function parameters.
     * 
     * @param function
     *            function name
     * @param parameters
     *            function parameters
     */
    public void setParameters( String function, String... parameters )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNull( parameters ) )
        {
            this.parameters.put( functionName, parameters );
            types.put( functionName, SqlFunctionType.WITH_PARAMETERS );
        }
    }

    /**
     * Get function type.
     * 
     * @param function
     *            function name
     * @return function type
     */
    public SqlFunctionType getType( String function )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) )
        {
            return types.get( functionName );
        }
        return null;
    }

    /**
     * Set function type.
     * 
     * @param function
     *            function name
     * @param type
     *            function type
     */
    public void setType( String function, SqlFunctionType type )
    {
        String functionName = getFunctionName( function );
        if ( SharedUtil.isNotNullAndNotEmpty( functionName ) && SharedUtil.isNotNull( type ) )
        {
            types.put( functionName, type );
        }
    }

}
