/**
 * Copyright (C) 2016  Dušan Vejnovič  <vaadin@dussan.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dussan.vaadin.dtabs;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.dussan.vaadin.dtabs.client.TabsState;
import org.dussan.vaadin.dtabs.helper.ManifestHelper;

import com.vaadin.server.Resource;
import com.vaadin.ui.Component;
import com.vaadin.ui.TabSheet;

public class DTabs
    extends TabSheet
{

    private static final long serialVersionUID = 937633672883052715L;

    private static final String VERSION = "Implementation-Version";

    private static final String GIT_VERSION = "Git-Version";

    private boolean tabsSorted = false;

    private boolean tabsSortedAscending = true;

    private List<DTabsProperty> properties = new ArrayList<>();

    public DTabs()
    {
        super();
    }

    public DTabs( Component... components )
    {
        super( components );
    }

    @Override
    protected TabsState getState()
    {
        return (TabsState) super.getState();
    }

    @Override
    protected TabsState getState( boolean markAsDirty )
    {
        return (TabsState) super.getState( markAsDirty );
    }

    @Override
    public Tab addTab( Component c )
    {
        return addTab( c, c.getCaption(), null );
    }

    @Override
    public Tab addTab( Component c, String caption )
    {
        return addTab( c, caption, null );
    }

    @Override
    public Tab addTab( Component c, String caption, Resource icon )
    {
        if ( isTabsSortEnabled() && 0 < getComponentCount() )
        {
            for ( int index = 0; index < getComponentCount(); index++ )
            {
                int result = caption.compareTo( getTab( index ).getCaption() );
                if ( ( tabsSortedAscending && result < 0 ) || ( !tabsSortedAscending && 0 < result ) )
                {
                    return super.addTab( c, caption, icon, index );
                }
            }
        }
        return super.addTab( c, caption, icon );
    }

    public static String getVersion()
        throws IOException
    {
        if ( ManifestHelper.getManifest() != null )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( VERSION );
        }
        return null;
    }

    public static String getGitVersion()
        throws IOException
    {
        if ( ManifestHelper.getManifest() != null )
        {
            return ManifestHelper.getManifest()
                                 .getMainAttributes()
                                 .getValue( GIT_VERSION );
        }
        return null;
    }

    public boolean isTabsSortEnabled()
    {
        return tabsSorted;
    }

    public void setTabsSortEnabled( boolean tabsSorted )
    {
        this.tabsSorted = tabsSorted;
        this.tabsSortedAscending = true;
    }

    public void setTabsSortEnabled( boolean tabsSorted, boolean tabsSortedAscending )
    {
        this.tabsSorted = tabsSorted;
        this.tabsSortedAscending = tabsSortedAscending;
    }

    private String getStyle()
    {
        StringBuilder style = new StringBuilder();
        for ( DTabsProperty property : properties )
        {
            if ( style.length() != 0 )
            {
                style.append( " " );
            }
            style.append( property.toString() );
        }
        return style.toString();
    }

    public DTabs setProperties( DTabsProperty... properties )
    {
        for ( DTabsProperty property : properties )
        {
            if ( !this.properties.contains( property ) )
            {
                this.properties.add( property );
            }
        }
        getState( true ).setTabBarBottom( isTabBarBottom() );
        setStyleName( getStyle() );
        return this;
    }

    public DTabs removeProperties( DTabsProperty... properties )
    {
        for ( DTabsProperty property : properties )
        {
            this.properties.remove( property );
        }
        getState( true ).setTabBarBottom( isTabBarBottom() );
        setStyleName( getStyle() );
        return this;
    }

    public DTabs setProperty( DTabsProperty property )
    {
        setProperties( property );
        return this;
    }

    public DTabs removeProperty( DTabsProperty property )
    {
        removeProperties( property );
        return this;
    }

    public boolean isPropertyEnabled( DTabsProperty property )
    {
        return properties.contains( property );
    }

    public boolean isPropertyDisabled( DTabsProperty property )
    {
        return !properties.contains( property );
    }

    public DTabs setTabBarBottom( boolean bottomTabbar )
    {
        if ( bottomTabbar )
        {
            setProperty( DTabsProperty.BOTTOM_TABBAR );
        }
        else
        {
            removeProperty( DTabsProperty.BOTTOM_TABBAR );
        }
        getState( true ).setTabBarBottom( bottomTabbar );
        return this;
    }

    public boolean isTabBarBottom()
    {
        return isPropertyEnabled( DTabsProperty.BOTTOM_TABBAR );
    }

    public boolean isTabBarTop()
    {
        return isPropertyDisabled( DTabsProperty.BOTTOM_TABBAR );
    }

    public DTabs setCenteredTabs( boolean centeredTabs )
    {
        if ( centeredTabs )
        {
            setProperty( DTabsProperty.CENTERED_TABS );
        }
        else
        {
            removeProperty( DTabsProperty.CENTERED_TABS );
        }
        return this;
    }

    public boolean isTabsCentered()
    {
        return isPropertyEnabled( DTabsProperty.CENTERED_TABS );
    }

    public DTabs setCompactedTabBar( boolean compactedTabBar )
    {
        if ( compactedTabBar )
        {
            setProperty( DTabsProperty.COMPACTED_TABBAR );
        }
        else
        {
            removeProperty( DTabsProperty.COMPACTED_TABBAR );
        }
        return this;
    }

    public boolean isTabBarCompacted()
    {
        return properties.contains( DTabsProperty.COMPACTED_TABBAR );
    }

    public DTabs setEqualWidthTabs( boolean equalWidthTabs )
    {
        if ( equalWidthTabs )
        {
            setProperty( DTabsProperty.EQUAL_WIDTH_TABS );
        }
        else
        {
            removeProperty( DTabsProperty.EQUAL_WIDTH_TABS );
        }
        return this;
    }

    public boolean isTabsEqualWidth()
    {
        return isPropertyEnabled( DTabsProperty.EQUAL_WIDTH_TABS );
    }

    public DTabs setFramedTabs( boolean framedTabs )
    {
        if ( framedTabs )
        {
            setProperty( DTabsProperty.FRAMED_TABS );
        }
        else
        {
            removeProperty( DTabsProperty.FRAMED_TABS );
        }
        return this;
    }

    public boolean isTabsFramed()
    {
        return isPropertyEnabled( DTabsProperty.FRAMED_TABS );
    }

    public DTabs setIconsOnTop( boolean iconsOnTop )
    {
        if ( iconsOnTop )
        {
            setProperty( DTabsProperty.ICONS_ON_TOP );
        }
        else
        {
            removeProperty( DTabsProperty.ICONS_ON_TOP );
        }
        return this;
    }

    public boolean isIconsOnTop()
    {
        return isPropertyEnabled( DTabsProperty.ICONS_ON_TOP );
    }

    public DTabs setOnlySelectedTabClosable( boolean onlySelectedTabClosable )
    {
        if ( onlySelectedTabClosable )
        {
            setProperty( DTabsProperty.ONLY_SELECTED_TAB_IS_CLOSABLE );
        }
        else
        {
            removeProperty( DTabsProperty.ONLY_SELECTED_TAB_IS_CLOSABLE );
        }
        return this;
    }

    public boolean isOnlySelectedTabClosable()
    {
        return isPropertyEnabled( DTabsProperty.ONLY_SELECTED_TAB_IS_CLOSABLE );
    }

    public DTabs setPaddedTabBar( boolean paddedTabbar )
    {
        if ( paddedTabbar )
        {
            setProperty( DTabsProperty.PADDED_TABBAR );
        }
        else
        {
            removeProperty( DTabsProperty.PADDED_TABBAR );
        }
        return this;
    }

    public boolean isTabBarPadded()
    {
        return isPropertyEnabled( DTabsProperty.PADDED_TABBAR );
    }

    public DTabs setRightAlignedTabs( boolean rightalignedTabs )
    {
        if ( rightalignedTabs )
        {
            setProperty( DTabsProperty.RIGHT_ALIGNED_TABS );
        }
        else
        {
            removeProperty( DTabsProperty.RIGHT_ALIGNED_TABS );
        }
        return this;
    }

    public boolean isTabsRightAligned()
    {
        return isPropertyEnabled( DTabsProperty.RIGHT_ALIGNED_TABS );
    }

}
